/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkframe.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkscrolledwindow.h>

#include "auto-completion.h"
#include "file-utils.h"
#include "typedefs.h"

#define MAX_VISIBLE_ROWS 8

static gchar *ac_dir            = NULL;
static gchar *ac_path           = NULL;
static GList *ac_subdirs        = NULL; 
static GList *ac_alternatives   = NULL;

static GtkWidget *ac_window     = NULL;
static GtkWidget *ac_clist      = NULL;
static GtkWidget *ac_entry      = NULL;

static void 
ac_dir_free ()
{
	if (!ac_dir) 
		return;

	g_free (ac_dir);
	ac_dir = NULL;
}


static void 
ac_path_free ()
{
	if (!ac_path)
		return;

	g_free (ac_path);
	ac_path = NULL;
}


static void 
ac_subdirs_free ()
{
	if (!ac_subdirs)
		return;

	g_list_foreach (ac_subdirs, (GFunc) g_free, NULL);
	g_list_free (ac_subdirs);
	ac_subdirs = NULL;
}


static void 
ac_alternatives_free ()
{
	if (!ac_alternatives)
		return;

	g_list_foreach (ac_alternatives, (GFunc) g_free, NULL);
	g_list_free (ac_alternatives);
	ac_alternatives = NULL;
}


void
auto_compl_reset () 
{
	ac_dir_free ();
	ac_path_free ();
	ac_subdirs_free ();
	ac_alternatives_free ();
}


gint
auto_compl_get_n_alternatives (gchar *path)
{
	gchar *dir;
	gint path_len;
	GList *scan;
	gint n;

	if (path == NULL)
		return 0;

	if (strcmp (path, "/") == 0)
		dir = g_strdup ("/");
	else
		dir = remove_level_from_path (path);
	
	if (! path_is_dir (dir)) {
		g_free (dir);
		return 0;
	}

	if ((ac_dir == NULL) || (strcmp (dir, ac_dir) != 0)) {
		GList *row_dir_list;

		ac_dir_free ();
		ac_subdirs_free ();

		ac_dir = g_strdup (dir);
		path_list_new (ac_dir, NULL, &row_dir_list);
		ac_subdirs = dir_list_filter_and_sort (row_dir_list, 
						       FALSE,
						       TRUE);
		path_list_free (row_dir_list);
	}

	ac_path_free ();
	ac_alternatives_free ();

	ac_path = g_strdup (path);
	path_len = strlen (ac_path);
	n = 0;

	for (scan = ac_subdirs; scan; scan = scan->next) {
		const gchar *subdir = (gchar*) scan->data;

		if (strncmp (path, subdir, path_len) != 0) 
			continue;
		
		ac_alternatives = g_list_prepend (ac_alternatives, 
						  g_strdup (subdir));
		n++;
	}

	g_free (dir);
	ac_alternatives = g_list_reverse (ac_alternatives);

	return n;
}


static gint
get_common_prefix_length () 
{
	gint n;
	GList *scan;
	gchar c1, c2;

	g_return_val_if_fail (ac_path != NULL, 0);
	g_return_val_if_fail (ac_alternatives != NULL, 0);

	/* if the number of alternatives is 1 return its length. */
	if (ac_alternatives->next == NULL)
		return strlen ((gchar*) ac_alternatives->data);

	n = strlen (ac_path);
	while (TRUE) {
		scan = ac_alternatives;

		c1 = ((gchar*) scan->data) [n];

		if (c1 == 0)
			return n;

		/* check that all other alternatives have the same 
		 * character at position n. */

		scan = scan->next;
		
		for (; scan; scan = scan->next) {
			c2 = ((gchar*) scan->data) [n];
			if (c1 != c2)
				return n;
		}

		n++;
	}

	return -1;
}


gchar *
auto_compl_get_common_prefix () 
{
	gchar *alternative;
	gint n;

	if (ac_path == NULL)
		return NULL;

	if (ac_alternatives == NULL)
		return NULL;

	n = get_common_prefix_length ();
	alternative = (gchar*) ac_alternatives->data;

	return g_strndup (alternative, n);
}


GList * 
auto_compl_get_alternatives ()
{
	return ac_alternatives;
}


static gboolean
ac_window_button_press_cb (GtkWidget *widget,
			   GdkEventButton *event,
			   gpointer *data)
{
	GtkWidget *event_widget;
	gint x, y, w, h;
	
	event_widget = gtk_get_event_widget ((GdkEvent *)event);

	gdk_window_get_origin (ac_window->window, &x, &y);
	gdk_window_get_size (ac_window->window, &w, &h);

	/* Checks if the button press happened inside the window, 
	 * if not closes the window. */
	if ((event->x >= x) && (event->x <= x + w)
	    && (event->y  >= y) && (event->y <= y + h)) {
		/* In window. */
		return FALSE;
	}

	auto_compl_hide_alternatives ();

	return TRUE;
}


static gboolean
ac_window_key_press_cb (GtkWidget *widget,
			GdkEventKey *event,
			gpointer *data)
{
	if (event->keyval == GDK_Escape) {
		auto_compl_hide_alternatives ();
		return TRUE;
	}

	/* allow keyboard navigation in the alternatives clist */
	if (event->keyval == GDK_Up 
	    || event->keyval == GDK_Down 
	    || event->keyval == GDK_Page_Up 
	    || event->keyval == GDK_Page_Down 
	    || event->keyval == GDK_space)
		return FALSE;
	
	if (event->keyval == GDK_Return) {
		event->keyval = GDK_space;
		return FALSE;
	}

	auto_compl_hide_alternatives ();
	gtk_widget_event (ac_entry, (GdkEvent*) event);
	return TRUE;
}


static void 
ac_clist_select_row_cb (GtkCList *clist,
			gint row, 
			gint column,
			GdkEventButton *event,
			gpointer *data)
{
	GdkEventKey tmp_event;
	gchar *text;
	gchar *full_path;

	auto_compl_hide_alternatives ();
	
	gtk_clist_get_text (GTK_CLIST (ac_clist), row, column, &text);
	full_path = g_strconcat (ac_dir,
				 "/",
				 text,
				 NULL);
	gtk_entry_set_text (GTK_ENTRY (ac_entry), full_path);
	g_free (full_path);

	/* send a synthetic return keypress to the entry */
	tmp_event.type = GDK_KEY_PRESS;
	tmp_event.window = ac_entry->window;
	tmp_event.send_event = TRUE;
	tmp_event.time = GDK_CURRENT_TIME;
	tmp_event.state = 0;
	tmp_event.keyval = GDK_Return;

	gtk_widget_event (ac_entry, (GdkEvent *) &tmp_event);
}


/* displays a list of alternatives under the entry widget. */
void
auto_compl_show_alternatives (GtkWidget *entry)
{
	gint x, y, w, h;
	GList *scan;
	gint row_height, n, width;

	if (ac_window == NULL) {
		GtkWidget *scroll;
		GtkWidget *frame;

		ac_window = gtk_window_new (GTK_WINDOW_POPUP);

		ac_clist = gtk_clist_new (1);
		row_height = (GTK_WIDGET (ac_clist)->style->font->ascent 
			      + GTK_WIDGET (ac_clist)->style->font->descent 
			      + CLIST_ROW_PAD); 
		gtk_clist_set_row_height (GTK_CLIST (ac_clist), row_height);

		scroll = gtk_scrolled_window_new (NULL, NULL);
		gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
						GTK_POLICY_AUTOMATIC,
						GTK_POLICY_AUTOMATIC);

		frame = gtk_frame_new (NULL);
		gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

		gtk_container_add (GTK_CONTAINER (ac_window), frame);
		gtk_container_add (GTK_CONTAINER (frame), scroll);
		gtk_container_add (GTK_CONTAINER (scroll), ac_clist);

		gtk_signal_connect (GTK_OBJECT (ac_window),
				    "button-press-event",
				    GTK_SIGNAL_FUNC(ac_window_button_press_cb),
				    NULL);
		gtk_signal_connect (GTK_OBJECT (ac_window),
				    "key-press-event",
				    GTK_SIGNAL_FUNC(ac_window_key_press_cb),
				    NULL);

		gtk_signal_connect (GTK_OBJECT (ac_clist), 
				    "select_row", 
				    GTK_SIGNAL_FUNC(ac_clist_select_row_cb), 
				    NULL);
	}

	ac_entry = entry;
	gtk_clist_freeze (GTK_CLIST (ac_clist));
	gtk_clist_clear (GTK_CLIST (ac_clist));

	width = 0;
	n = 0;

	for (scan = ac_alternatives; scan; scan = scan->next) {
		gchar *name_only = g_strdup (file_name_from_path (scan->data));
		gchar *text[] = {name_only, NULL};

		gtk_clist_append (GTK_CLIST (ac_clist), text);
		width = MAX (width, gdk_string_width (ac_clist->style->font, 
						      text[0]));
		g_free (name_only);

		n++;
	}

	gtk_clist_set_column_width (GTK_CLIST (ac_clist), 0, width);
	gtk_clist_thaw (GTK_CLIST (ac_clist));

	gdk_window_get_geometry (entry->window, &x, &y, &w, &h, NULL);
	gdk_window_get_deskrelative_origin (entry->window, &x, &y);
	gtk_widget_set_uposition (ac_window, x, y + h);
	gtk_widget_set_usize (ac_window, w, 200);

	gtk_widget_show_all (ac_window);
	gdk_pointer_grab (ac_window->window, 
			  TRUE,
			  (GDK_POINTER_MOTION_MASK 
			   | GDK_BUTTON_PRESS_MASK 
			   | GDK_BUTTON_RELEASE_MASK),
			  NULL, 
			  NULL, 
			  GDK_CURRENT_TIME);
	gdk_keyboard_grab (ac_window->window,
			   FALSE,
			   GDK_CURRENT_TIME);
	gtk_grab_add (ac_window);
}


void
auto_compl_hide_alternatives ()
{
	if (ac_window && GTK_WIDGET_VISIBLE (ac_window)) {
		gdk_pointer_ungrab (GDK_CURRENT_TIME);
		gdk_keyboard_ungrab (GDK_CURRENT_TIME);
		gtk_grab_remove (ac_window);
		gtk_widget_hide (ac_window);
	}
}
