/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <gtk/gtkmain.h>
#include <libgnome/libgnome.h>
#include <libgnomevfs/gnome-vfs-types.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include "gthumb-init.h"
#include "thumb-loader.h"
#include "typedefs.h"
#include "thumb-cache.h"
#include "image-loader.h"
#include "pixbuf-utils.h"
#include "file-utils.h"
#include "my-marshallers.h"


/* If the file size is greater than this the thumbnail will not be
 * created, for functionality reasons. **/
#define MAX_FILE_SIZE (4*1024*1024)
#define CACHE_MAX_W   85
#define CACHE_MAX_H   85


typedef struct
{
	ImageLoader *il;

	GdkPixbuf *pixbuf;	   /* Contains final (scaled if necessary) 
				    * image when done */

	gchar *path;

	gboolean use_cache : 1;
	gboolean from_cache : 1;
	gboolean from_nautilus_cache : 1;

	gfloat percent_done;

	gint max_w;
	gint max_h;
} ThumbLoaderPrivateData;


enum {
	ERROR,
	DONE,
	PROGRESS,
	LAST_SIGNAL
};


static GtkObjectClass *parent_class = NULL;
static guint thumb_loader_signals[LAST_SIGNAL] = { 0 };

static void         thumb_loader_done_cb    (ImageLoader *il,
					     gpointer data);

static void         thumb_loader_error_cb   (ImageLoader *il,
					     gpointer data);

static gint         normalize_thumb         (gint *width, 
					     gint *height, 
					     gint max_w, 
					     gint max_h);

static gint         scale_thumb             (gint *width, 
					     gint *height, 
					     gint max_w, 
					     gint max_h);


static void 
thumb_loader_destroy (GtkObject *object)
{
	ThumbLoader *tl;
	ThumbLoaderPrivateData *priv;

        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_THUMB_LOADER (object));
  
	tl = THUMB_LOADER (object);
	priv = tl->priv;

	if (priv->pixbuf) 
		gdk_pixbuf_unref (priv->pixbuf);

	gtk_object_destroy (GTK_OBJECT (priv->il));

	if (priv->path)
		g_free (priv->path);

	g_free (priv);
	tl->priv = NULL;

	/* Chain up */
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
thumb_loader_class_init (ThumbLoaderClass *class)
{
	GtkObjectClass *object_class;

	parent_class = gtk_type_class (gtk_object_get_type ());
	object_class = (GtkObjectClass*) class;

	thumb_loader_signals[ERROR] =
                gtk_signal_new ("error",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ThumbLoaderClass, error),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE, 0);
	thumb_loader_signals[DONE] =
                gtk_signal_new ("done",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ThumbLoaderClass, done),
                                gtk_marshal_NONE__NONE,
                                GTK_TYPE_NONE, 0);
	thumb_loader_signals[PROGRESS] =
                gtk_signal_new ("progress",
                                GTK_RUN_LAST,
                                object_class->type,
                                GTK_SIGNAL_OFFSET (ThumbLoaderClass, progress),
                                gtk_marshal_NONE__FLOAT,
                                GTK_TYPE_NONE, 1,
				GTK_TYPE_FLOAT);

	gtk_object_class_add_signals (object_class, thumb_loader_signals, 
                                      LAST_SIGNAL);

	object_class->destroy = thumb_loader_destroy;

	class->error = NULL;
	class->done = NULL;
	class->progress = NULL;
}


static void
thumb_loader_init (ThumbLoader *tl)
{
	ThumbLoaderPrivateData *priv;

	tl->priv = g_new (ThumbLoaderPrivateData, 1);
	priv = tl->priv;

	priv->path = NULL;
	priv->pixbuf = NULL;
	priv->use_cache = TRUE;
	priv->from_cache = FALSE;
	priv->from_nautilus_cache = FALSE;
	priv->percent_done = 0.0;
}


GtkType
thumb_loader_get_type ()
{
	static guint thumb_loader_type = 0;

	if (! thumb_loader_type) {
		GtkTypeInfo thumb_loader_info = {
			"ThumbLoader",
			sizeof (ThumbLoader),
			sizeof (ThumbLoaderClass),
			(GtkClassInitFunc) thumb_loader_class_init,
			(GtkObjectInitFunc) thumb_loader_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		thumb_loader_type = gtk_type_unique (gtk_object_get_type (), 
						     &thumb_loader_info);
	}
	
	return thumb_loader_type;
}


GtkObject*     
thumb_loader_new (const gchar *path, 
		  gint width, 
		  gint height)
{
	ThumbLoaderPrivateData *priv;
	ThumbLoader *tl;

	tl = THUMB_LOADER (gtk_type_new (thumb_loader_get_type ()));
	priv = tl->priv;

	priv->max_w = width;
	priv->max_h = height;

	priv->il = IMAGE_LOADER (image_loader_new (path, FALSE));
	if (path)
		priv->path = g_strdup (path);
	else
		priv->path = NULL;

	gtk_signal_connect (GTK_OBJECT (priv->il), "done", 
			    thumb_loader_done_cb,
			    tl);
	gtk_signal_connect (GTK_OBJECT (priv->il), "error", 
			    thumb_loader_error_cb,
			    tl);

	return GTK_OBJECT (tl);
}


void
thumb_loader_use_cache (ThumbLoader *tl,
			gboolean use)
{
	ThumbLoaderPrivateData *priv;

	g_return_if_fail (tl != NULL);

	priv = tl->priv;
	priv->use_cache = use;
}


void
thumb_loader_set_path (ThumbLoader *tl,
		       const gchar *path)
{
	ThumbLoaderPrivateData *priv;

	g_return_if_fail (tl != NULL);
	g_return_if_fail (path != NULL);

	priv = tl->priv;
	if (priv->path)
		g_free (priv->path);
	priv->path = g_strdup (path);
	
	image_loader_set_path (priv->il, path);
}


void
thumb_loader_set_uri (ThumbLoader *tl,
		      const GnomeVFSURI *uri)
{
	ThumbLoaderPrivateData *priv;

	g_return_if_fail (tl != NULL);
	g_return_if_fail (uri != NULL);

	priv = tl->priv;
	if (priv->path)
		g_free (priv->path);

	image_loader_set_uri (priv->il, uri);
	priv->path = image_loader_get_path (priv->il);
}


GnomeVFSURI *
thumb_loader_get_uri (ThumbLoader *tl)
{
	ThumbLoaderPrivateData *priv;
	GnomeVFSURI *uri;
	gchar *escaped_path;

	g_return_val_if_fail (tl != NULL, NULL);

	priv = tl->priv;

	escaped_path = gnome_vfs_escape_path_string (priv->path);
	uri = gnome_vfs_uri_new (escaped_path);
	g_free (escaped_path);

	return uri;
}


GdkPixbuf *
thumb_loader_get_pixbuf (ThumbLoader *tl)
{
	ThumbLoaderPrivateData *priv;

	g_return_val_if_fail (tl != NULL, NULL);

	priv = tl->priv;
	return priv->pixbuf;
}


ImageLoader *
thumb_loader_get_image_loader (ThumbLoader *tl)
{
	ThumbLoaderPrivateData *priv;

	g_return_val_if_fail (tl != NULL, NULL);

	priv = tl->priv;
	return priv->il;
}


gchar *
thumb_loader_get_path (ThumbLoader *tl)
{
	ThumbLoaderPrivateData *priv;

	g_return_val_if_fail (tl != NULL, NULL);

	priv = tl->priv;
	return priv->path;
}


void 
thumb_loader_start (ThumbLoader *tl)
{
	ThumbLoaderPrivateData *priv;
	gchar *cache_path = NULL;
	gchar *path;

	g_return_if_fail (tl != NULL);

	priv = tl->priv;
	path = priv->path;

	g_return_if_fail (path != NULL);

	if (priv->use_cache) {
		/* Try to load the thumbnail from the nautilus cache ... */
		cache_path = cache_get_nautilus_thumbnail_file (path);
		priv->from_nautilus_cache = cache_path != NULL;

		/* ... or from the gqview cache. */
		if (cache_path == NULL) 
			cache_path = cache_get_gthumb_cache_name (path);

		if (path_is_file (cache_path)) {
			if (get_file_mtime (cache_path) < get_file_mtime (path)) {
				/* The thumbnail is old. */
				g_free (cache_path);
				cache_path = NULL;
			}
		} else {
			/* The thumbnail does not exist. */
			g_free (cache_path);
			cache_path = NULL;
		}
	}

	if (cache_path) {
		if (priv->from_nautilus_cache) {
			/* Delete the gqview thumbnail if it exists. */
			gchar *gthumb_file;

			gthumb_file = cache_get_gthumb_cache_name (path);
			if (path_is_file (gthumb_file))
				unlink (gthumb_file);
			g_free (gthumb_file);
		}

		priv->from_cache = TRUE;
		image_loader_set_path (priv->il, cache_path);
		g_free (cache_path);
	} else {
		priv->from_cache = FALSE;
		priv->from_nautilus_cache = FALSE;
		image_loader_set_path (priv->il, path);

		/* Check file dimensions. */
		if (get_file_size (path) > MAX_FILE_SIZE) {
			gtk_signal_emit (GTK_OBJECT (tl), 
					 thumb_loader_signals[ERROR], 
					 tl);
			return;
		}
	}

	image_loader_start (priv->il);
}


void 
thumb_loader_stop (ThumbLoader *tl)
{
	ThumbLoaderPrivateData *priv;

	g_return_if_fail (tl != NULL);

	priv = tl->priv;

	g_return_if_fail (priv->il != NULL);

	image_loader_stop_with_error (priv->il);
}


gint 
thumb_from_xpm_d (const char **data, 
		  gint max_w, 
		  gint max_h, 
		  GdkPixmap **pixmap, 
		  GdkBitmap **mask)
{
	GdkPixbuf *pixbuf;
	gint w, h;

	pixbuf = gdk_pixbuf_new_from_xpm_data (data);
	w = gdk_pixbuf_get_width (pixbuf);
	h = gdk_pixbuf_get_height (pixbuf);

	if (normalize_thumb (&w, &h, max_w, max_h)) {
		/* Scale */
		GdkPixbuf *tmp;
		
		tmp = pixbuf;
		pixbuf = gdk_pixbuf_scale_simple (tmp, w, h, (preferences.thumb_quality == ZOOM_QUALITY_HIGH) ? GDK_INTERP_BILINEAR : GDK_INTERP_NEAREST);
		gdk_pixbuf_unref (tmp);
	}

	gdk_pixbuf_render_pixmap_and_mask (pixbuf, pixmap, mask, 127);
	gdk_pixbuf_unref (pixbuf);

	return w;
}





/* -- local functions -- */


static gint 
thumb_loader_save_to_cache (ThumbLoader *tl)
{
	ThumbLoaderPrivateData *priv;
	gchar *cache_dir;
	gchar *cache_path;
	gint success = FALSE;

	if (! tl) return FALSE;

	priv = tl->priv;

	if (! priv->pixbuf) return FALSE;

	cache_path = cache_get_nautilus_cache_name (priv->path); 
	cache_dir = remove_level_from_path (cache_path);

	if (cache_dir == NULL)
		return FALSE;

	if (ensure_dir_exists (cache_dir))
		success = pixbuf_to_file_as_png (priv->pixbuf, cache_path);

	g_free (cache_dir);
	g_free (cache_path);

	return success;
}


static void 
thumb_loader_done_cb (ImageLoader *il,
		      gpointer data)
{
	ThumbLoader *tl = data;
	ThumbLoaderPrivateData *priv = tl->priv;
	GdkPixbuf *pixbuf;
	gint width, height;
	gboolean modified;

	pixbuf = image_loader_get_pixbuf (priv->il);
	if (! pixbuf) {
		gtk_signal_emit (GTK_OBJECT (tl), thumb_loader_signals[ERROR]);
		return;
	}

	if (priv->pixbuf) 
		gdk_pixbuf_unref (priv->pixbuf);
	priv->pixbuf = pixbuf;
	gdk_pixbuf_ref (pixbuf);

	width = gdk_pixbuf_get_width (pixbuf);
	height = gdk_pixbuf_get_height (pixbuf);

	/* Thumbnails are always saved with the same size, then scaled if
	 * necessary. */

	/* Check whether to scale. */
	modified = normalize_thumb (&width, &height, CACHE_MAX_W, CACHE_MAX_H);
	if (modified) {
		gdk_pixbuf_unref (priv->pixbuf);
		priv->pixbuf = gdk_pixbuf_scale_simple (pixbuf, width, height, 
							(preferences.thumb_quality == ZOOM_QUALITY_HIGH) ? GDK_INTERP_BILINEAR : GDK_INTERP_NEAREST);
	}

	/* Save the thumbnail if necessary. */
	if (! priv->from_cache
	    && priv->use_cache
	    && preferences.enable_thumb_caching 
	    && modified) 
		thumb_loader_save_to_cache (tl);
       

	/* Scale if the user wants a different size. */
	modified = scale_thumb (&width, &height, priv->max_w, priv->max_h);
	if (modified) {
		pixbuf = priv->pixbuf;
		priv->pixbuf = gdk_pixbuf_scale_simple (pixbuf, 
							width, height, 
							(preferences.thumb_quality == ZOOM_QUALITY_HIGH) ? GDK_INTERP_BILINEAR : GDK_INTERP_NEAREST);
		gdk_pixbuf_unref (pixbuf);
	}

	gtk_signal_emit (GTK_OBJECT (tl), thumb_loader_signals[DONE]);
}


static void 
thumb_loader_error_cb (ImageLoader *il,
		       gpointer data)
{
	ThumbLoader *tl = data;
	ThumbLoaderPrivateData *priv = tl->priv;

	if (! priv->from_cache) {
		gtk_signal_emit (GTK_OBJECT (tl), thumb_loader_signals[ERROR]);
		return;
	}

	/* Try from the original if cache load attempt failed. */
	priv->from_cache = FALSE;
	g_warning ("Thumbnail image in cache failed to load, trying to recreate.");
	
	image_loader_set_path (priv->il, priv->path);
	image_loader_start (priv->il);
}


static gint 
normalize_thumb (gint *width, 
		 gint *height, 
		 gint max_width, 
		 gint max_height)
{
	gboolean modified;
	gfloat max_w = max_width;
	gfloat max_h = max_height;
	gfloat w = *width;
	gfloat h = *height;
	gfloat factor;
	gint new_width, new_height;

	if ((*width <= max_width) && (*height <= max_height)) return FALSE;

	factor = MIN (max_w / w, max_h / h);
	new_width  = MAX ((gint) (w * factor), 1);
	new_height = MAX ((gint) (h * factor), 1);
	
	modified = (new_width != *width) || (new_height != *height);

	*width = new_width;
	*height = new_height;

	return modified;
}


static gint 
scale_thumb (gint *width, 
	     gint *height, 
	     gint max_width, 
	     gint max_height)
{
	gboolean modified;
	gfloat max_w = max_width;
	gfloat max_h = max_height;
	gfloat w = *width;
	gfloat h = *height;
	gfloat factor;
	gint new_width, new_height;

	if ((*width < CACHE_MAX_W) && (*height < CACHE_MAX_H)) 
		return FALSE;

	factor = MIN (max_w / w, max_h / h);
	new_width  = MAX ((gint) (w * factor), 1);
	new_height = MAX ((gint) (h * factor), 1);
	
	modified = (new_width != *width) || (new_height != *height);

	*width = new_width;
	*height = new_height;

	return modified;
}
