/**
 * gstrtpjitterbuffer.c: Implementation of codec-independent playout buffer
 *                       Much of the code has been moved from basertpdepay
 *                       in the gstreamer cvs.
 *
 * Copyright (C) <2006> Nokia Corporation.
 *   Contact: Kai Vehmanen <kai.vehmanen@nokia.com>
 * Copyright (C) <2006> Zeeshan Ali <zali@movial.fi> 
 * Copyright (C) <2005> Philippe Khalaf <burger@speedy.org> 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gstrtpjitterbuffer.h"
#include <stdlib.h>

/* elementfactory information */
static GstElementDetails gst_rtp_jitter_buffer_details = {
  "RTP packet jitter-buffer",
  "Filter/Network",
  "Re-orders RTP packets and removes the duplicates",
  "Zeeshan Ali <zali@movial.fi>"
};

GST_DEBUG_CATEGORY (rtpjitterbuffer_debug);
#define GST_CAT_DEFAULT (rtpjitterbuffer_debug)

#define DEFAULT_SLEEP_TIME             5000000      /* nsec (5 ms) */
#define DEFAULT_SLEEP_TIME_MIN         1000000      /* nsec (1 ms) */
#define DEFAULT_INACTIVE_SLEEP_TIME    1000000      /* usec (1000 ms) */
#define GST_RTP_JB_MAX_PACKETS         1000          /* packets */
#define GST_RTP_JB_OVERFLOW_LIMIT      4            /* times queue_delay */


/* Filter signals and args */
enum
{
  /* FILL ME */
  LAST_SIGNAL
};

enum
{
  ARG_0,
  ARG_QUEUE_DELAY,
  ARG_CLOCK_RATE,
};

static GstStaticPadTemplate gst_rtp_jitter_buffer_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("application/x-rtp"));

static GstStaticPadTemplate gst_rtp_jitter_buffer_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("application/x-rtp"));

static GstElementClass *parent_class = NULL;

static void gst_rtp_jitter_buffer_base_init (GstRTPJitterBufferClass * klass);
static void gst_rtp_jitter_buffer_class_init (GstRTPJitterBufferClass * klass);
static void gst_rtp_jitter_buffer_init (GstRTPJitterBuffer * jbuffer,
    gpointer g_class);
static void gst_rtp_jitter_buffer_push (GstRTPJitterBuffer * jbuffer,
    GstBuffer * rtp_buf);
static void gst_rtp_jitter_buffer_queue_release (GstRTPJitterBuffer * jbuffer);
/* static GstClock *gst_rtp_jitter_buffer_provide_clock (GstElement * element); */
static GstFlowReturn gst_rtp_jitter_buffer_getrange (GstPad * pad,
    guint64 offset, guint length, GstBuffer ** buffer);
static gboolean gst_rtp_jitter_buffer_src_activate_pull (GstPad * pad,
    gboolean active);

/**
 * Performs comparison 'a < b' with check for overflows.
 */
static inline gboolean priv_compare_rtp_seq_lt(guint16 a, guint16 b)
{
  /* check if diff more than half of the 16bit range */
  if (abs(b - a) > (1 << 7)) {
    /* one of a/b has wrapped */
    return !(a < b);
  }
  else {
    return a < b;
  }
}

/**
 * Performs 'abs(a - b)' with check for overflows.
 */
static guint32 priv_delta_rtp_ts(guint32 a, guint32 b)
{
  /* check if diff more than half of the 32bit range */
  guint32 delta = (a > b ? a - b : b - a);
  if (delta > G_MAXUINT16) {
    /* one of a/b has wrapped */
    return G_MAXUINT32 - delta + 1;
  }
  else {
    return delta;
  }
}

GType
gst_rtp_jitter_buffer_get_type (void)
{
  static GType plugin_type = 0;

  if (!plugin_type) {
    static const GTypeInfo plugin_info = {
      sizeof (GstRTPJitterBufferClass),
      (GBaseInitFunc) gst_rtp_jitter_buffer_base_init,
      NULL,
      (GClassInitFunc) gst_rtp_jitter_buffer_class_init,
      NULL,
      NULL,
      sizeof (GstRTPJitterBuffer),
      0,
      (GInstanceInitFunc) gst_rtp_jitter_buffer_init,
    };
    plugin_type = g_type_register_static (GST_TYPE_ELEMENT,
        "GstRTPJitterBuffer", &plugin_info, 0);
  }
  return plugin_type;
}

static void gst_rtp_jitter_buffer_finalize (GObject * object);
static void gst_rtp_jitter_buffer_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_rtp_jitter_buffer_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

static gboolean gst_rtp_jitter_buffer_sink_setcaps (GstPad * pad,
    GstCaps * caps);

static GstFlowReturn gst_rtp_jitter_buffer_chain (GstPad * pad, GstBuffer * in);
static gboolean
gst_rtp_jitter_buffer_handle_sink_event (GstPad * pad, GstEvent * event);

static GstStateChangeReturn gst_rtp_jitter_buffer_change_state (GstElement *
    element, GstStateChange transition);
static GstFlowReturn gst_rtp_jitter_buffer_add_to_queue (GstRTPJitterBuffer *
    jbuffer, GstBuffer * in);
static void gst_rtp_jitter_buffer_set_gst_timestamp (GstRTPJitterBuffer *
    jbuffer, GstBuffer * buf);

static void
gst_rtp_jitter_buffer_base_init (GstRTPJitterBufferClass * klass)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_rtp_jitter_buffer_src_template));
  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_rtp_jitter_buffer_sink_template));
  gst_element_class_set_details (element_class, &gst_rtp_jitter_buffer_details);
}

static void
gst_rtp_jitter_buffer_class_init (GstRTPJitterBufferClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = G_OBJECT_CLASS (klass);
  gstelement_class = (GstElementClass *) klass;
  parent_class = g_type_class_peek_parent (klass);

  gobject_class->set_property = gst_rtp_jitter_buffer_set_property;
  gobject_class->get_property = gst_rtp_jitter_buffer_get_property;

  g_object_class_install_property (gobject_class, ARG_QUEUE_DELAY,
      g_param_spec_uint ("queue-delay", "Queue Delay",
          "Amount of ms to queue/buffer", 0, G_MAXUINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, ARG_CLOCK_RATE,
      g_param_spec_uint ("clock-rate", "Clock Rate",
          "clock-rate", 0, G_MAXUINT, 0, G_PARAM_READWRITE));

  gobject_class->finalize = gst_rtp_jitter_buffer_finalize;

  gstelement_class->change_state = gst_rtp_jitter_buffer_change_state;
  /* gstelement_class->provide_clock = gst_rtp_jitter_buffer_provide_clock; */

  GST_DEBUG_CATEGORY_INIT (rtpjitterbuffer_debug, "rtpjitterbuffer", 0,
      "Jitter Buffer for RTP packets");
}

static void
gst_rtp_jitter_buffer_init (GstRTPJitterBuffer * jbuffer, gpointer g_class)
{
  GstPadTemplate *pad_template;

  GST_DEBUG_OBJECT (jbuffer, "init");

  pad_template =
      gst_element_class_get_pad_template (GST_ELEMENT_CLASS (g_class), "sink");
  g_return_if_fail (pad_template != NULL);
  jbuffer->sinkpad = gst_pad_new_from_template (pad_template, "sink");
  gst_pad_set_setcaps_function (jbuffer->sinkpad,
      gst_rtp_jitter_buffer_sink_setcaps);
  gst_pad_set_chain_function (jbuffer->sinkpad, gst_rtp_jitter_buffer_chain);
  gst_pad_set_event_function (jbuffer->sinkpad,
      gst_rtp_jitter_buffer_handle_sink_event);
  gst_element_add_pad (GST_ELEMENT (jbuffer), jbuffer->sinkpad);

  pad_template =
      gst_element_class_get_pad_template (GST_ELEMENT_CLASS (g_class), "src");
  g_return_if_fail (pad_template != NULL);
  jbuffer->srcpad = gst_pad_new_from_template (pad_template, "src");
  gst_pad_set_getrange_function (jbuffer->srcpad,
      GST_DEBUG_FUNCPTR (gst_rtp_jitter_buffer_getrange));
  gst_pad_set_activatepull_function (jbuffer->srcpad,
      GST_DEBUG_FUNCPTR (gst_rtp_jitter_buffer_src_activate_pull));

  gst_element_add_pad (GST_ELEMENT (jbuffer), jbuffer->srcpad);

  jbuffer->mode = GST_ACTIVATE_PUSH;
  jbuffer->queue = g_queue_new ();
  QUEUE_LOCK_INIT (jbuffer);
  jbuffer->availability = g_cond_new ();

  jbuffer->queue_delay = RTP_QUEUE_DELAY;
  jbuffer->need_newsegment = TRUE;
  jbuffer->offset = 0;

  jbuffer->system_clock = gst_system_clock_obtain ();
  jbuffer->rtp_clock = gst_rtp_clock_new (0);

  /* we need to be supplied the appropriate value for this */
  jbuffer->clock_rate = 0;

  jbuffer->sleep_time = DEFAULT_SLEEP_TIME;
  jbuffer->first_time_received = GST_CLOCK_TIME_NONE;
  jbuffer->num_packets = 0;
}

static void
gst_rtp_jitter_buffer_finalize (GObject * object)
{
  GstRTPJitterBuffer *jbuffer = GST_RTP_JITTER_BUFFER (object);

  QUEUE_LOCK_FREE (jbuffer);
  g_cond_free (jbuffer->availability);
  g_queue_free (jbuffer->queue);
  gst_object_unref (GST_OBJECT (jbuffer->system_clock));
  gst_object_unref (GST_OBJECT (jbuffer->rtp_clock));

  if (G_OBJECT_CLASS (parent_class)->finalize)
    G_OBJECT_CLASS (parent_class)->finalize (object);
}
/* XXX: doesn't work atm, commented out
static GstClock *
gst_rtp_jitter_buffer_provide_clock (GstElement * element)
{
  return GST_RTP_JITTER_BUFFER (element)->rtp_clock;
}
*/

static gboolean
gst_rtp_jitter_buffer_sink_setcaps (GstPad * pad, GstCaps * caps)
{
  GstRTPJitterBuffer *jbuffer;
  GstStructure *structure;

  jbuffer = GST_RTP_JITTER_BUFFER (gst_pad_get_parent (pad));
  g_return_val_if_fail (jbuffer != NULL, FALSE);
  g_return_val_if_fail (GST_IS_RTP_JITTER_BUFFER (jbuffer), FALSE);

  structure = gst_caps_get_structure (caps, 0);

  if (structure)
    gst_structure_get_int (structure, "clock-rate",
        (gint *) & (jbuffer->clock_rate));

  if (jbuffer->queue_delay != 0 || jbuffer->mode != GST_ACTIVATE_PUSH) {
    /* Flush the queue */
    QUEUE_LOCK (jbuffer);
    GST_DEBUG_OBJECT (jbuffer,
        "clock-rate changed, poping all packet from queue");
    while (!g_queue_is_empty (jbuffer->queue)) {
      GstBuffer *in = g_queue_pop_head (jbuffer->queue);

      gst_rtp_jitter_buffer_push (jbuffer, in);
    }
    QUEUE_UNLOCK (jbuffer);
  }
  gst_object_unref(jbuffer);
  return TRUE;
}

static GstFlowReturn
gst_rtp_jitter_buffer_chain (GstPad * pad, GstBuffer * in)
{
  GstRTPJitterBuffer *jbuffer;
  GstFlowReturn ret = GST_FLOW_OK;
  GstClockTime now;

  jbuffer = GST_RTP_JITTER_BUFFER (GST_OBJECT_PARENT (pad));

  if (jbuffer->clock_rate <= 0)
  {
    GST_DEBUG_OBJECT (jbuffer, "Buffer does not have a clock_rate! skipping!");
    gst_buffer_unref (in);
    return GST_FLOW_OK;
  }

  if (jbuffer->queue_delay == 0 && jbuffer->mode == GST_ACTIVATE_PUSH) {
    GST_DEBUG_OBJECT (jbuffer, "Pushing directly!");
    gst_rtp_jitter_buffer_push (jbuffer, in);
  } else {
    if (GST_ELEMENT_CLOCK (jbuffer) != NULL) {
      now = gst_clock_get_time (jbuffer->system_clock);
      jbuffer->num_packets++;

      QUEUE_LOCK (jbuffer);
      if (GST_CLOCK_TIME_IS_VALID (jbuffer->first_time_received)) {
        jbuffer->sleep_time =
            ABS (GST_CLOCK_DIFF (jbuffer->first_time_received,
                now)) / jbuffer->num_packets;
        if (jbuffer->sleep_time < DEFAULT_SLEEP_TIME_MIN) {
            /* limit the minimum sleep interval */
            jbuffer->sleep_time = DEFAULT_SLEEP_TIME_MIN;
        }
      }

      else {
        jbuffer->first_time_received = now;
      }

      QUEUE_UNLOCK (jbuffer);
    }

    gst_rtp_jitter_buffer_add_to_queue (jbuffer, in);
  }

  return ret;
}

static GstFlowReturn
gst_rtp_jitter_buffer_getrange (GstPad * pad, guint64 offset,
    guint length, GstBuffer ** buffer)
{
  GstRTPJitterBuffer *jbuffer;

  jbuffer = GST_RTP_JITTER_BUFFER (GST_PAD_PARENT (pad));

  /* TODO: how to handle offset and length or how would the sink query us
   *       if we support these parameters or not? */
  QUEUE_LOCK (jbuffer);
  *buffer = g_queue_pop_head (jbuffer->queue);
  while (*buffer == NULL) {
    g_cond_wait (jbuffer->availability, jbuffer->queuelock);
    *buffer = g_queue_pop_head (jbuffer->queue);
  }

  gst_rtp_jitter_buffer_set_gst_timestamp (jbuffer, *buffer);
  QUEUE_UNLOCK (jbuffer);

  return GST_FLOW_OK;
}

static gboolean
gst_rtp_jitter_buffer_handle_sink_event (GstPad * pad, GstEvent * event)
{
  GstRTPJitterBuffer *jbuffer = GST_RTP_JITTER_BUFFER (GST_OBJECT_PARENT (pad));
  gboolean res = TRUE;

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_NEWSEGMENT:
    {
      GST_DEBUG_OBJECT (jbuffer,
          "Upstream sent a NEWSEGMENT, handle in worker thread.");
      /* the worker thread will assign a new RTP-TS<->GST-TS mapping
       * based on the next processed RTP packet */
      jbuffer->need_newsegment = TRUE;
      gst_event_unref (event);
      break;
    }
    default:
      /* pass other events forward */
      res = gst_pad_push_event (jbuffer->srcpad, event);
      break;
  }

  return res;
}

static GstFlowReturn
gst_rtp_jitter_buffer_add_to_queue (GstRTPJitterBuffer * jbuffer,
    GstBuffer * in)
{
  guint16 seqnum;
  guint32 timestamp;
  GQueue *queue = jbuffer->queue;
  GList *list;
  int inserts = 0;
  guint jbtsunits = (gfloat) jbuffer->clock_rate * jbuffer->queue_delay / 1000.0f;
  guint queue_len = g_queue_get_length (queue);

  seqnum = gst_rtp_buffer_get_seq (in);
  timestamp = gst_rtp_buffer_get_timestamp (in);

  GST_DEBUG_OBJECT (jbuffer,
		    "Packet to be queued (size %d), timestamp %u sn %d",
		    queue_len, gst_rtp_buffer_get_timestamp (in), seqnum);

  QUEUE_LOCK (jbuffer);
  if (queue_len == 0) {
    /* case: our first packet, just push it */
    g_queue_push_tail (queue, in);
    ++inserts;
  } 
  else if (queue_len > GST_RTP_JB_MAX_PACKETS) {
    /* case: buffer overflow, drop */
    GST_DEBUG_OBJECT (jbuffer, "buffer overflow, dropping packet#%d!", seqnum);
    gst_buffer_unref (in);
  }
  else if (priv_delta_rtp_ts(timestamp, jbuffer->min_ts) > 
	   jbtsunits * GST_RTP_JB_OVERFLOW_LIMIT) {
    /* case: late packet, drop */
    GST_DEBUG_OBJECT (jbuffer, 
		      "packet#%d too late, dropping (ts-delta=%lu, max=%u!", 
		      seqnum, priv_delta_rtp_ts(timestamp, jbuffer->min_ts),
		      jbtsunits * GST_RTP_JB_OVERFLOW_LIMIT);
    gst_buffer_unref (in);
    
  }
  else {
    /* case: packet ready to be queued, find correct place and insert */

    guint16 queueseq;

    /* note: the common case is to add the packet at the end, so we'll
     *       start from tail */
    for (list =  queue->tail; list && inserts == 0;) {
      queueseq = gst_rtp_buffer_get_seq (GST_BUFFER (list->data));
      if (priv_compare_rtp_seq_lt(seqnum, queueseq)) {
          list = list->prev;
          if (list == NULL) {
              /* reached start of the queue (min-seqno), pushing at head */
              g_queue_push_head (queue, in);
              ++inserts;
          }
      }
      else if (seqnum == queueseq) {
          /* it's a duplicate, drop it on floor! */
          gst_buffer_unref (in);
          GST_DEBUG_OBJECT (jbuffer,
                  "Duplicate packet for packet#%d detected, dropping!", seqnum);
          break;
      }
      else {
          /* insert after current item as 'in->seqno > q->seqno'  */
          g_queue_insert_after (queue, list, in);
          ++inserts;
      }
    }
  }

  /* step: update last stats */
  jbuffer->min_ts = gst_rtp_buffer_get_timestamp (GST_BUFFER (queue->head->data));

  QUEUE_UNLOCK (jbuffer);

  if (inserts) {
    GST_DEBUG_OBJECT (jbuffer,
		      "Packet added to queue (size %d) sn %d",
		      g_queue_get_length (queue), seqnum);

    g_cond_signal (jbuffer->availability);

    /* note: the cond_signal() mechanism is not reliable/fast enough 
     *       to wakeup the queue_release mechanism, so it is triggered
     *       here as well */
    gst_rtp_jitter_buffer_queue_release (jbuffer);
  }

  return GST_FLOW_OK;
}

static void
gst_rtp_jitter_buffer_push (GstRTPJitterBuffer * jbuffer, GstBuffer * buf)
{
  gst_rtp_jitter_buffer_set_gst_timestamp (jbuffer, buf);

  /* push it */
  GST_DEBUG_OBJECT (jbuffer, "Pushing buffer size %d", GST_BUFFER_SIZE (buf));
  gst_pad_push (jbuffer->srcpad, GST_BUFFER (buf));
  GST_DEBUG_OBJECT (jbuffer, "Pushed buffer");
}

static void
gst_rtp_jitter_buffer_set_gst_timestamp (GstRTPJitterBuffer * jbuffer,
    GstBuffer * buf)
{
  /* rtp timestamps are based on the clock_rate
   * gst timesamps are in nanoseconds
   */
  guint32 timestamp = gst_rtp_buffer_get_timestamp (buf);
  guint64 ts = ((timestamp * GST_SECOND) / jbuffer->clock_rate);

  GST_DEBUG_OBJECT (jbuffer, "calculating ts : timestamp : %u, clockrate : %u",
      timestamp, jbuffer->clock_rate);

  /* add delay to timestamp */
  GST_BUFFER_TIMESTAMP (buf) = ts + (jbuffer->queue_delay * GST_MSECOND);

  GST_DEBUG_OBJECT (jbuffer, "calculated ts %"
      GST_TIME_FORMAT, GST_TIME_ARGS (GST_BUFFER_TIMESTAMP (buf)));

  /* if this is the first buf send a discont */
  if (jbuffer->need_newsegment) {
#if 0
    /* does not work, commented out */
    gst_rtp_clock_update (GST_RTP_CLOCK (jbuffer->rtp_clock), ts, TRUE);
#endif

    /* send discont */
    GstEvent *event = gst_event_new_new_segment (FALSE, 1.0, GST_FORMAT_TIME,
        GST_BUFFER_TIMESTAMP (buf), GST_CLOCK_TIME_NONE, 0);

    gst_pad_push_event (jbuffer->srcpad, event);
    jbuffer->need_newsegment = FALSE;
    jbuffer->offset = jbuffer->sleep_time = jbuffer->num_packets = 0;
    jbuffer->sleep_time = DEFAULT_SLEEP_TIME;
    jbuffer->first_time_received = GST_CLOCK_TIME_NONE;
    GST_DEBUG_OBJECT (jbuffer, "Pushed newsegment event on this first buffer");
  }

  else {
    gst_rtp_clock_update (GST_RTP_CLOCK (jbuffer->rtp_clock), ts, FALSE);
  }
}

/**
 * Pushes out any packets that have reached 'playable' status,
 * from the jitter buffer.
 *
 * Note: can be called from multiple threads
 */
static void
gst_rtp_jitter_buffer_queue_release (GstRTPJitterBuffer * jbuffer)
{
  GQueue *queue = jbuffer->queue;
  guint32 headts, tailts;
  GstRTPJitterBufferClass *bclass;

  QUEUE_LOCK (jbuffer);

  if (g_queue_is_empty (queue)) {
    goto end;
  }

  /* if our queue is getting too big (more than RTP_QUEUEDELAY ms of data)
   * release heading buffers */
  GST_DEBUG_OBJECT (jbuffer, "clockrate %d, queue_delay %d",
      jbuffer->clock_rate, jbuffer->queue_delay);

  gfloat q_size_secs = (gfloat) jbuffer->queue_delay / 1000;
  guint maxtsunits = (gfloat) jbuffer->clock_rate * q_size_secs;

  bclass = GST_RTP_JITTER_BUFFER_GET_CLASS (jbuffer);
  tailts =
    gst_rtp_buffer_get_timestamp (GST_BUFFER (g_queue_peek_tail (queue)));
  
  while (g_queue_is_empty (queue) != TRUE) {

    headts =
      gst_rtp_buffer_get_timestamp (GST_BUFFER (g_queue_peek_head (queue)));

    GST_DEBUG("maxtsunit is %u %u %u %u", maxtsunits, headts, tailts, headts - tailts);
    
    if (priv_delta_rtp_ts(tailts, headts) <= maxtsunits)
      break;
    
    GST_DEBUG_OBJECT (jbuffer, "Poping packet from queue");
    
    GstBuffer *in = g_queue_pop_head (queue);
    gst_rtp_jitter_buffer_push (jbuffer, in);
  }

 end:
  QUEUE_UNLOCK (jbuffer);
}


static gpointer
gst_rtp_jitter_buffer_thread (GstRTPJitterBuffer * jbuffer)
{
  while (jbuffer->thread_running) {
    /* note: queue_release is also called from add_to_queue() to 
     *       make ensure as steady stream of released packets as
     *       possible */
    gst_rtp_jitter_buffer_queue_release (jbuffer);
    g_usleep (jbuffer->sleep_time / 1000);
  }
  return NULL;
}

static gboolean
gst_rtp_jitter_buffer_start_thread (GstRTPJitterBuffer * jbuffer)
{
  /* only launch the thread if processing is needed */
  if (jbuffer->queue_delay) {
    GST_DEBUG_OBJECT (jbuffer, "Starting queue release thread");
    jbuffer->thread_running = TRUE;
    jbuffer->thread = g_thread_create ((GThreadFunc) gst_rtp_jitter_buffer_thread,
				       jbuffer, TRUE, NULL);
    GST_DEBUG_OBJECT (jbuffer, "Started queue release thread");
  }
  return TRUE;
}

static gboolean
gst_rtp_jitter_buffer_stop_thread (GstRTPJitterBuffer * jbuffer)
{
  jbuffer->thread_running = FALSE;

  if (jbuffer->thread) {
    /* Send a fake signal to wake-up the thread */
    QUEUE_LOCK (jbuffer);
    g_cond_signal (jbuffer->availability);
    QUEUE_UNLOCK (jbuffer);

    g_thread_join (jbuffer->thread);
    jbuffer->thread = NULL;
  }
  return TRUE;
}

static gboolean
gst_rtp_jitter_buffer_src_activate_pull (GstPad * pad, gboolean active)
{
  GstRTPJitterBuffer *jbuffer;

  jbuffer = GST_RTP_JITTER_BUFFER (GST_OBJECT_PARENT (pad));

  if (jbuffer->mode == GST_ACTIVATE_PUSH)
    gst_rtp_jitter_buffer_stop_thread (jbuffer);

  jbuffer->mode = GST_ACTIVATE_PULL;
  jbuffer->queue_delay = 0;

  return TRUE;
}

static GstStateChangeReturn
gst_rtp_jitter_buffer_change_state (GstElement * element,
    GstStateChange transition)
{
  GstRTPJitterBuffer *jbuffer;

  g_return_val_if_fail (GST_IS_RTP_JITTER_BUFFER (element),
      GST_STATE_CHANGE_FAILURE);
  jbuffer = GST_RTP_JITTER_BUFFER (element);

  /* we disallow changing the state from the thread */
  if (g_thread_self () == jbuffer->thread)
    return GST_STATE_CHANGE_FAILURE;


  switch (transition) {
    case GST_STATE_CHANGE_NULL_TO_READY:
      if (jbuffer->mode == GST_ACTIVATE_PUSH
          && !gst_rtp_jitter_buffer_start_thread (jbuffer))
        goto start_failed;
      break;
    case GST_STATE_CHANGE_READY_TO_PAUSED:
      break;
    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      break;
    default:
      break;
  }

  switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      break;
    case GST_STATE_CHANGE_PAUSED_TO_READY:
      break;
    case GST_STATE_CHANGE_READY_TO_NULL:
      if (jbuffer->mode == GST_ACTIVATE_PUSH)
        gst_rtp_jitter_buffer_stop_thread (jbuffer);
      break;
    default:
      break;
  }
  return GST_STATE_CHANGE_SUCCESS;

  /* ERRORS */
start_failed:
  {
    return GST_STATE_CHANGE_FAILURE;
  }
}

static void
gst_rtp_jitter_buffer_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstRTPJitterBuffer *jbuffer;

  g_return_if_fail (GST_IS_RTP_JITTER_BUFFER (object));
  jbuffer = GST_RTP_JITTER_BUFFER (object);

  switch (prop_id) {
    case ARG_QUEUE_DELAY:
      jbuffer->queue_delay = g_value_get_uint (value);
      break;
    case ARG_CLOCK_RATE:
      jbuffer->clock_rate = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_rtp_jitter_buffer_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{
  GstRTPJitterBuffer *jbuffer;

  g_return_if_fail (GST_IS_RTP_JITTER_BUFFER (object));
  jbuffer = GST_RTP_JITTER_BUFFER (object);

  switch (prop_id) {
    case ARG_QUEUE_DELAY:
      g_value_set_uint (value, jbuffer->queue_delay);
      break;
    case ARG_CLOCK_RATE:
      g_value_set_uint (value, jbuffer->clock_rate);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}


static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin,
      "rtpjitterbuffer", GST_RANK_NONE, GST_TYPE_RTP_JITTER_BUFFER);
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "rtpjitterbuffer",
    "RTP Jitter Buffer",
    plugin_init, VERSION, "LGPL", "Farsight", "http://farsight.sf.net")
