//*****************************************************************************
//                                  SpinCtrl.cpp                              *
//                                 --------------                             *
//  Started     : 20/03/2004                                                  *
//  Last Update : 23/08/2007                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "utility/SpinCtrl.hpp"

//*****************************************************************************
// Implement an event table.

BEGIN_EVENT_TABLE( SpinCtrl, wxPanel )

  EVT_CHAR       (             SpinCtrl::OnTxtChar   )
  EVT_TEXT_MAXLEN( ID_TXTCTRL, SpinCtrl::OnTxtMaxLen )
  EVT_SPIN_UP    ( ID_SPINBTN, SpinCtrl::OnSbnInc    )
  EVT_SPIN_DOWN  ( ID_SPINBTN, SpinCtrl::OnSbnDec    )

END_EVENT_TABLE( )

//*****************************************************************************
// Constructor.
//
// Argument List:
//   eVType - The type of the variable to be displayed

SpinCtrl::SpinCtrl( eVarType eVType ) : wxPanel( )
{
  if( ! bSetVarType( eVType ) ) // Set the         variable type to display
    m_eVarType = eVAR_FLT;      // Set the default variable type to display
  m_fDefValue  =     0.0;       // Set the default value
  m_fMinValue  = -1000.0;       // Set the default minimum value
  m_fMaxValue  =  1000.0;       // Set the default maximum value
  m_fMinIncSz  =    10.0;       // Set the default minimum increment size
  m_fMaxIncSz  =    10.0;       // Set the default maximum increment size
}

//*****************************************************************************
// Destructor.

SpinCtrl::~SpinCtrl( )
{
}

//*****************************************************************************
// Layout the display objects.

void  SpinCtrl::DoLayout( void )
{
  wxBoxSizer * poSizer;
  wxSizerFlags  oFlags;

  poSizer = new wxBoxSizer( wxHORIZONTAL );

  oFlags.Expand( );
  oFlags.Proportion( 1 );
  poSizer->Add( &m_oTxtValue, oFlags );
  oFlags.Proportion( 0 );
  poSizer->Add( &m_oSbnValue, oFlags );

  SetSizer( poSizer );

  // Set minimum size and initial size as calculated by the sizer
  poSizer->SetSizeHints( this );
}

//*****************************************************************************
// Create an instance of this object.
//
// Argument List:
//   poWin  - The parent window
//   oWinID - The window identifier
//   iWidth - The width the spin control in pixels
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  SpinCtrl::bCreate( wxWindow * poWin, wxWindowID oWinID, int iWidth )
{
  wxSize  oSize=wxDefaultSize;

  if( bIsCreated( ) )             return( TRUE );

  // Create the display objects
  if( ! Create( poWin, oWinID ) ) return( FALSE );
  if( iWidth > 0 ) oSize.SetWidth( iWidth );
  m_oTxtValue.Create( this, ID_TXTCTRL, wxT(""), wxDefaultPosition,
                      oSize,         wxTE_RIGHT );
  m_oSbnValue.Create( this, ID_SPINBTN,          wxDefaultPosition,
                      wxDefaultSize, wxSP_VERTICAL | wxSP_ARROW_KEYS );

  // Set the display object attributes
  m_oTxtValue.SetMaxLength( SPNCTL_MAXLEN );
  m_oSbnValue.SetRange( -0x8000, 0x7FFF );
  bSetParms( m_fDefValue, m_fMinValue, m_fMaxValue, m_fMinIncSz, m_fMaxIncSz );
  bSetValue( m_fDefValue );

  // Layout the display objects
  DoLayout( );

  return( TRUE );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  SpinCtrl::bClear( void )
{
  return( bSetValue( m_fDefValue ) );
}

//*****************************************************************************
// Set the variable type to displayed by the spin control.
//
// Argument List:
//   eVType - The variable type specifier
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  SpinCtrl::bSetVarType( eVarType eVType )
{
  float  f1;

  // Argument validity checks
  if( m_eVarType == eVType )               return( TRUE );
  if( eVType<eVAR_FST || eVType>eVAR_LST ) return( FALSE );

  m_eVarType = eVType; // Set the new variable type

  // Display the new variable type
  if( bIsCreated( ) )
  {
    f1 = fGetValue( );
    if( ! bSetValue( f1 ) )                return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Set the current value of the spin control.
//
// Argument List:
//   fValue - The float value
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  SpinCtrl::bSetValue( float fValue )
{
  wxString  osValue;
  float     f1;
  int       i1;

  if( ! bIsCreated( ) )      return( FALSE );
  if( fValue < m_fMinValue )
    if( !( (m_eVarType==eVAR_SCI || m_eVarType==eVAR_ENG) && fValue==0.0 ) )
                             return( FALSE );
  if( fValue > m_fMaxValue ) return( FALSE );

  m_oTxtValue.Clear( );
  switch( m_eVarType )
  {
    case eVAR_HEX :
      if( fValue < 0.0 )                 return( FALSE );
      i1 = (int) ( fValue + 0.5 );
      osValue.Printf( wxT("%X"), i1 );
      break;

    case eVAR_INT :
      if( fValue >= 0.0 ) i1 = (int) ( fValue + 0.5 );
      else                i1 = (int) ( fValue - 0.5 );
      osValue << i1;
      break;

    case eVAR_FLT :
      osValue.Printf( wxT("%#.2f"), fValue );
      break;

    case eVAR_SCI :
      osValue.Printf( wxT("%#.2E"), fValue );
      break;

    case eVAR_ENG :
      if( ! ConvertType::bParseFlt( (double) fValue, &f1, &i1 ) )
        return( FALSE );
      while( i1 % 3 ) { f1 *= 10.0; i1--; }
      osValue.Printf( wxT("%#.2fE%02i"), f1, i1 );
      break;

    default:                 return( FALSE );
  }

  osValue.Append( wxT(" ") );
  m_oTxtValue.SetValue( osValue );

  return( TRUE );
}

//*****************************************************************************
// Set the current value of the spin control.
//
// Argument List:
//   fValue - The string value
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  SpinCtrl::bSetValue( const wxString & rosValue )
{
  double  fValue;

  if( ! ConvertType::bStrToFlt( rosValue, &fValue ) ) return( FALSE );
  if( ! bSetValue( (float) fValue ) )                 return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Set the initial value of the spin control.
//
// Argument List:
//   fDefValue - The spin control default value
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  SpinCtrl::bSetDefValue( float fDefValue )
{
  if( fDefValue<m_fMinValue || fDefValue>m_fMaxValue ) return( FALSE );

  m_fDefValue = fDefValue;

  return( TRUE );
}

//*****************************************************************************
// Set the increment sizes of the spin control.
//
// This spin control can be incremented using two different approaches,
// constant or variable step sizes. A constant step size means that the spin
// control is incremented by the same amount throughout it's range. A variable
// step size means that the spin control is incremented by an amount dependent
// on it's current value.
//
// Argument List:
//   fMinIncSz - The minimum spin control increment size
//   fMaxIncSz - The maximum spin control increment size
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  SpinCtrl::bSetIncSz( float fMinIncSz, float fMaxIncSz )
{
  // Constant or variable incrementing?
  if( fMinIncSz < 0.0 )                         return( FALSE );
  if( fMaxIncSz < 0.0 ) fMaxIncSz = fMinIncSz;

  // Do some validity checks on the arguments
  if( fMinIncSz > fMaxIncSz )                   return( FALSE );
  // Check that the new increment sizes fit within the current range
  if( fMaxIncSz > (m_fMaxValue - m_fMinValue) ) return( FALSE );

  m_fMinIncSz = fMinIncSz; // Set increment sizes
  m_fMaxIncSz = fMaxIncSz;

  return( TRUE );
}

//*****************************************************************************
// Set the spincontrol range.
//
// Argument List:
//   fMinValue - The spin control minimum value
//   fMaxValue - The spin control maximum value
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  SpinCtrl::bSetRange( float fMinValue, float fMaxValue )
{
  return( bSetParms( m_fDefValue, fMinValue, fMaxValue ) );
}

//*****************************************************************************
// Set all the spin control parameters.
//
// Argument List:
//   fDefValue - The spin control default value
//   fMinValue - The spin control minimum value
//   fMaxValue - The spin control maximum value
//   fMinIncSz - The spin control minimum increment size (must be positive)
//   fMaxIncSz - The spin control maximum increment size (must be positive)
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  SpinCtrl::bSetParms( float fDefValue, float fMinValue, float fMaxValue,
                           float fMinIncSz, float fMaxIncSz )
{
  wxString  os1;
  double    df1;

  // First decide what needs to be done with the increment sizes
  if(      fMinIncSz < 0.0 ) { fMinIncSz = m_fMinIncSz; fMaxIncSz = m_fMaxIncSz; }
  else if( fMaxIncSz < 0.0 ) { fMaxIncSz = fMinIncSz; }

  // Check argument validity
  if( fMinValue >= fMaxValue )                         return( FALSE );
  if( m_eVarType!=eVAR_SCI && m_eVarType!=eVAR_ENG )
  {
    if( fDefValue<fMinValue || fDefValue>fMaxValue )   return( FALSE );
    if( fMinIncSz > fMaxIncSz )                        return( FALSE );
    if( fMaxIncSz > (fMaxValue - fMinValue) )          return( FALSE );
    if( fMinIncSz<=0.0 || fMaxIncSz<=0.0 )             return( FALSE );
  }
  else
  {
    if( fDefValue != 0.0 )
      if( fDefValue<fMinValue || fDefValue>fMaxValue ) return( FALSE );
  }

  // Set new parameter values
  m_fDefValue = fDefValue;
  m_fMinValue = fMinValue;
  m_fMaxValue = fMaxValue;
  if( m_eVarType!=eVAR_SCI && m_eVarType!=eVAR_ENG )
    if( ! bSetIncSz( fMinIncSz, fMaxIncSz ) )          return( FALSE );

  // Update the text control if necessary
  if( bIsCreated( ) )
  {
    os1 = m_oTxtValue.GetValue( );
    ConvertType::bStrToFlt( os1, &df1 );
    if(      df1 < m_fMinValue ) bSetValue( m_fMinValue );
    else if( df1 > m_fMaxValue ) bSetValue( m_fMaxValue );
  }

  return( TRUE );
}

//*****************************************************************************
// Get the current spin control value as an integer.
//
// Return Values:
//   Success - The integer value
//   Failure - INT_MIN

int  SpinCtrl::iGetValue( void )
{
  float  f1;
  int    i1;

  // Get the value as a float
  f1 = fGetValue( );
  if( f1 == -FLT_MAX )  return( INT_MIN );

  // Round the value
  if( f1 >= 0.0 ) i1 = (int) ( f1 + 0.5 );
  else            i1 = (int) ( f1 - 0.5 );

  return( i1 );
}

//*****************************************************************************
// Get the current spin control value as a float.
//
// Return Values:
//   Success - The float value
//   Failure - -FLT_MAX

float  SpinCtrl::fGetValue( void )
{
  wxString  os1;
  double    df1;

  // Has the control been created?
  if( ! bIsCreated( ) )                       return( -FLT_MAX );

  // Get the value as a float
  os1 = m_oTxtValue.GetValue( );
  if( ! ConvertType::bStrToFlt( os1, &df1 ) ) return( -FLT_MAX );

  return( (float) df1 );
}

//*****************************************************************************
// Get the current spin control value as a string.
//
// Return Values:
//   Success - The string value
//   Failure - An empty string

const wxString & SpinCtrl::rosGetValue( void )
{
  static  wxString  osValue;

  osValue.Empty( );

  if( fGetValue( ) != FLT_MAX )
  {
    osValue = m_oTxtValue.GetValue( );
    osValue.Trim( );
  }

  return( osValue );
}

//*****************************************************************************
//
//                             Event Handlers
//
//*****************************************************************************
// Key press event handler.
//
// Argument List:
//   roEvtKey - An object holding information about the event

void  SpinCtrl::OnTxtChar( wxKeyEvent & roEvtKey )
{
  wxSpinEvent  oEvtSpn;
  int          iKeyCode;

  // Look for the modifier keys <Ctrl> and <Alt>
  if( roEvtKey.HasModifiers( ) ) return;

  // Get the key code
  iKeyCode = roEvtKey.GetKeyCode( );

  // Process arrow keys etc.
  if(      iKeyCode == WXK_UP   ) { OnSbnInc( oEvtSpn ); return; }
  else if( iKeyCode == WXK_DOWN ) { OnSbnDec( oEvtSpn ); return; }

  // Test that the key code is valid and reject invalid characters
  // ??? (07/07/2007) This doesn't work but it would be good if it did. Only up
  // and down arrow keys get through to this function, don't know why.
  switch( iKeyCode )
  {
    case (int) '0' :
    case (int) '1' :
    case (int) '2' :
    case (int) '3' :
    case (int) '4' :
    case (int) '5' :
    case (int) '6' :
    case (int) '7' :
    case (int) '8' :
    case (int) '9' :
    case (int) '+' :
    case (int) '-' :
    case (int) 'e' :
    case (int) 'E' :
    case WXK_BACK :
    case WXK_DELETE :
      break;
    default :
      wxBell( );  // Sound the system bell
std::cerr << "SpinCtrl::OnTxtChar( ) - Invalid character\n"; // ??? 08/08/2007
  }
}

//*****************************************************************************
// Text control maximum length reached event handler.
//
// Argument List:
//   roEvtCmd - An object holding information about the event

void  SpinCtrl::OnTxtMaxLen( wxCommandEvent & roEvtCmd )
{
  wxBell( );  // Sound the system bell
}

//*****************************************************************************
// Spin button scroll up event handler.
//
// Argument List:
//   roEvtSpn - An object holding information about the event

void  SpinCtrl::OnSbnInc( wxSpinEvent & roEvtSpn )
{
  wxString  os1;
  double    df1;
  float     fValue, fIncSz, f1;
  int       i1;

  // Get the current text control value
  os1 = m_oTxtValue.GetValue( );
  df1 = 0.0;
  if( ! ConvertType::bStrToFlt( os1, &df1 ) )     return;
  if( ! ConvertType::bParseFlt( df1, &f1, &i1 ) ) return;
  fValue = (float) df1;

  // Determine the increment size to use
  if( f1 == -1.0 ) i1--;
  if( m_eVarType==eVAR_SCI || m_eVarType==eVAR_ENG )
  {
    fIncSz = (float) pow( 10.0, (double) i1-1 );
    if( fValue == 0.0 ) fIncSz = m_fMinValue;
  }
  else
  {
    fIncSz = (float) pow( 10.0, (double) i1 );
    if( fIncSz < m_fMinIncSz ) fIncSz = m_fMinIncSz;
    if( fIncSz > m_fMaxIncSz ) fIncSz = m_fMaxIncSz;
  }

  // Determine if the text control has been altered by the user
  f1 = ceilf( fValue / fIncSz ) * fIncSz - fValue;
  // Allow for small errors in the floating point calculations
  if( fabs(f1 / fIncSz)>0.0001 && fabs((f1 - fIncSz) / fIncSz)>0.0001 )
    fIncSz = f1;

  // Increment the value
  if( fValue < m_fMaxValue )
  {
    fValue += fIncSz;
    if( fValue > m_fMaxValue ) fValue = m_fMaxValue;
  }
  else if( m_eVarType==eVAR_SCI || m_eVarType==eVAR_ENG )
  {
    if( fValue < m_fMinValue ) fValue = m_fMinValue;
  }

  bSetValue( fValue );    // Set the text control value

  roEvtSpn.Skip( true );  // Allow additional event handlers to be called
}

//*****************************************************************************
// Spin button scroll down event handler.
//
// Argument List:
//   roEvtSpn - An object holding information about the event

void  SpinCtrl::OnSbnDec( wxSpinEvent & roEvtSpn )
{
  wxString  os1;
  double    df1=0.0;
  float     fValue, fIncSz, f1;
  int       i1;

  // Get the current text control value
  if( ! ConvertType::bStrToFlt( m_oTxtValue.GetValue( ), &df1 ) ) return;
  if( ! ConvertType::bParseFlt( df1, &f1, &i1 ) )                 return;
  fValue = (float) df1;

  // Determine the decrement size to use
  if( f1 == 1.0 ) i1--;
  if( m_eVarType==eVAR_SCI || m_eVarType==eVAR_ENG )
  {
    fIncSz = (float) pow( 10.0, (double) i1-1 );
    if( fValue == 0.0 ) fIncSz = m_fMinValue;
  }
  else
  {
    fIncSz = (float) pow( 10.0, (double) i1 );
    if( fIncSz < m_fMinIncSz ) fIncSz = m_fMinIncSz;
    if( fIncSz > m_fMaxIncSz ) fIncSz = m_fMaxIncSz;
  }

  // Determine if the text control has been altered by the user
  f1 = fValue - floorf( fValue / fIncSz ) * fIncSz;
  // Allow for small errors in the floating point calculations
  if( fabs(f1 / fIncSz)>0.0001 && fabs((f1 - fIncSz) / fIncSz)>0.0001 )
    fIncSz = f1;

  // Decrement the value
  if( fValue > m_fMinValue )
  {
    fValue -= fIncSz;
    if( fValue < m_fMinValue ) fValue = m_fMinValue;
  }
  else if( m_eVarType==eVAR_SCI || m_eVarType==eVAR_ENG )
  {
    fValue = 0.0;
  }

  bSetValue( fValue );    // Set the text control value

  roEvtSpn.Skip( true );  // Allow additional event handlers to be called
}

//*****************************************************************************
