//*****************************************************************************
//                                GNetList.cpp                                *
//                               --------------                               *
//  Started     : 28/01/2004                                                  *
//  Last Update : 10/05/2006                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "process/PrcGNetList.hpp"

#include <wx/arrimpl.cpp>
WX_DEFINE_OBJARRAY( ArrayFileName );

//*****************************************************************************
// Constructor.

PrcGNetList::PrcGNetList( void ) : PrcBase( wxPROCESS_REDIRECT )
{
  // Clear the object attributes
  bClear( );

  // Initialize the array of Guile procedure names and set the default
  InitGuileProcs( );

  // Attempt to set and find the gnetlist binary
  bSetBinary( GNLST_BIN_NAME );

  // Set the log file name
  bSetLogFile( GNLST_LOG_FILE );
}

//*****************************************************************************
// Destructor.

PrcGNetList::~PrcGNetList( )
{
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGNetList::bClear( void )
{
  bSetSchems   ( wxT("") );
  bSetNetList  ( wxT("") );
  bSetGuileProc( wxT("") );

  return( TRUE );
}

//*****************************************************************************
// Initialize the array of possible Guile procedure names.

void  PrcGNetList::InitGuileProcs( void )
{
  m_oasGuileProcs.Clear( );

  m_oasGuileProcs.Add( wxT("")          ); // None
  m_oasGuileProcs.Add( wxT("allegro")   ); // Allegro netlist format
  m_oasGuileProcs.Add( wxT("bae")       ); // BAE netlist format
  m_oasGuileProcs.Add( wxT("bom")       ); // BOM2 - Bill of Materials
  m_oasGuileProcs.Add( wxT("bom2")      ); // BOM  - Bill of Materials
  m_oasGuileProcs.Add( wxT("drc")       ); // DRC - Start of a design rule checker
  m_oasGuileProcs.Add( wxT("geda")      ); // gEDA - native format, mainly used for testing
  m_oasGuileProcs.Add( wxT("gossip")    ); // Gossip netlist format
  m_oasGuileProcs.Add( wxT("pads")      ); // PADS netlist format
  m_oasGuileProcs.Add( wxT("PCB")       ); // PCB
  m_oasGuileProcs.Add( wxT("PCBboard")  ); // PCBboard
  m_oasGuileProcs.Add( wxT("protelII")  ); // ProtelII netlist format
  m_oasGuileProcs.Add( wxT("spice")     ); // Spice compatible netlist format
  m_oasGuileProcs.Add( wxT("spice-sdb") ); // Enhanced spice compatible netlist format
  m_oasGuileProcs.Add( wxT("switcap")   ); // Switcap netlist format
  m_oasGuileProcs.Add( wxT("tango")     ); // Tango netlist format
  m_oasGuileProcs.Add( wxT("verilog")   ); // Verilog code
  m_oasGuileProcs.Add( wxT("vhdl")      ); // VHDL code
  m_oasGuileProcs.Add( wxT("vipec")     ); // VIPEC netlist format
}

//*****************************************************************************
// Set the schematic to net list conversion utility file name.
//
// Argument List:
//   rosFileName - The full path and file name of the GNetList binary
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGNetList::bSetGNetList( const wxString & rosFileName )
{
  return( bSetBinary( rosFileName.c_str( ) ) );
}

//*****************************************************************************
// Set the schematic file name/s.
//
// Argument List:
//   rosFileNames - A string containing the full path and file name/s
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGNetList::bSetSchems( const wxString & rosFileNames )
{
  wxStringTokenizer  ostk1;
  wxArrayString      oas1;

  ostk1.SetString( rosFileNames );
  while( ostk1.HasMoreTokens( ) )
    oas1.Add( ostk1.GetNextToken( ) );

  return( bSetSchems( oas1 ) );
}

//*****************************************************************************
// Set the schematic file name/s.
//
// Argument List:
//   roasFileNames - A string array containing the full path and file name/s
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGNetList::bSetSchems( const wxArrayString & roasFileNames )
{
  wxFileName  ofn1;
  size_t      szt1;

  // Clear the current list of schematic files
  m_oafnSchems.Clear( );

  // Check the argument is empty
  if( roasFileNames.IsEmpty( ) ) return( TRUE );

  // Add the new schematic file name/s to the list
  for( szt1=0; szt1<roasFileNames.GetCount( ); szt1++ )
  {
    ofn1 = roasFileNames.Item( szt1 );
    if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( wxT(".") );

    if( ! ofn1.IsOk( )       ) continue;
    if( ! ofn1.FileExists( ) ) continue;

    m_oafnSchems.Add( ofn1 );
  }

  // Check that at least one schematic file name was accepted
  if( m_oafnSchems.IsEmpty( ) )  return( FALSE );

  // Set the log file path
  ofn1 = rofnGetLogFile( );
  ofn1.SetPath( m_oafnSchems.Item( 0 ).GetPath( ) );
  bSetLogFile( ofn1.GetFullPath( ) );

  // Check if any of the schematic files were invalid
  if( m_oafnSchems.GetCount( ) != roasFileNames.GetCount( ) ) return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Set the full net list file name.
//
// Argument List:
//   rosFileName - A string containing the full path and file name
//                 (If rosFileName = "use-schem-name" the net list name is to
//                  be generated from the schematic file name)
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGNetList::bSetNetList( const wxString & rosFileName )
{
  wxFileName  ofn1;

  // Set the net list name
  if( ! rosFileName.IsEmpty( ) )
  {
    if( rosFileName == GNLST_USE_SCHEM )
    { // Generate the net list file name from the schematic file name
      ofn1 = rofnGetSchem( );
      ofn1.SetExt( wxT("ckt") );
    }
    else ofn1 = rosFileName;

    // Set the net list path if it hasn't been set
    if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( wxT(".") );

    // Check that the file name is OK
    if( ! ofn1.IsOk( ) ) return( FALSE );
  }
  else m_ofnNetList.Clear( );

  // Set the net list file path and name
  m_ofnNetList = ofn1;

  // Set the log file path
  ofn1 = rofnGetLogFile( );
  ofn1.SetPath( m_ofnNetList.GetPath( ) );
  bSetLogFile( ofn1.GetFullPath( ) );

  return( TRUE );
}

//*****************************************************************************
// Set the Guile procedure to be used to import the schematic file.
//
// Argument List:
//   rosGuileProc - The Guile procedure name
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGNetList::bSetGuileProc( const wxString & rosGuileProc )
{
  int  i1;

  i1 = m_oasGuileProcs.Index( rosGuileProc );
  if( i1 == wxNOT_FOUND ) return( FALSE );

  m_sztGuileProc = (size_t) i1;

  return( TRUE );
}

//*****************************************************************************
// Get a schematic file from the list of schematic files.
//
// Argument List:
//   sztIndex - A zero based index to the file name
//
// Return Values:
//   Success - The procedure name
//   Failure - An empty file name

const wxFileName & PrcGNetList::rofnGetSchem( size_t sztIndex )
{
  static  wxFileName  oFnEmpty;

  if( sztIndex<0 || sztIndex>=sztGetSchemCnt( ) ) return( oFnEmpty );

  return( m_oafnSchems[ sztIndex ] );
}

//*****************************************************************************
// Get a Guile procedure from the list of procedure names.
//
// Argument List:
//   sztIndex - A zero based index to the procedure name
//              (If sztIndex = CUR_SEL_PROC return currently selected procedure)
//
// Return Values:
//   Success - The procedure name
//   Failure - An empty string

const wxString & PrcGNetList::rosGetGuileProc( size_t sztIndex )
{
  static  wxString  osEmpty;

  if( sztIndex == GNLST_CUR_PROC ) sztIndex = m_sztGuileProc;

  if( sztIndex >= sztGetGuileProcCnt( ) ) return( osEmpty );

  return( m_oasGuileProcs[ sztIndex ] );
}

//*****************************************************************************
// Make a net list file from a schematic file.
// (Eg. using the following: gnetlist -v -g spice-sdb -o test.ckt test.sch)
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  PrcGNetList::bExec( void )
{
  wxString  osArgLst;
  wxString  os1;
  size_t    szt1;

  // Test file names needed by this function
  if( ! bBinExists( ) )         return( FALSE );
  if( m_oafnSchems.IsEmpty( ) ) return( FALSE );
  if( ! m_ofnNetList.IsOk( ) )
    if( ! bSetNetList( ) )      return( FALSE );

  // Enable GNetList verbose mode
  osArgLst = wxT("-v");

  // Specify the guile procedure name to be used
  osArgLst << wxT(" -g");
  os1 = m_oasGuileProcs.Item( m_sztGuileProc );
  if( ! os1.IsEmpty( ) ) osArgLst << wxT(' ') << os1;

  // Append input and output file names
  osArgLst << wxT(" -o ") << m_ofnNetList.GetFullPath( );
  for( szt1=0; szt1<m_oafnSchems.GetCount( ); szt1++ )
    osArgLst << wxT(" ") << m_oafnSchems.Item( szt1 ).GetFullPath( );
  bSetArgLst( osArgLst.c_str( ) );

  // Set the working directory
  os1 = rofnGetSchem( ).GetPath( );
  wxSetWorkingDirectory( os1 );

  // Execute the process
  if( ! PrcBase::bExec( ) ) return( FALSE );

  // Capture the process output
  if( ! bLogOutput( ) )     return( FALSE );

  return( TRUE );
}

//*****************************************************************************
