/* Gnome Smart Card
 * Copyright (C) 2000  Oliver Feiler (kiza@claws-and-paws.com)
 * 
 * prefs.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "prefs.h"
#include "main.h"
#include "support.h"

#define CONFIG_FILE ".gsmartcardrc" /* filename of the config file */

struct option_t {
  char *option;
  char *value;
  struct option_t *next;
};

struct option_t *first_opt = NULL;

struct option_t *prefs_find_option(char *name);


/* Read the configuration options from the file ~/.gsmartcardrc */

int prefs_read_config(void){

  char *homedir, *config_location;
  char *foo;			/* temp data storage */
  char *option, *value;
  int line = 0;			/* current line we're reading from the config file */
  FILE *config_file;		/* pointer to the config file */
  char buf[1024];		/* holds content of line */
  struct option_t *cur_opt = NULL;

  homedir = getenv("HOME");	/* get the path to the user's home directory
				 and complain if it is not set */
  if ( homedir == NULL ) {
    fprintf(stderr, "HOME not set. Check your environment!\n");
    return -1;
  }

  /* allocate memory for the pathname */
  config_location = malloc(strlen(homedir) + sizeof(char) + strlen(CONFIG_FILE) +1 );

  /* now construct the pathname: homedir + CONFIG_FILE (defined line 26) */
  sprintf(config_location, "%s/%s", homedir, CONFIG_FILE);

  /* open the file for reading */
  config_file = fopen(config_location, "r");
  if ( config_file == NULL ) {
    /* if there is no config file touch one and write default values */
    config_file = fopen(config_location, "w+");
    if ( config_file == NULL ) {
      foo = malloc(strlen(config_location) + strlen("Error opening %s") +1);
      sprintf(foo, "Error opening %s", config_location);
      perror(foo);
      free(foo);
      return -1;
    }
    printf("Creating file %s...\n", config_location);
    fprintf(config_file, "# Configuration file for gsmartcard\n"
	    "# This file is generated automatically. Do not edit!\n\n"
	    "card_auto_read 0\n"
	    "card_auto_save 0\n"
	    "autosave_file set_me\n"
	    "card_auto_save_path none\n"
	    "display ascii\n"
	    "use_dev_rand 0\n"
	    "reader_com_port 1\n");

    /* close the file and reopen it again for reading only */
    fclose(config_file);
    config_file = fopen(config_location, "r");
  }

  /* read line by line and put each line into buf */
  while( fgets(buf, sizeof(buf), config_file) != NULL ) {
    ++line;
    if( (buf[0] == '#') || (buf[0] == '\n') )
      continue;
    buf[strlen(buf)-1] = '\0';

    option = buf;
    value = strchr(buf, ' ');
    if ( value == NULL ) {
      fprintf(stderr, "Error in config file line %d\n"
	      "Missing value for option %s.\n", line, option);
      continue;
    }
    value[0] = '\0';
    ++value;

    if ( strlen(option) == 0 ) {
      fprintf(stderr, "Error in config file line %d\n"
	      "Missing option for value %s.\n", line, value);
      continue;
    }

    /* add each option/value pair to a linked list */

    if ( first_opt == NULL ) {
      first_opt = malloc(sizeof(struct option_t));
      first_opt->option = strdup(option);
      first_opt->value = strdup(value);
      first_opt->next = NULL;
      cur_opt = first_opt;
    } else {
      cur_opt->next = malloc(sizeof(struct option_t));
      cur_opt = cur_opt->next;
      cur_opt->option = strdup(option);
      cur_opt->value = strdup(value);
      cur_opt->next = NULL;
    }
  }

  fclose(config_file);
  free(config_location);
  return 0;
}

/* find a given option in the linked list */

struct option_t *prefs_find_option(char *name) {
  
  struct option_t *cur_opt = first_opt;

  while( cur_opt != NULL ) {

    if( strcmp(cur_opt->option, name) == 0 )
      return cur_opt;

    cur_opt = cur_opt->next;
  }

  return NULL;
}

/* find a value for a given option and return it */

char const *prefs_get_value(char *name) {

  struct option_t *cur_opt;

  cur_opt = prefs_find_option(name);

  if ( cur_opt != NULL )
    return (cur_opt->value);
  else
    return "";

}

/* add a new option to the linked list */

void prefs_new_opt(char *name, char *value) {
 
  struct option_t *cur_opt = first_opt;

  while( cur_opt != NULL ) {
    if ( cur_opt->next == NULL )
      break;
    else
      cur_opt = cur_opt->next;
  }

  if ( first_opt == NULL ) {
    first_opt = malloc(sizeof(struct option_t));
    cur_opt = first_opt;
  } else {
    cur_opt->next = malloc(sizeof(struct option_t));
    cur_opt = cur_opt->next;
  }

  cur_opt->option = strdup(name);
  cur_opt->value = strdup(value);
  cur_opt->next = NULL;
  
}

/* set a value */

void prefs_set_value(char *name, char *value) {

  struct option_t *cur_opt;

  cur_opt = prefs_find_option(name);

  if ( cur_opt != NULL )
    cur_opt->value = strdup(value);
  else
    prefs_new_opt(name, value);
}

/* write the whole config file */

int prefs_write_config(void) {

  char *homedir, *config_location;
  char *foo;			/* temporary foo variable */
  FILE *config_file;		/* config file pointer */
  struct option_t *cur_opt = first_opt;

  /* most of this is copy&paste from prefs_read_config() */

  homedir = getenv("HOME");
  if ( homedir == NULL ) {
    fprintf(stderr, "HOME not set. Check your environment!\n");
    return -1;
  }

  config_location = malloc(strlen(homedir) + sizeof(char) + strlen(CONFIG_FILE) +1 );
  sprintf(config_location, "%s/%s", homedir, CONFIG_FILE);

  config_file = fopen(config_location, "w");
  if ( config_file == NULL ) {
    foo = malloc(strlen(config_location) + strlen("Error opening %s") +1);
    sprintf(foo, "Error opening %s", config_location);
    perror(foo);
    free(foo);
    return -1;
  }

  fprintf(config_file, "# Configuration file for gsmartcard\n"
	  "# This file is generated automatically. Do not edit!\n\n");

  while ( cur_opt != NULL ) {
    fprintf(config_file, "%s %s\n", cur_opt->option, cur_opt->value);
    cur_opt = cur_opt->next;
  }

  fclose(config_file);
  return 0;
}
