/* GSDictDocument.m Subclass of NSGSDictDocument for Ink application

   Copyright (C) 2000 Free Software Foundation, Inc.

   Author:  Fred Kiefer <fredkiefer@gmx.de>
   Date: 2000
   
   This file is part of GNUstep.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <AppKit/AppKit.h>
#include <AppKit/NSWindowController.h>
#include <GSDictParameters.h>
#include <GSDictDocument.h>
#include <GSDictService.h>

NSString *GSDictDidChangeDictParametersHistoryNotification = @"GSDictDidChangedDictParametersHistoryNotification";

NSString *GSDictException = @"GSDictException";
NSString *DictCommandNotAvailable = @"dict command is not available";

static NSMutableArray* dictParametersHistory = nil;

@interface GSDictDocument (Private)

- (NSWindow*) makeWindow;
- (void)updateBackForwardButtonsStatus;

@end

@implementation GSDictDocument

- (id) init
{
  [super init];

  ts = [[NSMutableAttributedString alloc] init];
	if ( dictParametersHistory == nil )
	{
    NSUserDefaults *defaults = nil;
		id storedParameters = nil;
		int index = 0;
		defaults = [NSUserDefaults standardUserDefaults];
		NSAssert( defaults, @"no user defaults found" );
		storedParameters = [defaults arrayForKey:@"dictParametersHistory"];
		for ( index = 0; index < [storedParameters count]; index++ )
		{
			GSDictParameters* params = nil;
			if ( index == 0 )
			{
				dictParametersHistory = [[NSMutableArray alloc] init];
			}
			params = [GSDictParameters parametersWithDictionary:[storedParameters objectAtIndex:index]];
			[dictParametersHistory addObject:params];
		}
		if ( dictParametersHistory == nil )
		{
			dictParametersHistory = [[NSMutableArray alloc] init];
		}
	}
	_lastWord = nil;
	_navigationPos = 0;
	_navigationHistory = [[NSMutableArray alloc] init];
	_logNavigation = YES;
  return self;
}

- (void) dealloc
{
  RELEASE(ts);
  RELEASE(tv);
  RELEASE(pi);
	RELEASE(_navigationHistory);
	RELEASE(_lastWord);
	[super dealloc];
}

- (void)updateDefinitionWithParameters:(GSDictParameters*)dictParams
{
	NSString* word = nil;
  NSData *data = nil;
	NSString* database = [_databasePopup titleOfSelectedItem];
	id provider = [[NSApplication sharedApplication] servicesProvider];

	NSAssert( (dictParams!=nil), @"No GSDictParameters found!" );
	if (( provider != nil ) && ([provider isKindOfClass:[GSDictService class]]))
	{
		NSCursor* busyCursor = nil;
		NSString* fileName = nil;
		NSRange aRange = [tv selectedRange];
		NSRange emptyRange = {0,0};
		word = [[tv string] substringWithRange:aRange];
	  if (word == nil || [word isEqualToString:@""])
		{
			word = [_lastWords stringValue];
		}
	  if (word == nil || [word isEqualToString:@""])
		{
			return;
		}
		busyCursor = [NSCursor IBeamCursor];
		[busyCursor push];
	  [busyCursor setOnMouseEntered:YES];
		[tv addCursorRect:[tv frame] cursor:busyCursor];
//		[[NSGraphicsContext currentContext] flushGraphics];
//		[[tv window] invalidateCursorRectsForView:tv];
//		[[tv window] flushWindow];
		[[NSGraphicsContext currentContext] flush];

		[dictParams setWord:word];
		[dictParams setDatabase:database];
		NS_DURING
			data = [provider queryDictWithParameters:dictParams];
		NS_HANDLER
			[localException raise];
		NS_ENDHANDLER
//		[self loadDataRepresentation:data ofType:@"text"];
		fileName = [NSString stringWithFormat:@"/tmp/%@.rtf", word ];
		[data writeToFile:fileName atomically:NO];
		[self readFromFile:fileName ofType:@"rtf"];
		[self addDictParametersToHistory:dictParams];
		[busyCursor pop];
		[tv addCursorRect:[tv frame] cursor:[NSCursor currentCursor]];
		[tv setSelectedRange:emptyRange];
		[self setFileName:word];
	}
}

- (NSData *)dataRepresentationOfType:(NSString *)aType 
{
  NSAttributedString *attr;

  if (tv != nil)
    attr = [tv textStorage];
  else
    attr = ts;

  if ([aType isEqualToString:@"rtf"])
    {
      return [attr RTFFromRange: NSMakeRange(0, [attr length]) documentAttributes: 
		       [[self printInfo] dictionary]]; 
    }
  else if ([aType isEqualToString:@"text"])
    {
      return [[attr string] dataUsingEncoding: [NSString defaultCStringEncoding]]; 
    }
  else
    {
      NSAssert(NO, ([NSString stringWithFormat: @"Unknown type %@", aType]));
      return nil;
    }
}

- (BOOL)loadDataRepresentation:(NSData *)data ofType:(NSString *)aType 
{
  NSAttributedString *attr;

  if ([aType isEqualToString:@"rtf"])
    {
      attr = [[NSAttributedString alloc] initWithRTF: data 
					 documentAttributes: NULL];
    }
  else if ([aType isEqualToString:@"text"])
    {
      attr = [[NSAttributedString alloc] 
		 initWithString: AUTORELEASE([[NSString alloc] initWithData: data
							       encoding: [NSString defaultCStringEncoding]])];
    }
  else
    {
      NSAssert(NO, ([NSString stringWithFormat: @"Unknown type %@", aType]));
      return NO;
    }

  if (tv != nil)
    [[tv textStorage] setAttributedString: attr];
  else 
    [ts setAttributedString: attr];

  RELEASE(attr);
  return YES;
}

- (void)makeWindowControllers
{
  NSWindowController *controller;
  NSWindow *win = [self makeWindow];
  
  controller = [[NSWindowController alloc] initWithWindow: win];
  [self addWindowController: controller];
  RELEASE(controller);

	[[NSNotificationCenter defaultCenter] 
			addObserver:self
			selector:@selector(dictParametersHistoryChanged:) 
			name:GSDictDidChangeDictParametersHistoryNotification
			object:nil];

  // We have to do this ourself, as there is currently no nib file
  [self windowControllerDidLoadNib: controller];
} 

- (void)printShowingPrintPanel:(BOOL)flag
{
  NSPrintOperation *po = [NSPrintOperation printOperationWithView: tv
					   printInfo: [self printInfo]];

  [po setShowPanels: flag];
  [po runOperation];
}


- (void)windowControllerDidLoadNib:(NSWindowController *) aController;
{
  [super windowControllerDidLoadNib:aController];

  [[tv textStorage] setAttributedString: ts];
  DESTROY(ts);
}

- (void)textDidChange:(NSNotification *)textObject 
{
  [self updateChangeCount: NSChangeDone];
}

- (void)addDictParametersToHistory:(GSDictParameters*)dictParams
{
	NSUserDefaults *defaults = nil;
	ASSIGN( _lastWord, [dictParams word] );
	
	[dictParametersHistory addObject:dictParams];
	if ( [dictParametersHistory count] > 10 )
	{
		[dictParametersHistory removeObjectAtIndex:0];
	}
	defaults = [NSUserDefaults standardUserDefaults];	
	[defaults setObject:dictParametersHistory forKey:@"dictParametersHistory"];
	[defaults synchronize];
	[[NSNotificationCenter defaultCenter]
			postNotificationName:GSDictDidChangeDictParametersHistoryNotification
										object:nil];

	if ( _logNavigation == YES )
	{
		[_navigationHistory addObject:dictParams];
		_navigationPos = [_navigationHistory count]-1;
	}
	[self updateBackForwardButtonsStatus];
}

@end

@implementation GSDictDocument (Private)

- (NSWindow*)makeWindow
{
  NSWindow *window;
  NSText* txt1;
  NSText* txt2;
  NSScrollView* scrollView;
  NSTextView* textView;
  NSColor* backColor;
  NSRect scrollViewRect = {{0, 0}, {600, 380}};
  NSRect winRect = {{100, 100}, {600, 400}};
  NSRect textRect;
	NSButton* definePb;
	NSButton* matchPb;
  unsigned int style = NSTitledWindowMask | NSClosableWindowMask | 
      NSMiniaturizableWindowMask | NSResizableWindowMask;
  
  // This is expected to be retained, as it would normaly come from a nib file,
  // where the owner would retain it.
  window = [[NSWindow alloc] initWithContentRect: winRect
			     styleMask: style
			     backing: NSBackingStoreRetained
			     defer: NO];

  scrollView = [[NSScrollView alloc] initWithFrame: scrollViewRect];
  [scrollView setHasHorizontalScroller: NO];
  [scrollView setHasVerticalScroller: YES]; 
  [scrollView setAutoresizingMask: NSViewHeightSizable | NSViewWidthSizable];
  [[scrollView contentView] setAutoresizingMask: NSViewHeightSizable | NSViewWidthSizable];
  [[scrollView contentView] setAutoresizesSubviews:YES];

  txt1 = [[NSText alloc] initWithFrame:NSMakeRect(0,380,60,15)];
	[txt1 setText:@"Look For:"];
	[txt1 setEditable:NO];
	[txt1 setSelectable:NO];
	[txt1 setAlignment:NSCenterTextAlignment];
	[txt1 setBackgroundColor:[NSColor lightGrayColor]];
	[txt1 setVerticallyResizable:YES];
	[txt1 sizeToFit];
  [txt1 setAutoresizingMask: NSViewMinYMargin];
  [[window contentView] addSubview:txt1];

  _databasePopup = [[NSPopUpButton alloc] initWithFrame:NSMakeRect(220,380,80,20)];
	{
		id databases = [[GSDictParameters defaultParameters] databases];
		int databasesCount = [databases count];
		int databaseIndex = 0;
		NSAssert( databases, @"No databases found" );
		for ( databaseIndex = 0; databaseIndex < databasesCount; databaseIndex++ )
		{
			[_databasePopup addItemWithTitle:[databases objectAtIndex:databaseIndex]];
		}
	}
	[_databasePopup selectItemWithTitle:[[GSDictParameters defaultParameters] database]];
	[_databasePopup setTarget:self];
  [_databasePopup setAction:@selector(buttonSwitchDatabase:)];
  [_databasePopup setAutoresizingMask: NSViewMinYMargin];
  [[window contentView] addSubview:_databasePopup];

  _lastWords = [[NSComboBox alloc] initWithFrame:NSMakeRect(60,380,120,20)];
  [_lastWords setTarget:self];
  [_lastWords setDelegate:self];
  [_lastWords setAction:@selector(comboSwitchWord:)];
  [_lastWords setAutoresizingMask: NSViewMinYMargin];
	[_lastWords addItemsWithObjectValues:dictParametersHistory];
  [[NSNotificationCenter defaultCenter] 
    addObserver:self 
    selector: @selector(comboBoxSelectionDidChange:)
    name:NSComboBoxSelectionDidChangeNotification
    object: (id)_lastWords];
  [[window contentView] addSubview:_lastWords];

  txt2 = [[NSText alloc] initWithFrame:NSMakeRect(185,380,30,20)];
	[txt2 setText:@"Into:"];
	[txt2 setEditable:NO];
	[txt2 setSelectable:NO];
	[txt2 setAlignment:NSCenterTextAlignment];
	[txt2 setBackgroundColor:[NSColor lightGrayColor]];
	[txt2 setVerticallyResizable:YES];
	[txt2 sizeToFit];
  [txt2 setAutoresizingMask: NSViewMinYMargin];
  [[window contentView] addSubview:txt2];

  _back = [[NSButton alloc] initWithFrame:NSMakeRect(310,380,60,20)];
	[_back setTitle:@"Back"];
  [_back setTarget:self];
  [_back setAction:@selector(backButton:)];
  [_back setAutoresizingMask: NSViewMinYMargin];
	[_back setEnabled:NO];
  [[window contentView] addSubview:_back];

  _forward = [[NSButton alloc] initWithFrame:NSMakeRect(380,380,60,20)];
	[_forward setTitle:@"Forward"];
  [_forward setTarget:self];
  [_forward setAction:@selector(forwardButton:)];
  [_forward setAutoresizingMask: NSViewMinYMargin];
	[_forward setEnabled:NO];
  [[window contentView] addSubview:_forward];

  definePb = [[NSButton alloc] initWithFrame:NSMakeRect(450,380,70,20)];
	[definePb setTitle:@"Definition"];
  [definePb setTarget:self];
  [definePb setAction:@selector(defineButton:)];
  [definePb setAutoresizingMask: NSViewMinYMargin];
  [[window contentView] addSubview:definePb];

  matchPb = [[NSButton alloc] initWithFrame:NSMakeRect(530,380,60,20)];
	[matchPb setTitle:@"Match"];
  [matchPb setTarget:self];
  [matchPb setAction:@selector(matchButton:)];
  [matchPb setAutoresizingMask: NSViewMinYMargin];
  [[window contentView] addSubview:matchPb];

	// Build up the text network
  textRect = [[scrollView contentView] frame];
  textView = [[NSTextView alloc] initWithFrame: textRect];
  // off white
  backColor = [NSColor colorWithCalibratedWhite: 0.85 alpha: 1.0];
  [textView setBackgroundColor: backColor];
  [textView setMinSize: NSMakeSize (0, 0)];
  [textView setMaxSize:NSMakeSize(1e7, 1e7)];
  [textView setRichText: YES];
	[textView setSelectable:YES];
	[textView setEditable:NO];
//  [textView setUsesFontPanel: YES];
  [textView setDelegate: self];
  [textView setVerticallyResizable: YES];
  [textView setMinSize: NSMakeSize(0,0)];
  [textView setMaxSize: NSMakeSize(1E7,1E7)];
  [textView setAutoresizingMask: NSViewHeightSizable | NSViewWidthSizable];
  // Store the text view in an ivar
  ASSIGN(tv, textView);

	[scrollView setDocumentView: textView];
  [[textView textContainer] setContainerSize:NSMakeSize([scrollView contentSize].width,1e7)];

  RELEASE(textView);


  [[window contentView] addSubview:scrollView];
//  [window setContentView: scrollView];
  RELEASE(scrollView);
  // Make the GSDictDocument the delegate of the window
  [window setDelegate: self];
  
  [window display];
  [window orderFront: nil];

  return window;
}

- (void)comboSwitchWord:(id)sender
{
	GSDictParameters* dictParams = nil;
	if ( ([sender indexOfSelectedItem] >= 0) &&
			( [sender indexOfSelectedItem] < [dictParametersHistory count] ) )
	{
		dictParams = [dictParametersHistory objectAtIndex:[sender indexOfSelectedItem]];
	}
	else
	{
		dictParams = [GSDictParameters defaultParameters];
		[dictParams setWord:[sender stringValue]];
	}
	NSAssert( dictParams, @"no dictParams found" );
	[self updateDefinitionWithParameters:dictParams];
}

- (void)comboBoxSelectionDidChange:(id)sender
{
}

- (void)buttonSwitchDatabase:(id)sender
{
	GSDictParameters* dictParams = [GSDictParameters defaultParameters];
	[dictParams setDatabase:[sender titleOfSelectedItem]];
	[self updateDefinitionWithParameters:dictParams];
}

- (void)dictParametersHistoryChanged:(id)sender
{
	[_lastWords removeAllItems];
	[_lastWords addItemsWithObjectValues:dictParametersHistory];
  [_lastWords setNeedsDisplay: YES];
	[_lastWords setStringValue:_lastWord];
}

- (void)updateBackForwardButtonsStatus
{
	if ( _navigationPos == 0)
	{
		[_back setEnabled:NO];
	}
	else
	{
		[_back setEnabled:YES];
	}
	if ( _navigationPos < [_navigationHistory count]-1 )
	{
		[_forward setEnabled:YES];
	}
	else
	{
		[_forward setEnabled:NO];
	}
}

- (void)backButton:(id)sender
{
	GSDictParameters* dictParams = [_navigationHistory objectAtIndex:--_navigationPos];
	NSAssert( (_navigationPos>=0), @"" );
	[_lastWords setStringValue:[dictParams word]];
	_logNavigation = NO;
	[self updateDefinitionWithParameters:dictParams];
	_logNavigation = YES;
	[self updateBackForwardButtonsStatus];
}

- (void)forwardButton:(id)sender
{
	GSDictParameters* dictParams = [_navigationHistory objectAtIndex:++_navigationPos];
	NSAssert( (_navigationPos <= ([_navigationHistory count]-1) ), @"" );
	[_lastWords setStringValue:[dictParams word]];
	_logNavigation = NO;
	[self updateDefinitionWithParameters:dictParams];
	_logNavigation = YES;
	[self updateBackForwardButtonsStatus];
}

- (void)defineButton:(id)sender
{
	GSDictParameters* dictParams = [GSDictParameters defaultParameters];
	[self updateDefinitionWithParameters:dictParams];
}

- (void)matchButton:(id)sender
{
	GSDictParameters* dictParams = [GSDictParameters defaultParameters];
	[dictParams setMatch:YES];
	[self updateDefinitionWithParameters:dictParams];
}

@end
