#ifndef _SocketComm_H_
#define _SocketComm_H_

#include "msgface.h"

typedef int(* SocketReadFunc)(void *, const char *, unsigned int);
typedef void(* SocketErrorFunc)(void *, const char *);

class SocketComm {
public:
	SocketComm();

	// server constructor
	SocketComm(
		SOCKET server_sock,
		unsigned short port,
		SocketReadFunc read_func,
		SocketErrorFunc error_func = 0 );

	// client constructor
	SocketComm(
		const char *host,
		unsigned short port,
		SocketReadFunc read_func,
		SocketErrorFunc error_func = 0 );

	~SocketComm();

	void setClient();
	void setServerSock( SOCKET sock );

	SOCKET getServerSock();
	inline bool isServer() { return is_server; }
	void setHost( const char *host );
	void setPort( unsigned short port );
	unsigned short getPort();
	void setMaxDocLen( unsigned int len );
	unsigned int getMaxDocLen();
	void setReadCallback( SocketReadFunc read_func );
	SocketReadFunc getReadCallback();
	void setReadCallbackArg( void *arg );
	void *getReadCallbackArg();
	// must cast void * arg to char *, and do delete [] on it
	void setErrorCallback( SocketErrorFunc error_func );
	void setErrorCallbackArg( void *arg );

	int initSock();
	void endSock();
	bool isSockInit();

	void lock();
	void unlock();

	// must NOT be called by input_thread
	int send( const char *str, unsigned int len );

	void setError( const char *msg );
	const char *getError();

	// used by input_thread_main()
	// set by recv. thread; true if running, false otherwise
	bool is_threading;
	bool is_threading_picked;
	// before thread is created, set to false.  When you want
	// the thread to die, set it to true, and thread will
	// read it and die
	bool thread_die;

private:
	void constr_init();

	int startThread();
	void endThread();
	int createThread();
	void cleanupThread();

	char *my_strdup( const char *str );
	void msleep( unsigned int msec );

	bool is_server;

	SOCKET sock;
	SOCKET server_sock;
	bool is_sock_init;

	char *error_msg;
	bool error_set;

	unsigned short port;
	char *host;
	unsigned int max_doclen;

	SocketReadFunc read_func;
	SocketErrorFunc error_func;
	void *read_func_arg;
	void *error_func_arg;

#ifdef GRUB_UNIX
#else
	HANDLE hThread;
#endif
	ThreadId thread_id;
	MutexType mutex;
};

#endif  /* _SocketComm_H_ */

/*
Two assumptions:

  1) You do select() for reading on srv_sock to see if accept()
  has a client ready to connect.

    - Test this in guiface_srv.cpp code and see if it works

  2) If one thread does a select() on a socket for reading and
  another one does closesocket() on it, the first thread will
  immediately exit with error code

    - Even if the call to select() isn't interrupted, it's OK
	since if does thread does not wait too long

*/
