// mandel.C : new demo for the Mandelbrot set, with parallelizing option
// wolf 1/97 - 3/00

// ************* header part (to include in both sources) **************
struct Complex {
  double x,y;
  Complex() {}
  Complex(double x, double y) : x(x),y(y) {}
};

inline double real(Complex c) { return c.x; }
inline double imag(Complex c) { return c.y; }

#define PARALLEL

// ************** Start of parallelized part **************
inline int iterate(const Complex& c, int itmax) {
  static int limit = 100; // quite in-sensitive value 
  int it = 0;
  // Complex z = 0;
  // do { z = z*z + c; if (++it == itmax) break; } while (norm(z) < limit); return it;  
  // the real version is 3 x faster !
  double cx = real(c), cy = imag(c); 
  double x = cx, y = cy;
  for (; ++it < itmax;) { 
    double x2 = x*x, y2=y*y; 
    if (x2+y2 > limit) break;
    y = 2*x*y + cy; x = x2 - y2 + cx; 
  } 
  return it;
}

#ifdef PARALLEL
// compute iteration number for vector of np Complex numbers 'cv' 
// and store into vector 'it'
void iterate(int np, Complex* cv, int* it, int itmax) {
  for (int i=0; i<np; i++) it[i] = iterate(cv[i],itmax);
}
#endif
// ************** End of parallelized part **************

#include "window.h"
#undef Complex

static int itmax = 512;
static char *itmenue = "itmax";

class mandel_main;
mandel_main *newman(double x1, double y1, double x2, double y2, Bool top=False);

class mandel_win : public coord_window {
  info_window *iw;
  int xoff,yoff;
public:
  int rectmode; // bool : rectangle mode , else shift mode
  mandel_win(window &parent, int nx, int ny, info_window *iw) :
  coord_window(parent,nx,ny,0,20,0,0,0,0), iw(iw) {
    selection_mask |= ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
    xoff = yoff = 0;
    rectmode = 0;
  }
  Complex c_org(int x, int y) { // map window coords -> real coords
    Complex z(x_org(x+xoff),y_org(y+yoff));
    return z;
  }
  Complex c_org(XPoint p) { return c_org(p.x,p.y); }

  void shift(int nx, int ny) { // shift nx pix -> right (nx > 0) or left <- (nx < 0)
    // and ny down (>0) or up <0
    xoff -= nx; yoff -= ny;  
    int xs = (nx > 0) ? 0 : -nx; // source x
    int ys = (ny > 0) ? 0 : -ny; // source y
    XCopyArea(display, pix, pix, gc_copy, xs, ys,width-nx,height-ny,xs+nx,ys+ny);
    if (nx > 0) { // shift nx -> right
      for (int x=0; x<nx; x++) vline(x,0,height); 
    } else if (nx < 0) { // shift nx <- left
      nx = -nx;
      for (int x=0; x<nx; x++) vline(width-1-x,0,height); 
    }
    if (ny > 0) { // 
      for (int y=0; y<ny; y++) hline(0,width,y); 
    } else if (ny < 0) { 
      ny = -ny;
      for (int y=0; y<ny; y++) hline(0,width,height-1-y); 
    }
    set_color(black);
    XCopyArea(display,pix, Win, gc_copy, 0, 0, width, height, 0, 0);
    XMapWindow(display, Win); 
  }
  XPoint p1,p2;
  void BPress_CB(XButtonEvent ev) {
    p1.x = ev.x; p1.y = ev.y; p2 = p1;
  }
  void set_coords(double x0, double y0, double dx, double dy) {
    define_coord(x0,y0,x0+dx,y0+dy);
    xoff = yoff = 0;
  }
  Complex get_coords(double& dx, double& dy) {
    dx = xr - xl; dy = yu - yd;
    return c_org(0,height);
  }
  void zoom(Complex z, double fact) { // fact < 1 zoom, fact > 1 unzoom
    double x = real(z), y = imag(z);
    float scx = xr - xl, scy = yu - yd;
    scx *= fact; scy *= fact;
    // set coords so that x,y is centered
    set_coords(x-scx/2, y-scy/2, scx, scy);
    redraw(); 
    info(x,y);
  }

  // Button 1 : make new window for selected Rectangle or shift
  // Button 2 : zoom by factor 3
  // Button 3 : unzoom
  void BRelease_CB(XButtonEvent ev) {  
    Complex z = c_org(ev.x,ev.y);
    if (ev.state & (Button2Mask | Button3Mask)) {
      float fact = (ev.state & Button2Mask) ? 1/3. : 3.;
      zoom(z,fact);
    } else { // button 1
      if (rectmode && fabs(p2.x-p1.x) > 10 && fabs(p2.y-p1.y) > 10) {
	Complex z1 = c_org(p1), z2 = c_org(p2);
	newman(real(z1),real(z2),imag(z2),imag(z1));
      }
    }
  }
  void info(double x, double y) { 
    Complex z = c_org(x,y);
    sprintf(iw->info,"z = %.8f + %.8fi dx=%g it=%4d itmax=%d %5d",
	    real(z),imag(z),xr-xl,iterate(z,itmax),itmax,xoff);
    iw->redraw();
  }
  void Motion_CB(XMotionEvent ev) { 
    if (ev.state & Button1Mask) { 
      if (rectmode) {
	// clear old rect
	if (ev.x > p1.x && ev.y > p1.y) { 
	  XDrawRectangle(display,Win,gc_rubber,p1.x-1, p1.y-1,p2.x-p1.x+2,p2.y-p1.y+2);
	  p2.x = ev.x; p2.y = ev.y;
	  XDrawRectangle(display,Win,gc_rubber,p1.x-1, p1.y-1,p2.x-p1.x+2,p2.y-p1.y+2);
	}
      } else {
	shift(ev.x-p2.x,ev.y-p2.y); p2.x = ev.x; p2.y = ev.y; 
      }
    }
    else info(ev.x,ev.y);
  }
  void setpix(int x, int y, int it) {
    static int cmax = 0xffff;
    int col = cmax - (it*cmax)/itmax;
    set_color(col);
    DrawPoint(x,y);
  }  
  void setpix(int x, int y) {
    int it = iterate(c_org(x,y),itmax);
    setpix(x,y,it);
  } 
  void hline(int xl, int xr, int y) {
#ifdef PARALLEL
    int np = xr-xl+1;
    Complex z[np]; int it[np];
    for (int x = xl, i=0; x <= xr; x++,i++) z[i] = c_org(x,y);
    iterate(np,z,it,itmax);  
    for (int x = xl, i=0; x <= xr; x++,i++) setpix(x,y,it[i]);
#else
    for (int x = xl; x <= xr; x++) setpix(x,y);
#endif
    XCopyArea(display,pix, Win, gc_copy, xl, y, xr-xl+1, 1, xl, y);
  } 
  void vline(int x, int yb, int yt) {
#ifdef PARALLEL
    int np = yt-yb+1;
    Complex z[np]; int it[np];
    for (int y = yb, i=0; y <= yt; y++,i++) z[i] = c_org(x,y);
    iterate(np,z,it,itmax);  
    for (int y = yb, i=0; y <= yt; y++,i++) setpix(x,y,it[i]);
#else
    for (int y = yb; y <= yt; y++) setpix(x,y);
#endif
    XCopyArea(display,pix, Win, gc_copy, x, yb, 1, yt-yb+1, x, yb);
  }
  void draw_interior() {
    int nx = width, ny = height;
    int xm = nx/2, ym = ny/2, xl = xm, xr = xm, yt = ym, yb = ym;
    while (yt <= height) {
      while (xr*ym <= xm*yt) { // left & right lines
	vline(xl,yb,yt); vline(xr,yb,yt);
	xr++; xl--;
      }
      yb--; yt++; // top & bottom lines
      hline(xl,xr,yb); hline(xl,xr,yt);
      // to show the progress :
      XMapWindow(display, Win); 
      
    }
    set_color(black);
  }
};

class mandel_main : public main_window {
  mandel_win *cw;
public:
  menu_bar *mb;
  mandel_main(char *str, int nx, int ny, double x1, double x2, double y1, 
	    double y2, bool top, int xp, int yp) : main_window(str,nx,ny,1,xp,yp) {
    info_window *iw = new info_window(*this,nx,20,0,ny-20);
    cw = new mandel_win(*this,nx,ny-40,iw);
    cw->define_coord(x1,y1,x2,y2);
    mb = new menu_bar(*this,nx,20,0,0,60,80);
    window *ww = (top) ? new quit_button(*mb) : new unmap_button(*mb,"close",this);
    ww->add_help("Button 1 : Verschieben","Button 2 : Zoom","Button 3 : Unzoom",
		 "rect : rectangle mode (button 1)",0);
    char *it_list[] = {"16","32","64","128","256","512","1024","2048",0};
    make_radio_menu(*mb,itmenue,it_list,this);
    //  char *li_list[] = {"10","20","40","80","150","400","1000","2000","4000",0};
    // make_radio_menu(*mb,limenue,li_list,this);
	
    if (top) new toggle_button(*mb,"run",&polling_mode);
    new toggle_button(*mb,"rect",&cw->rectmode);
    new instance_button<mandel_main>(*mb,"save",&mandel_main::save,this);
    FILE *fl = fopen("mandel.lst","r");
    if (fl) {
      int nl = 0;
      static char *llist[100];
      char s[100];
      while (fgets(s,100,fl)) {
	s[strlen(s)-1] = 0; // NL
	llist[nl++] = strdup(s);
      }
      llist[nl] = 0;
      char *llmenue = "list";
      make_radio_menu(*mb,llmenue,llist,this);
    }
    RealizeChildren();
  } 
  void save() {
    double dx,dy;
    Complex z0 = cw->get_coords(dx,dy);
    printf("%.10g %.10g %g %g %d\n",real(z0),imag(z0),dx,dy,itmax);
  }
  virtual void polling_handler() { 
    Complex z(-0.81079279,0.20699848);
    // itmax = 2048;
    cw->zoom(z,0.5);
    // cw->shift(5,1);
  }  
  virtual void action(char* menu, char *value) {
    if (menu == itmenue) itmax = atoi(value); 
    else {
      double x0,y0,dx,dy; int it;
      int ns = sscanf(value,"%lg%lg%lg%lg%d",&x0,&y0,&dx,&dy,&it);
      if (ns >= 4) cw->set_coords(x0,y0,dx,dy);
      if (ns >= 5) itmax = it;
    }
    cw->redraw();
  } 

};

mandel_main * newman(double x1, double x2, double y1, double y2, Bool top) {
  int nx = 400, ny = 360; 
  char str[200]; sprintf(str,"x [%g+%g] y [%g+%g]",x1,x2-x1,y1,y2-y1);
  static int xp = 100, yp = 100; // window pos
  mandel_main *mw = new mandel_main(str,nx,ny,x1,x2,y1,y2,top,xp,yp);
  xp += 50; yp += 20;
  return mw;
}

int main() {
  mandel_main *mw = newman(-2,0.5,-1,1,True);        
  mw->main_loop(); 
}










