/*
 * GQview
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "collect-table.h"

#include "collect-dlg.h"
#include "collect-io.h"
#include "dnd.h"
#include "dupe.h"
#include "filelist.h"
#include "img-main.h"
#include "img-view.h"
#include "menu.h"
#include "utilops.h"
#include "ui_clist_edit.h"
#include "ui_fileops.h"
#include "ui_menu.h"

#include "icons/marker.xpm"
#define MARKER_WIDTH 26
#define MARKER_HEIGHT 32

#include <gdk/gdkkeysyms.h> /* for keyboard values */

#define COLLECT_TABLE_MAX_COLUMNS 32
#define COLLECT_TABLE_TEXT_COLUMNS 33	/* this is ridiculous */


/*
 *-------------------------------------------------------------------
 * more misc
 *-------------------------------------------------------------------
 */

static gint collection_table_find_position(CollectTable *ct, CollectInfo *info, gint *row, gint *col)
{
	gint n;

	n = g_list_index(ct->cd->list, info);

	if (n < 0) return FALSE;

	*row = n / ct->columns;
	*col = n - (*row * ct->columns);

	return TRUE;
}

static CollectInfo *collection_table_find_info(CollectTable *ct, gint row, gint col)
{
	GList *list;

	list = gtk_clist_get_row_data(GTK_CLIST(ct->clist), row);

	if (!list) return NULL;

	return (CollectInfo *)g_list_nth_data(list, col);
}

static void collection_table_update_status(CollectTable *ct)
{
	if (ct->status_label)
		{
		gchar *buf;

		if (!ct->cd->list)
			{
			buf = g_strdup(_("Empty"));
			}
		else if (ct->selection)
			{
			buf = g_strdup_printf(_("%d images (%d)"), g_list_length(ct->cd->list), g_list_length(ct->selection));
			}
		else
			{
			buf = g_strdup_printf(_("%d images"), g_list_length(ct->cd->list));
			}

		gtk_label_set_text(GTK_LABEL(ct->status_label), buf);
		g_free(buf);
		}
}

static void collection_table_update_extras(CollectTable *ct, gint loading, gfloat value)
{
	if (ct->extra_label)
		{
		gchar *text;
		if (loading)
			text = _("Loading thumbs...");
		else
			text = "";

		gtk_progress_set_percentage(GTK_PROGRESS(ct->extra_label), value);
		gtk_progress_set_format_string(GTK_PROGRESS(ct->extra_label), text);
		}
}

/*
 *-------------------------------------------------------------------
 * cell updates
 *-------------------------------------------------------------------
 */

static void collection_table_update_cell(CollectTable *ct, gint row, gint col, gint selected)
{
	if (selected || (row == ct->focus_row && col == ct->focus_column))
		{
		GtkStyle *style;

		style = gtk_style_copy(GTK_WIDGET(ct->clist)->style);

		if (selected)
			{
			style->base[GTK_STATE_NORMAL] = style->bg[GTK_STATE_SELECTED];
			style->fg[GTK_STATE_NORMAL] = style->fg[GTK_STATE_SELECTED];
			}
		else
			{
			style->base[GTK_STATE_NORMAL] = style->bg[GTK_STATE_ACTIVE];
			style->fg[GTK_STATE_NORMAL] = style->fg[GTK_STATE_ACTIVE];
			}

		if (row == ct->focus_row && col == ct->focus_column)
			{
			clist_edit_shift_color(style);
			}

		gtk_clist_set_cell_style (GTK_CLIST(ct->clist), row, col, style);
		gtk_style_unref(style);
		}
        else
		{
		gtk_clist_set_cell_style (GTK_CLIST(ct->clist), row, col, NULL);
		}
}

static void collection_table_update_cell_by_info(CollectTable *ct, CollectInfo *info, gint selected)
{
	gint r, c;

	if (!collection_table_find_position(ct, info, &r, &c)) return;

	collection_table_update_cell(ct, r, c, selected);
}

/*
 *-------------------------------------------------------------------
 * selections
 *-------------------------------------------------------------------
 */

static void collection_table_verify_selections(CollectTable *ct)
{
	GList *work;

	work = ct->selection;
	while(work)
		{
		CollectInfo *info = work->data;
		work = work->next;
		if (!g_list_find(ct->cd->list, info))
			{
			ct->selection = g_list_remove(ct->selection, info);
			}
		}
}

static gint collection_table_is_selected(CollectTable *ct, CollectInfo *info)
{
	return (g_list_find(ct->selection, info) != NULL);
}

void collection_table_select_all(CollectTable *ct)
{
	GList *work;

	g_list_free(ct->selection);
	ct->selection = NULL;

	work = ct->cd->list;
	while(work)
		{
		ct->selection = g_list_append(ct->selection, work->data);
		collection_table_update_cell_by_info(ct, work->data, TRUE);
		work = work->next;
		}

	collection_table_update_status(ct);
}

void collection_table_unselect_all(CollectTable *ct)
{
	GList *work;

	work = ct->selection;
	while(work)
		{
		collection_table_update_cell_by_info(ct, work->data, FALSE);
		work = work->next;
		}

	g_list_free(ct->selection);
	ct->selection = NULL;

	collection_table_update_status(ct);
}

static void collection_table_select(CollectTable *ct, CollectInfo *info)
{
	ct->prev_selection = info;

	if (!info || collection_table_is_selected(ct, info)) return;

	ct->selection = g_list_append(ct->selection, info);

	collection_table_update_cell_by_info(ct, info, TRUE);

	collection_table_update_status(ct);
}

static void collection_table_unselect(CollectTable *ct, CollectInfo *info)
{
	ct->prev_selection = info;

	if (!info || !collection_table_is_selected(ct, info)) return;

	ct->selection = g_list_remove(ct->selection, info);

	collection_table_update_cell_by_info(ct, info, FALSE);

	collection_table_update_status(ct);
}

static void collection_table_select_util(CollectTable *ct, CollectInfo *info, gint select)
{
	if (select)
		collection_table_select(ct, info);
	else
		collection_table_unselect(ct, info);
}

static void collection_table_select_region_util(CollectTable *ct, CollectInfo *start, CollectInfo *end, gint select)
{
	gint row1, col1;
	gint row2, col2;
	gint t;
	gint i, j;

	if (!collection_table_find_position(ct, start, &row1, &col1) ||
	    !collection_table_find_position(ct, end, &row2, &col2) ) return;

	if (!collection_rectangular_selection)
		{
		GList *work;
		CollectInfo *info;

		if (g_list_index(ct->cd->list, start) > g_list_index(ct->cd->list, end))
			{
			info = start;
			start = end;
			end = info;
			}

		work = g_list_find(ct->cd->list, start);
		while(work)
			{
			info = work->data;
			collection_table_select_util(ct, info, select);
			
			if (work->data != end)
				work = work->next;
			else
				work = NULL;
			}
		return;
		}

	if (row2 < row1)
		{
		t = row1;
		row1 = row2;
		row2 = t;
		}
	if (col2 < col1)
		{
		t = col1;
		col1 = col2;
		col2 = t;
		}

	if (debug) printf("table: %d x %d to %d x %d\n", row1, col1, row2, col2);

	for (i = row1; i <= row2; i++)
		{
		for (j = col1; j <= col2; j++)
			{
			CollectInfo *info = collection_table_find_info(ct, i, j);
			if (info) collection_table_select_util(ct, info, select);
			}
		}

	ct->prev_selection = end;
}

GList *collection_table_selection_get_list(CollectTable *ct)
{
	GList *list = NULL;
	GList *work;

	work = ct->selection;
	while(work)
		{
		CollectInfo *info = work->data;

		list = g_list_prepend(list, g_strdup(info->path));

		work = work->next;
		}

	list = g_list_reverse(list);

	return list;
}

/*
 *-------------------------------------------------------------------
 * tooltip type window
 *-------------------------------------------------------------------
 */

static void tip_show(CollectTable *ct)
{
	GtkWidget *label;
	gchar *buf;
	gint x, y;
	gint row, col;

	if (ct->tip_window) return;

	gdk_window_get_pointer(GTK_CLIST(ct->clist)->clist_window, &x, &y, NULL);
	gtk_clist_get_selection_info(GTK_CLIST(ct->clist), x, y, &row, &col);

	ct->tip_info = collection_table_find_info(ct, row, col);
	if (!ct->tip_info) return;

	ct->tip_window = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_window_set_policy(GTK_WINDOW(ct->tip_window), FALSE, FALSE, TRUE);
	gtk_container_border_width(GTK_CONTAINER(ct->tip_window), 2);

	buf = g_strdup_printf("%s", filename_from_path(ct->tip_info->path));
	label = gtk_label_new(buf);
	g_free(buf);

	gtk_object_set_data(GTK_OBJECT(ct->tip_window), "tip_label", label);
	gtk_container_add(GTK_CONTAINER(ct->tip_window), label);
	gtk_widget_show(label);

	gdk_window_get_pointer(NULL, &x, &y, NULL);

	gtk_widget_popup(ct->tip_window, x + 16, y + 16);
}

static void tip_hide(CollectTable *ct)
{
	if (ct->tip_window) gtk_widget_destroy(ct->tip_window);
	ct->tip_window = NULL;
}

static gint tip_schedule_cb(gpointer data)
{
	CollectTable *ct = data;

	if (ct->tip_delay_id == -1) return FALSE;

	tip_show(ct);

	ct->tip_delay_id = -1;
	return FALSE;
}

static void tip_schedule(CollectTable *ct)
{
	tip_hide(ct);

	if (ct->tip_delay_id != -1) gtk_timeout_remove(ct->tip_delay_id);

	ct->tip_delay_id = gtk_timeout_add(1000, tip_schedule_cb, ct);
}

static void tip_unschedule(CollectTable *ct)
{
	tip_hide(ct);

	if (ct->tip_delay_id != -1) gtk_timeout_remove(ct->tip_delay_id);
	ct->tip_delay_id = -1;
}

static void tip_update(CollectTable *ct, CollectInfo *info)
{
	if (ct->tip_window)
		{
		gint x, y;

		gdk_window_get_pointer(NULL, &x, &y, NULL);
		gdk_window_move(ct->tip_window->window, x + 16, y + 16);

		if (info != ct->tip_info)
			{
			GtkWidget *label;
			gchar *buf;

			ct->tip_info = info;

			if (!ct->tip_info)
				{
				tip_hide(ct);
				tip_schedule(ct);
				return;
				}

			label = gtk_object_get_data(GTK_OBJECT(ct->tip_window), "tip_label");
			buf = g_strdup_printf("%s", filename_from_path(ct->tip_info->path));
			gtk_label_set_text(GTK_LABEL(label), buf);
			g_free(buf);
			}
		}
	else
		{
		tip_schedule(ct);
		}
}

/*
 *-------------------------------------------------------------------
 * popup menus
 *-------------------------------------------------------------------
 */

static void collection_table_popup_save_as_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	collection_dialog_save_as(NULL, ct->cd);
}

static void collection_table_popup_save_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	if (!ct->cd->path)
		{
		collection_table_popup_save_as_cb(widget, data);
		return;
		}

	if (!collection_save(ct->cd, ct->cd->path))
		{
		printf("failed saving to collection path: %s\n", ct->cd->path);
		}
}

static void collection_table_popup_edit_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct;
	gint n;
	GList *list;

	ct = submenu_item_get_data(widget);

	if (!ct) return;
	n = GPOINTER_TO_INT(data);

	list = collection_table_selection_get_list(ct);
	if (list)
		{
		start_editor_from_path_list(n, list);

		path_list_free(list);
		}
}

static void collection_table_popup_copy_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	file_util_copy(NULL, collection_table_selection_get_list(ct), NULL);
}

static void collection_table_popup_move_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	file_util_move(NULL, collection_table_selection_get_list(ct), NULL);
}

static void collection_table_popup_rename_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	file_util_rename(NULL, collection_table_selection_get_list(ct));
}

static void collection_table_popup_delete_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	file_util_delete(NULL, collection_table_selection_get_list(ct));
}

static void collection_table_popup_sort_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct;
	SortType type;

	ct = submenu_item_get_data(widget);

	if (!ct) return;

	type = (SortType)GPOINTER_TO_INT(data);

	collection_set_sort_method(ct->cd, type);
}

static void collection_table_popup_view_new_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	if (ct->mouse_current && g_list_find(ct->cd->list, ct->mouse_current))
		{
		view_window_new_from_collection(ct->cd, ct->mouse_current);
		}
}

static void collection_table_popup_view_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	if (ct->mouse_current && g_list_find(ct->cd->list, ct->mouse_current))
		{
		main_image_change_to_from_collection(ct->cd, ct->mouse_current);
		}
}

static void collection_table_popup_selectall_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	collection_table_select_all(ct);
	ct->prev_selection= ct->mouse_current;
}

static void collection_table_popup_unselectall_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	collection_table_unselect_all(ct);
	ct->prev_selection= ct->mouse_current;
}

static void collection_table_popup_remove_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;
	GList *list;

	list = g_list_copy(ct->selection);
	collection_remove_by_info_list(ct->cd, list);
	g_list_free(list);
}

static void collection_table_popup_add_filelist_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;
	GList *list;

	list = file_get_complete_list();

	if (list)
		{
		collection_table_add_path_list(ct, list);
		path_list_free(list);
		}
}

static void collection_table_popup_add_collection_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	collection_dialog_append(NULL, ct->cd);
}

static void collection_table_popup_find_dupes_cb(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;
	DupeWindow *dw;

	dw = dupe_window_new(DUPE_MATCH_NAME);
	dupe_window_add_collection(dw, ct->cd);
}

static GtkWidget *collection_table_popup_menu(CollectTable *ct, gint over_icon)
{
	GtkWidget *menu;

	menu = popup_menu_short_lived();

	if (over_icon)
		{
		menu_item_add(menu, _("View"), collection_table_popup_view_cb, ct);
		menu_item_add(menu, _("View in new window"), collection_table_popup_view_new_cb, ct);
		menu_item_add_divider(menu);
		menu_item_add(menu, _("Remove"), collection_table_popup_remove_cb, ct);
		}
	menu_item_add(menu, _("Append from file list"), collection_table_popup_add_filelist_cb, ct);
	menu_item_add(menu, _("Append from collection..."), collection_table_popup_add_collection_cb, ct);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Select all"), collection_table_popup_selectall_cb, ct);
	menu_item_add(menu, _("Select none"), collection_table_popup_unselectall_cb, ct);
	menu_item_add_divider(menu);
	if (over_icon)
		{
		submenu_add_edit(menu, NULL, collection_table_popup_edit_cb, ct);
		menu_item_add(menu, _("Copy..."), collection_table_popup_copy_cb, ct);
		menu_item_add(menu, _("Move..."), collection_table_popup_move_cb, ct);
		menu_item_add(menu, _("Rename..."), collection_table_popup_rename_cb, ct);
		menu_item_add(menu, _("Delete..."), collection_table_popup_delete_cb, ct);
		menu_item_add_divider(menu);
		}
	submenu_add_sort(menu, collection_table_popup_sort_cb, ct, FALSE, TRUE);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Save collection"), collection_table_popup_save_cb, ct);
	menu_item_add(menu, _("Save collection as..."), collection_table_popup_save_as_cb, ct);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Find duplicates..."), collection_table_popup_find_dupes_cb, ct);

	return menu;
}
/*
 *-------------------------------------------------------------------
 * keyboard callbacks
 *-------------------------------------------------------------------
 */

static void collection_table_move_focus(CollectTable *ct, gint row, gint col, gint relative)
{
	gint new_row;
	gint new_col;

	if (relative)
		{
		new_row = ct->focus_row;
		new_col = ct->focus_column;

		new_row += row;
		if (new_row < 0) new_row = 0;
		if (new_row >= ct->rows) new_row = ct->rows - 1;

		while(col != 0)
			{
			if (col < 0)
				{
				new_col--;
				col++;
				}
			else
				{
				new_col++;
				col--;
				}

			if (new_col < 0)
				{
				if (new_row > 0)
					{
					new_row--;
					new_col = ct->columns - 1;
					}
				else
					{
					new_col = 0;
					}
				}
			if (new_col >= ct->columns)
				{
				if (new_row < ct->rows - 1)
					{
					new_row++;
					new_col = 0;
					}
				else
					{
					new_col = ct->columns - 1;
					}
				}
			}
		}
	else
		{
		new_row = row;
		new_col = col;

		if (new_row >= ct->rows)
			{
			if (ct->rows > 0)
				new_row = ct->rows - 1;
			else
				new_row = 0;
			new_col = ct->columns - 1;
			}
		if (new_col >= ct->columns) new_col = ct->columns - 1;
		}

	if (new_row == ct->rows - 1)
		{
		gint l;

		/* if we moved beyond the last image, go to the last image */

		l = g_list_length(ct->cd->list);
		if (ct->rows > 1) l -= (ct->rows - 1) * ct->columns;
		if (new_col >= l) new_col = l - 1;
		}

	if (ct->cd->list && (new_row != ct->focus_row || new_col != ct->focus_column) )
		{
		gint old_row;
		gint old_col;

		if (debug) printf("collect-table.c: highlight updated\n");

		GTK_CLIST(ct->clist)->focus_row = new_row;

		old_row = ct->focus_row;
		old_col = ct->focus_column;
		ct->focus_row = -1;
		ct->focus_column = -1;
		collection_table_update_cell(ct, old_row, old_col,
			collection_table_is_selected(ct, collection_table_find_info(ct, old_row, old_col)) );

		ct->focus_row = new_row;
		ct->focus_column = new_col;
		collection_table_update_cell(ct, new_row, new_col,
			collection_table_is_selected(ct, collection_table_find_info(ct, new_row, new_col)) );

		/* scroll, if needed */
		if (gtk_clist_row_is_visible(GTK_CLIST(ct->clist), new_row) != GTK_VISIBILITY_FULL)
			{
			if (new_row < old_row)
				gtk_clist_moveto(GTK_CLIST(ct->clist), new_row, new_col, 0.0, -1);
			else
				gtk_clist_moveto(GTK_CLIST(ct->clist), new_row, new_col, 1.0, -1);
			}
		}

	if (debug) printf("collect-table.c: highlight pos %d x %d\n", ct->focus_column, ct->focus_row);
	ct->focus_info = collection_table_find_info(ct, ct->focus_row, ct->focus_column);
}

static void collection_table_update_focus(CollectTable *ct)
{
	gint new_row = 0;
	gint new_col = 0;

	if (ct->focus_info && collection_table_find_position(ct, ct->focus_info, &new_row, &new_col))
		{
		/* first find the old focus, if it exists and is valid */
		}
	else
		{
		/* (try to) stay where we were */
		new_row = ct->focus_row;
		new_col = ct->focus_column;
		}

	collection_table_move_focus(ct, new_row, new_col, FALSE);
}

/* used to figure the page up/down distances */
static gint clist_row_height(GtkCList *clist)
{
	GtkAdjustment* adj;
	gint ret;

	adj = gtk_clist_get_vadjustment(clist);

	if (!adj) return 1;

	ret = adj->page_size / adj->step_increment;
	if (ret < 1)
		ret = 1;
	else if (ret > 1)
		ret--;

	return ret;
}

static gint clist_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	CollectTable *ct = data;
	gint stop_signal = FALSE;
	gint focus_row = 0;
	gint focus_col = 0;
	CollectInfo *info;

	switch (event->keyval)
		{
		case GDK_Left: case GDK_KP_Left:
			focus_col = -1;
			stop_signal = TRUE;
			break;
		case GDK_Right: case GDK_KP_Right:
			focus_col = 1;
			stop_signal = TRUE;
			break;
		case GDK_Up: case GDK_KP_Up:
			focus_row = -1;
			stop_signal = TRUE;
			break;
		case GDK_Down: case GDK_KP_Down:
			focus_row = 1;
			stop_signal = TRUE;
			break;
		case GDK_Page_Up: case GDK_KP_Page_Up:
			focus_row = -clist_row_height(GTK_CLIST(ct->clist));
			stop_signal = TRUE;
			break;
		case GDK_Page_Down: case GDK_KP_Page_Down:
			focus_row = clist_row_height(GTK_CLIST(ct->clist));
			stop_signal = TRUE;
			break;
		case GDK_Home: case GDK_KP_Home:
			focus_row = -ct->focus_row;
			focus_col = -ct->focus_column;
			stop_signal = TRUE;
			break;
		case GDK_End: case GDK_KP_End:
			focus_row = ct->rows - 1 - ct->focus_row;
			focus_col = ct->columns - 1 - ct->focus_column;
			stop_signal = TRUE;
			break;
		case GDK_space:
			info = collection_table_find_info(ct, ct->focus_row, ct->focus_column);
			if (info)
				{
				ct->mouse_start = info;
				if (event->state & GDK_CONTROL_MASK)
					{
					collection_table_select_util(ct, info, !collection_table_is_selected(ct, info));
					}
				else
					{
					collection_table_unselect_all(ct);
					collection_table_select(ct, info);
					}
				}
			stop_signal = TRUE;
			break;
		default:
			break;
		}

	if (focus_row != 0 || focus_col != 0)
		{
		CollectInfo *new_info;
		CollectInfo *old_info;

		old_info = collection_table_find_info(ct, ct->focus_row, ct->focus_column);
		collection_table_move_focus(ct, focus_row, focus_col, TRUE);
		new_info = collection_table_find_info(ct, ct->focus_row, ct->focus_column);

		if (new_info != old_info)
			{
			if (event->state & GDK_SHIFT_MASK)
				{
				gtk_clist_freeze(GTK_CLIST(ct->clist));
				if (!collection_rectangular_selection)
					{
					/* faster */
					collection_table_select_region_util(ct, old_info, new_info, FALSE);
					}
				else
					{
					/* slower */
					collection_table_select_region_util(ct, ct->mouse_start, old_info, FALSE);
					}
				collection_table_select_region_util(ct, ct->mouse_start, new_info, TRUE);
				gtk_clist_thaw(GTK_CLIST(ct->clist));
				}
			else if (event->state & GDK_CONTROL_MASK)
				{
				ct->mouse_start = new_info;
				}
			else
				{
				ct->mouse_start = new_info;
				collection_table_unselect_all(ct);
				collection_table_select(ct, new_info);
				}
			}
		}

	if (stop_signal)
		{
		gtk_signal_emit_stop_by_name(GTK_OBJECT(widget), "key_press_event");
		tip_unschedule(ct);
		}

	return stop_signal;
}

/*
 *-------------------------------------------------------------------
 * insert marker
 *-------------------------------------------------------------------
 */
#define COLUMN_PADDING 3        /* stolen from gtkclist.c */
#define COLUMN_SPACING 1        /* same */

static void clist_insert_marker(CollectTable *ct, CollectInfo *info, gint enable)
{
	gint x, y;
	gint row, col;

	if (!enable)
		{
		if (ct->marker_window) gtk_widget_destroy(ct->marker_window);
		ct->marker_window = NULL;

		return;
		}

	if (!info)
		{
		row = -1;
		col = -1;

		gdk_window_get_pointer(GTK_CLIST(ct->clist)->clist_window, &x, &y, NULL);
		gtk_clist_get_selection_info(GTK_CLIST(ct->clist), x, y, &row, &col);

		info = collection_table_find_info(ct, row, col);
		}

	ct->marker_info = info;

	row = -1;
	col = -1;

	if (!ct->marker_window)
		{
		GtkWidget *pmap;
		GdkPixmap *pixmap;
		GdkBitmap *mask;
		GdkPixbuf *pb;

		ct->marker_window = gtk_window_new(GTK_WINDOW_POPUP);
		gtk_window_set_policy(GTK_WINDOW(ct->marker_window), FALSE, FALSE, TRUE);

		pb = gdk_pixbuf_new_from_xpm_data((const char **)marker_xpm);
		gdk_pixbuf_render_pixmap_and_mask(pb, &pixmap, &mask, 128);
		gdk_pixbuf_unref(pb);

		pmap = gtk_pixmap_new(pixmap, mask);
		gtk_container_add(GTK_CONTAINER(ct->marker_window), pmap);
		gtk_widget_show(pmap);

		gtk_widget_shape_combine_mask(ct->marker_window, mask, 0, 0);

		gdk_pixmap_unref(pixmap);
		if (mask) gdk_bitmap_unref(mask);
		}

	if (info)
		{
		gint wx, wy;
		gint px, py;
		gint l, h;
		gint ww, wh;

		collection_table_find_position(ct, info, &row, &col);

		gdk_window_get_pointer(NULL, &px, &py, NULL);

		x = GTK_CLIST(ct->clist)->column[(col)].area.x + GTK_CLIST(ct->clist)->hoffset - COLUMN_SPACING - COLUMN_PADDING;
		y = GTK_CLIST(ct->clist)->row_height * (row) + (((row) + 1) * 1) + GTK_CLIST(ct->clist)->voffset;

		gdk_window_get_origin(GTK_CLIST(ct->clist)->clist_window, &wx, &wy);
		gdk_window_get_size(GTK_CLIST(ct->clist)->clist_window, &ww, &wh);

	        x += wx;
	        y += wy;

		l = y;
		h = y + GTK_CLIST(ct->clist)->row_height;

		x -= MARKER_WIDTH / 2;
		y = ((h + l) / 2) - (MARKER_HEIGHT / 2);

#if 1
		/*
		 * try to keep marker window away from the mouse
		 * (the window interferes with dnd)
		 */
		if (px >= x - 4 && px < x + MARKER_WIDTH + 4 && py >= y - 4 && py < y + MARKER_HEIGHT + 4)
			{
			if (py > l + MARKER_HEIGHT)
				{
				y = l;
				}
			else if (py < h - MARKER_HEIGHT)
				{
				y = h - MARKER_HEIGHT;
				}
			}
#endif

		if (y >= wy + wh - MARKER_HEIGHT) y = wy + wh - MARKER_HEIGHT;
		if (y < wy) y = wy;

		if (gtk_clist_row_is_visible(GTK_CLIST(ct->clist), row))
			{
			if (!GTK_WIDGET_VISIBLE(ct->marker_window))
				{
				gtk_widget_popup(ct->marker_window, x, y);
				}
			else
				{
				gdk_window_move(ct->marker_window->window, x, y);
				}
			}
		}
	else
		{
		if (GTK_WIDGET_VISIBLE(ct->marker_window)) gtk_widget_hide(ct->marker_window);
		}
}

/*
 *-------------------------------------------------------------------
 * mouse drag auto-scroll
 *-------------------------------------------------------------------
 */

#define CTABLE_SCROLL_THRESHHOLD (thumb_max_height / 2)

static gint collection_table_drag_scroll_cb(gpointer data)
{
	CollectTable *ct = data;
	GdkWindow *window;
	gint x, y;
	gint w, h;
	gint amt = 0;
	GtkAdjustment *adj;

	if (ct->scroll_timeout_id == -1) return FALSE;

	window = GTK_CLIST(ct->clist)->clist_window;
	gdk_window_get_pointer(window, &x, &y, NULL);
	gdk_window_get_size(window, &w, &h);

	/* check bounds, remove if out */
	if (x < 0 || x >= w || y < 0 || y >= h)
		{
		ct->scroll_timeout_id = -1;
		clist_insert_marker(ct, NULL, FALSE);
		return FALSE;
		}

	if (h < CTABLE_SCROLL_THRESHHOLD * 2)
		{
		if (y < h / 2)
			{
			amt = 0 - ((h / 2) - y);
			}
		else
			{
			amt = y - (h / 2);
			}
		}
	else if (y < CTABLE_SCROLL_THRESHHOLD)
		{
		amt = 0 - (CTABLE_SCROLL_THRESHHOLD - y);
		}
	else if (y >= h - CTABLE_SCROLL_THRESHHOLD)
		{
		amt = y - (h - CTABLE_SCROLL_THRESHHOLD);
		}

	if (amt != 0)
		{
		adj = gtk_clist_get_vadjustment(GTK_CLIST(ct->clist));
		gtk_adjustment_set_value(adj, MIN(adj->value + amt, adj->upper - adj->page_size));
		if (ct->marker_window) clist_insert_marker(ct, NULL, TRUE);
		}

	return TRUE;
}

static void collection_table_drag_scroll(CollectTable *ct, gint scroll)
{
	if (!scroll)
		{
		if (ct->scroll_timeout_id != -1)
			{
			gtk_timeout_remove(ct->scroll_timeout_id);
			}
		ct->scroll_timeout_id = -1;
		clist_insert_marker(ct, NULL, FALSE);
		return;
		}

	if (ct->scroll_timeout_id == -1)
		{
		ct->scroll_timeout_id = gtk_timeout_add(80, collection_table_drag_scroll_cb, ct);
		}
}

/*
 *-------------------------------------------------------------------
 * mouse callbacks
 *-------------------------------------------------------------------
 */

static void clist_button_motion(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	CollectTable *ct = data;
	gint row, col;
	CollectInfo *info;

	if (bevent->window != GTK_CLIST(ct->clist)->clist_window) return;

	gtk_clist_get_selection_info(GTK_CLIST(ct->clist), bevent->x, bevent->y, &row, &col);

	info = collection_table_find_info(ct, row, col);

	if (!ct->mouse_in_drag)
		{
		tip_update(ct, info);
		collection_table_drag_scroll(ct, FALSE);
		return;
		}

	if (!info || info == ct->mouse_current) return;

	gtk_clist_freeze(GTK_CLIST(ct->clist));

	if (ct->mouse_add)
		{
		collection_table_select_region_util(ct, ct->mouse_start, ct->mouse_current, FALSE);
		}

	ct->mouse_current = info;
	collection_table_select_region_util(ct, ct->mouse_start, ct->mouse_current, ct->mouse_add);

	gtk_clist_thaw(GTK_CLIST(ct->clist));
}

static void clist_button_press(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	CollectTable *ct = data;
	gint row, col;
	CollectInfo *info;
	GtkWidget *menu;

	tip_unschedule(ct);

	if (bevent->window != GTK_CLIST(ct->clist)->clist_window) return;

	gtk_clist_get_selection_info(GTK_CLIST(ct->clist), bevent->x, bevent->y, &row, &col);
	info = collection_table_find_info(ct, row, col);

	switch (bevent->button)
		{
		case 1:
			if (info)
				{
				collection_table_move_focus(ct, row, col, FALSE);
				GTK_CLIST(ct->clist)->focus_row = ct->focus_row;
				}

			if (!info) return;
			if (bevent->type == GDK_2BUTTON_PRESS)
				{
				main_image_change_to_from_collection(ct->cd, info);
				return;
				}

			ct->mouse_in_drag = TRUE;
			gdk_pointer_grab(GTK_CLIST(ct->clist)->clist_window, FALSE,
					 GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
					 NULL, NULL, bevent->time);
                        gtk_grab_add(ct->clist);

			gtk_clist_freeze(GTK_CLIST(ct->clist));

			if (bevent->state & GDK_CONTROL_MASK)
				{
				ct->mouse_add = !collection_table_is_selected(ct, info);
				if ((bevent->state & GDK_SHIFT_MASK) && ct->prev_selection)
					{
					collection_table_select_region_util(ct, ct->prev_selection, info, ct->mouse_add);
					}
				else
					{
					collection_table_select_util(ct, info, ct->mouse_add);
					}
				}
			else if (bevent->state & GDK_SHIFT_MASK)
				{
				ct->mouse_add = TRUE;
				collection_table_unselect_all(ct);

				if (ct->prev_selection)
					{
					collection_table_select_region_util(ct, ct->prev_selection, info, TRUE);
					}
				else
					{
					collection_table_select_util(ct, info, TRUE);
					}
				}
			else
				{
				ct->mouse_add = TRUE;
				collection_table_unselect_all(ct);

				collection_table_select_util(ct, info, TRUE);
				}

			gtk_clist_thaw(GTK_CLIST(ct->clist));

			ct->mouse_start = info;
			ct->mouse_current = info;

			break;
		case 2:
		case 3:
			if (info)
				{
				collection_table_move_focus(ct, row, col, FALSE);
				GTK_CLIST(ct->clist)->focus_row = ct->focus_row;
				}

			if (info && !collection_table_is_selected(ct, info))
				{
				collection_table_unselect_all(ct);
				collection_table_select_util(ct, info, TRUE);
				}
			if (bevent->button == 3)
				{
				menu = collection_table_popup_menu(ct, (info != NULL));
				gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, bevent->button, bevent->time);
				}
			ct->mouse_current = info;

			break;
		default:
			break;
		}
}

static void clist_button_release(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	CollectTable *ct = data;

	tip_schedule(ct);

	if (!ct->mouse_in_drag) return;

	gtk_grab_remove(ct->clist);
	gdk_pointer_ungrab(bevent->time);

	ct->mouse_in_drag = FALSE;
}

static void clist_button_leave(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	CollectTable *ct = data;

	tip_unschedule(ct);

	collection_table_drag_scroll(ct, FALSE);
	clist_insert_marker(ct, NULL, FALSE);
}

/*
 *-------------------------------------------------------------------
 * clist populate, add, insert, etc.
 *-------------------------------------------------------------------
 */
static void collection_table_cell_update(CollectTable *ct, CollectInfo *ci, gint row, gint col)
{
	if (ci && ci->pixmap)
		{
		gint hs, vs;

		gdk_window_get_size(ci->pixmap, &hs, &vs);
		hs = (thumb_max_width + 10 - hs) / 2;
		vs = (thumb_max_height + 6 - vs) / 2;
		gtk_clist_set_shift(GTK_CLIST(ct->clist), row, col, 0, hs);
		gtk_clist_set_pixmap(GTK_CLIST(ct->clist), row, col, ci->pixmap, ci->mask);
		}
	else
		{
		gtk_clist_set_text(GTK_CLIST(ct->clist), row, col, NULL);
		}
}

static void collection_table_add_icon(CollectTable *ct, CollectInfo *ci, gint row, gint col)
{
	GList *temp;

	collection_table_cell_update(ct, ci, row, col);

	temp = gtk_clist_get_row_data(GTK_CLIST(ct->clist), row);
	temp = g_list_nth(temp, col);
	temp->data = ci;

	collection_table_update_cell(ct, row, col, collection_table_is_selected(ct, ci));
}

static gint collection_table_add_row(CollectTable *ct)
{
	gchar *text[COLLECT_TABLE_TEXT_COLUMNS];
	GList *temp = NULL;
	gint i;
	gint row;

	for (i = 0; i < COLLECT_TABLE_TEXT_COLUMNS; i++) text[i] = NULL;

	row = gtk_clist_append(GTK_CLIST(ct->clist), text);

	for (i = 0; i < ct->columns; i++) temp = g_list_prepend(temp, NULL);
			
	gtk_clist_set_row_data_full(GTK_CLIST(ct->clist), row, temp, (GtkDestroyNotify)g_list_free);

	return row;
}

static void collection_table_populate(CollectTable *ct, gint resize)
{
	gint col, row;
	GList *work;

	collection_table_verify_selections(ct);

	gtk_clist_freeze(GTK_CLIST(ct->clist));
	gtk_clist_clear(GTK_CLIST(ct->clist));

	if (resize)
		{
		gint i;
		gtk_clist_set_row_height(GTK_CLIST(ct->clist), thumb_max_height + 6);
		for (i = 0; i < COLLECT_TABLE_MAX_COLUMNS; i++)
			{
			gtk_clist_set_column_resizeable(GTK_CLIST(ct->clist), i, FALSE);
			gtk_clist_set_column_justification(GTK_CLIST(ct->clist), i, GTK_JUSTIFY_LEFT);
			if (i < ct->columns)
				{
				gtk_clist_set_column_visibility(GTK_CLIST(ct->clist), i, TRUE);
				gtk_clist_set_column_auto_resize(GTK_CLIST(ct->clist), i, FALSE);
				gtk_clist_set_column_width(GTK_CLIST(ct->clist), i, thumb_max_width + 10);
				}
			else
				{
				gtk_clist_set_column_visibility(GTK_CLIST(ct->clist), i, FALSE);
				}
			}
		}

	col = 0;
	row = -1;
	work = ct->cd->list;
	while(work)
		{
		CollectInfo *ci = work->data;

		if (col == 0)
			{
			row = collection_table_add_row(ct);
			}

		collection_table_add_icon(ct, ci, row, col);

		work = work->next;
		col++;
		if (col >= ct->columns) col = 0;
		}

	gtk_clist_thaw(GTK_CLIST(ct->clist));

	ct->rows = row + 1;

	collection_table_update_focus(ct);
	collection_table_update_status(ct);
}

static void collection_table_populate_at_new_size(CollectTable *ct, gint w, gint h)
{
	gint new_cols;

	new_cols = (w  - 6) / (thumb_max_width + 10 + 6 + 1);
	if (new_cols < 1) new_cols = 1;

	if (new_cols == ct->columns) return;

	ct->columns = new_cols;

	collection_table_populate(ct, TRUE);

	if (debug) printf("col tab pop cols=%d rows=%d\n", ct->columns, ct->rows);
}

static void collection_table_sync(CollectTable *ct)
{
	GList *work;

	if (ct->frozen) return;

	gtk_clist_freeze(GTK_CLIST(ct->clist));
	work = ct->cd->list;
	while (work)
		{
		CollectInfo *ci;
		gint r, c;

		ci = work->data;
		
		if (collection_table_find_position(ct, ci, &r, &c))
			{
			collection_table_add_icon(ct, ci, r, c);
			}
		work = work->next;
		}
	gtk_clist_thaw(GTK_CLIST(ct->clist));
}

static void collection_table_append(CollectTable *ct, CollectInfo *ci)
{
	gint row, col;

	row = GTK_CLIST(ct->clist)->rows - 1;

	if (row < 0)
		{
		row = collection_table_add_row(ct);
		col = 0;
		}
	else
		{
		GList *temp = gtk_clist_get_row_data(GTK_CLIST(ct->clist), row);
		col = g_list_index(temp, NULL);
		if (col < 0)
			{
			row = collection_table_add_row(ct);
			col = 0;
			}
		}

	if (debug) printf("col tab app col=%d row=%d\n", col, row);

	collection_table_add_icon(ct, ci, row, col);

	ct->rows = row + 1;

	collection_table_update_status(ct);
}

/* list is a list of paths */
static void collection_table_inserted(CollectTable *ct, CollectInfo *ci)
{
	GList *point;

	point = g_list_find(ct->cd->list, ci);

	if (!point) return;

	collection_table_append(ct, ci);

	collection_table_sync(ct);
}

static void collection_table_remove(CollectTable *ct, CollectInfo *ci)
{
	gint row;

	collection_table_unselect(ct, ci);

	for (row = 0; row < ct->rows; row++)
		{
		GList *temp;
		gint col;

		temp = gtk_clist_get_row_data(GTK_CLIST(ct->clist), row);
		col = g_list_index(temp, ci);
		if (col >= 0)
			{
			GList *work;
			CollectInfo *new;

			if (col + 1 >= ct->columns)
				{
				if (row + 1 < ct->rows)
					{
					temp = gtk_clist_get_row_data(GTK_CLIST(ct->clist), row + 1);
					new = temp->data;
					}
				else
					{
					new = NULL;
					}
				}
			else
				{
				new = g_list_nth_data(temp, col + 1);
				}

			work = g_list_find(ct->cd->list, new);

			gtk_clist_freeze(GTK_CLIST(ct->clist));

			while(work)
				{
				CollectInfo *wci = work->data;
				collection_table_add_icon(ct, wci, row, col);
				work = work->next;
				col++;
				if (col >= ct->columns)
					{
					col = 0;
					row++;
					}
				}
			if (col == 0)
				{
				gtk_clist_remove(GTK_CLIST(ct->clist), row);
				ct->rows--;
				}
			else
				{
				collection_table_add_icon(ct, NULL, row, col);
				}

			gtk_clist_thaw(GTK_CLIST(ct->clist));

			collection_table_update_focus(ct);
			collection_table_update_status(ct);
			return;
			}
		}
}

void collection_table_add_path_list(CollectTable *ct, GList *list)
{
	GList *work;

	if (!list) return;

	gtk_clist_freeze(GTK_CLIST(ct->clist));

	work = list;
	while(work)
		{
		collection_add(ct->cd, (gchar *)work->data, FALSE);
		work = work->next;
		}

	gtk_clist_thaw(GTK_CLIST(ct->clist));
}

static void collection_table_insert_path_list(CollectTable *ct, GList *list, CollectInfo *insert_ci)
{
	GList *work;

	if (!list) return;

	gtk_clist_freeze(GTK_CLIST(ct->clist));
	ct->frozen = TRUE;

	work = list;
	while(work)
		{
		collection_insert(ct->cd, (gchar *)work->data, insert_ci, FALSE);
		work = work->next;
		}

	ct->frozen = FALSE;
	gtk_clist_thaw(GTK_CLIST(ct->clist));

	collection_table_sync(ct);
}

static void collection_table_move_by_info_list(CollectTable *ct, GList *info_list, gint row, gint col)
{
	GList *work;
	GList *insert_pos = NULL;
	GList *temp;
	CollectInfo *info;

	if (!info_list) return;

	info = collection_table_find_info(ct, row, col);

	if (!info_list->next && info_list->data == info) return;

	if (info) insert_pos = g_list_find(ct->cd->list, info);

	/* FIXME: this may get slow for large lists */
	work = info_list;
	while (insert_pos && work)
		{
		if (insert_pos->data == work->data)
			{
			insert_pos = insert_pos->next;
			work = info_list;
			}
		else
			{
			work = work->next;
			}
		}

	work = info_list;
	while (work)
		{
		ct->cd->list = g_list_remove(ct->cd->list, work->data);
		work = work->next;
		}

	/* place them back in */
	temp = g_list_copy(info_list);

	if (insert_pos)
		{
		ct->cd->list = uig_list_insert_list(ct->cd->list, insert_pos, temp);
		}
	else if (info)
		{
		ct->cd->list = g_list_concat(temp, ct->cd->list);
		}
	else
		{
		ct->cd->list = g_list_concat(ct->cd->list, temp);
		}

	ct->cd->changed = TRUE;

	collection_table_sync(ct);
}


/*
 *-------------------------------------------------------------------
 * updating
 *-------------------------------------------------------------------
 */

void collection_table_file_update(CollectTable *ct, CollectInfo *ci)
{
	gint row, col;
	gfloat value;

	if (!ci)
		{
		collection_table_update_extras(ct, FALSE, 0.0);
		return;
		}

	if (!collection_table_find_position(ct, ci, &row, &col)) return;

	value = (float)(row * ct->columns + col) / (ct->columns * ct->rows);

	collection_table_update_extras(ct, TRUE, value);
	collection_table_cell_update(ct, ci, row, col);
}

void collection_table_file_add(CollectTable *ct, CollectInfo *ci)
{
	if (g_list_index(ct->cd->list, ci) < 0) return;

	collection_table_append(ct, ci);
}

void collection_table_file_insert(CollectTable *ct, CollectInfo *ci)
{
	if (g_list_index(ct->cd->list, ci) < 0) return;

	collection_table_inserted(ct, ci);
}

void collection_table_file_remove(CollectTable *ct, CollectInfo *ci)
{
	collection_table_remove(ct, ci);
}

void collection_table_refresh(CollectTable *ct)
{
	collection_table_populate(ct, FALSE);
}

/*
 *-------------------------------------------------------------------
 * dnd
 *-------------------------------------------------------------------
 */

typedef struct {
	CollectTable *ct;
	GList *list;
} CTableConfirmD;

static void confirm_dir_list_cancel(GtkWidget *widget, gpointer data)
{
	/* do nothing */
}


static void collection_table_add_list_proper(CollectTable *ct, GList *list)
{
	if (collection_drop_insert && ct->marker_info)
		{
		collection_table_insert_path_list(ct, list, ct->marker_info);
		}
	else
		{
		collection_table_add_path_list(ct, list);
		}
}

static void collection_table_add_dir_recursive(CollectTable *ct, gchar *path, gint recursive)
{
	GList *d = NULL;
	GList *f = NULL;

	if (path_list(path, &f, recursive ? &d : NULL))
		{
		GList *work;

		f = path_list_filter(f, FALSE);
		d = path_list_filter(d, TRUE);

		f = path_list_sort(f);
		d = path_list_sort(d);

		collection_table_add_list_proper(ct, f);

		work = g_list_last(d);
		while(work)
			{
			collection_table_add_dir_recursive(ct, (gchar *)work->data, TRUE);
			work = work->prev;
			}
		path_list_free(f);
		path_list_free(d);
		}
}

static void confirm_dir_list_do(CollectTable *ct, GList *list, gint recursive)
{
	GList *work = list;
	while (work)
		{
		gchar *path = work->data;
		work = work->next;
		if (isdir(path)) collection_table_add_dir_recursive(ct, path, recursive);
		}
	collection_table_add_list_proper(ct, list);
}


static void confirm_dir_list_add(GtkWidget *widget, gpointer data)
{
	CTableConfirmD *d = data;
	confirm_dir_list_do(d->ct, d->list, FALSE);
}

static void confirm_dir_list_recurse(GtkWidget *widget, gpointer data)
{
	CTableConfirmD *d = data;
	confirm_dir_list_do(d->ct, d->list, TRUE);
}

static void confirm_dir_list_skip(GtkWidget *widget, gpointer data)
{
	CTableConfirmD *d = data;

	collection_table_add_list_proper(d->ct, d->list);
}

static void confirm_dir_list_destroy(GtkWidget *widget, gpointer data)
{
	CTableConfirmD *d = data;
	path_list_free(d->list);
	g_free(d);
}

static GtkWidget *confirm_dir_list(CollectTable *ct, GList *list)
{
	GtkWidget *menu;
	CTableConfirmD *d;

	d = g_new(CTableConfirmD, 1);
	d->ct = ct;
	d->list = list;

	menu = popup_menu_short_lived();
	gtk_signal_connect(GTK_OBJECT(menu), "destroy", confirm_dir_list_destroy, d);

	menu_item_add(menu, _("Dropped list includes directories."), NULL, NULL);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Add contents"), confirm_dir_list_add, d);
	menu_item_add(menu, _("Add contents recursive"), confirm_dir_list_recurse, d);
	menu_item_add(menu, _("Skip directories"), confirm_dir_list_skip, d);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Cancel"), confirm_dir_list_cancel, d);

	return menu;
}

/*
 *-------------------------------------------------------------------
 * dnd
 *-------------------------------------------------------------------
 */
enum {
	TARGET_APP_COLLECTION_MEMBER,
	TARGET_URI_LIST,
	TARGET_TEXT_PLAIN
};

static GtkTargetEntry collection_drag_types[] = {
	{ "application/x-gqview-collection-member", 0, TARGET_APP_COLLECTION_MEMBER },
	{ "text/uri-list", 0, TARGET_URI_LIST },
	{ "text/plain", 0, TARGET_TEXT_PLAIN }
};
static gint n_collection_drag_types = 3;

static GtkTargetEntry collection_drop_types[] = {
	{ "application/x-gqview-collection-member", 0, TARGET_APP_COLLECTION_MEMBER },
	{ "text/uri-list", 0, TARGET_URI_LIST }
};
static gint n_collection_drop_types = 2;


static void collection_table_dnd_data_set(GtkWidget *widget, GdkDragContext *context,
					  GtkSelectionData *selection_data, guint info,
					  guint time, gpointer data)
{
	CollectTable *ct = data;
	GList *list;
	gchar *uri_text;
	gint total;

	switch (info)
		{
		case TARGET_APP_COLLECTION_MEMBER:
			if (!ct->selection) return;

			uri_text = collection_info_list_to_dnd_data(ct->cd, ct->selection, &total);
			if (!uri_text) return;
			break;
		case TARGET_URI_LIST:
		case TARGET_TEXT_PLAIN:
		default:
			list = collection_table_selection_get_list(ct);
			if (!list) return;

			uri_text = make_uri_file_list(list, (info == TARGET_TEXT_PLAIN), &total);
			if (!uri_text) return;

			path_list_free(list);
			break;
		}

	gtk_selection_data_set(selection_data, selection_data->target,
			       8, uri_text, total);
	g_free(uri_text);
}


static void collection_table_dnd_data_get(GtkWidget *widget, GdkDragContext *context,
					  gint x, gint y,
					  GtkSelectionData *selection_data, guint info,
					  guint time, gpointer data)
{
	CollectTable *ct = data;
	GList *list = NULL;
	GList *info_list = NULL;
	CollectionData *source;
	GList *work;

	if (debug) printf(selection_data->data);

	collection_table_drag_scroll(ct, FALSE);
	clist_insert_marker(ct, NULL, FALSE);

	switch (info)
		{
		case TARGET_APP_COLLECTION_MEMBER:
			source = collection_from_dnd_data((gchar *)selection_data->data, &list, &info_list);
			if (source)
				{
				if (source == ct->cd)
					{
					gint row = -1;
					gint col = -1;

					/* it is a move within a collection */
					path_list_free(list);
					list = NULL;

					gtk_clist_get_selection_info(GTK_CLIST(ct->clist), x, y, &row, &col);
					collection_table_move_by_info_list(ct, info_list, row, col);
					}
				else
					{
					/* it is a move/copy across collections */
					if (context->action == GDK_ACTION_MOVE)
						{
						collection_remove_by_info_list(source, info_list);
						}
					}
				g_list_free(info_list);
				}
			break;
		case TARGET_URI_LIST:
			list = get_uri_file_list(selection_data->data);
			work = list;
			while(work)
				{
				if (isdir((gchar *)work->data))
					{
					GtkWidget *menu;
					menu = confirm_dir_list(ct, list);
					gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, time);
					return;
					}
				work = work->next;
				}
			break;
		default:
			list = NULL;
			break;
		}

	if (list)
		{
		if (!collection_drop_insert)
			{
			collection_table_add_path_list(ct, list);
			}
		else
			{
			CollectInfo * info;
			gint row = -1;
			gint col = -1;

			gtk_clist_get_selection_info(GTK_CLIST(ct->clist), x, y, &row, &col);
			info = collection_table_find_info(ct, row, col);
			collection_table_insert_path_list(ct, list, info);
			}
		path_list_free(list);
		}
}

#if 0
static void collection_table_dnd_begin(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	CollectTable *ct = data;

	printf("begin\n");

	tip_unschedule(ct);
}
#endif

static void collection_table_dnd_end(GtkWidget *widget, GdkDragContext *context, gpointer data)
{
	CollectTable *ct = data;

	/* apparently a leave event is not generated on a drop */
	tip_unschedule(ct);

	collection_table_drag_scroll(ct, FALSE);
	clist_insert_marker(ct, NULL, FALSE);
}

static void collection_table_dnd_motion(GtkWidget *widget, GdkDragContext *context,
					gint x, gint y, guint time, gpointer data)
{
	CollectTable *ct = data;

	tip_unschedule(ct);

	/* handle autoscroll */
	collection_table_drag_scroll(ct, TRUE);

	if (collection_drop_insert ||
	    gtk_drag_get_source_widget(context) == ct->clist)
		{
		clist_insert_marker(ct, NULL, TRUE);
		}
}

static void collection_table_dnd_init(CollectTable *ct)
{
	gtk_drag_source_set(ct->clist, GDK_BUTTON2_MASK,
			    collection_drag_types, n_collection_drag_types,
			    GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK);
#if 0
	gtk_signal_connect_(GTK_OBJECT(ct->clist), "drag_begin",
			   collection_table_dnd_begin, ct);
#endif
	gtk_signal_connect(GTK_OBJECT(ct->clist), "drag_data_get",
			   collection_table_dnd_data_set, ct);
	gtk_signal_connect(GTK_OBJECT(ct->clist), "drag_end",
			   collection_table_dnd_end, ct);

	gtk_drag_dest_set(ct->clist,
			  GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
			  collection_drop_types, n_collection_drop_types,
			  GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_ASK);
	gtk_signal_connect(GTK_OBJECT(ct->clist), "drag_motion",
			   collection_table_dnd_motion, ct);
	gtk_signal_connect(GTK_OBJECT(ct->clist), "drag_data_received",
			   collection_table_dnd_data_get, ct);
}

/*
 *-------------------------------------------------------------------
 * init, destruction
 *-------------------------------------------------------------------
 */

static void collection_table_destroy(GtkWidget *widget, gpointer data)
{
	CollectTable *ct = data;

	tip_unschedule(ct);
	clist_insert_marker(ct, NULL, FALSE);
	collection_table_drag_scroll(ct, FALSE);

	g_free(ct);
}

static void collection_table_select_row_cb(GtkCList *clist, gint row, gint column,
					   GdkEvent *event, gpointer data)
{
	gtk_clist_unselect_row(clist, row, column);
}

static void collection_table_sized(GtkWidget *widget, GtkAllocation *allocation, gpointer data)
{
	CollectTable *ct = data;

	collection_table_populate_at_new_size(ct, allocation->width, allocation->height);
}

CollectTable *collection_table_new(CollectionData *cd)
{
	CollectTable *ct;

	ct = g_new0(CollectTable, 1);
	ct->cd = cd;
	ct->columns = 0;
	ct->rows = 0;

	ct->selection = NULL;
	ct->prev_selection = NULL;

	ct->tip_window = NULL;
	ct->tip_delay_id = -1;

	ct->marker_window = NULL;
	ct->marker_info = NULL;

	ct->status_label = NULL;
	ct->extra_label = NULL;

	ct->focus_row = 0;
	ct->focus_column = 0;
	ct->focus_info = NULL;

	ct->scroll_timeout_id = -1;

	ct->frozen = FALSE;

	ct->scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (ct->scrolled),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	ct->clist = gtk_clist_new(COLLECT_TABLE_MAX_COLUMNS);

	/* we roll our own selection, connect to select row, then unselect it */
	gtk_clist_set_selection_mode(GTK_CLIST(ct->clist), GTK_SELECTION_SINGLE);
	gtk_signal_connect(GTK_OBJECT(ct->clist), "select_row", collection_table_select_row_cb, ct);

	gtk_container_add(GTK_CONTAINER(ct->scrolled), ct->clist);
	gtk_widget_show(ct->clist);
	
	gtk_signal_connect(GTK_OBJECT(ct->clist), "destroy", collection_table_destroy, ct);
	gtk_signal_connect(GTK_OBJECT(ct->clist), "size_allocate", collection_table_sized, ct);

	collection_table_dnd_init(ct);

	gtk_signal_connect(GTK_OBJECT(ct->clist),"motion_notify_event",
			   clist_button_motion, ct);
	gtk_signal_connect(GTK_OBJECT(ct->clist),"button_press_event",
                           clist_button_press, ct);
	gtk_signal_connect(GTK_OBJECT(ct->clist),"button_release_event",
			   clist_button_release, ct);
	gtk_signal_connect(GTK_OBJECT(ct->clist),"leave_notify_event",
			   clist_button_leave, ct);
	gtk_widget_set_events(ct->clist, GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK |
			      GDK_BUTTON_PRESS_MASK | GDK_LEAVE_NOTIFY_MASK);

	gtk_signal_connect(GTK_OBJECT(ct->clist),"key_press_event",
			   GTK_SIGNAL_FUNC(clist_keypress), ct);

	return ct;
}

void collection_table_set_labels(CollectTable *ct, GtkWidget *status, GtkWidget *extra)
{
	ct->status_label = status;
	ct->extra_label = extra;
	collection_table_update_status(ct);
	collection_table_update_extras(ct, FALSE, 0.0);
}

CollectInfo *collection_table_get_focus_info(CollectTable *ct)
{
	return collection_table_find_info(ct, ct->focus_row, ct->focus_column);
}

