/* This is for emacs: -*-Mode: C++;-*- */
#if !defined(__INC_GDBI_SESSION_H)
#define __INC_GDBI_SESSION_H

#include <vector>

#include <sigc++/signal_system.h>
#include <sigc++/tunnel.h>
#include <sigc++/dispatch.h>
#include <gql++/driver-manager.h>
#include <uc++/error-handler.h>

#include "gdbi++/driver.h"

namespace gdbi
{

class Connection;


/** The \c Session class represents the \c DataSource and \c
 *  Connection level of database access.  */
class Session : public uC::ErrorHandler
{
    friend class Connection;
  public:
    struct ConnectionInfo
    {
        std::string url;
        Connection *connection;
    };

    /// Constructor.
    Session(uC::PluginManager *pm, const std::string& root,
            SigC::Tunnel *tunnel, bool mt = false);

    /** Destructor
     * Closes all connections made.
     */
    ~Session();
    
    /** Try to establish a new connection.
     * If the connect succeeds, the \c connect signal is emmited.
     */
    void open_connection(const std::string& url,
                         const std::string& user,
                         const std::string& passwd) {
      SigC::pack(slot(this, &Session::do_open_connection), url, user, passwd)->tunnel(requests_);
    }

    /** Get all currently open connections.
     */
    const std::vector<ConnectionInfo>& connections() { return(connections_); }
    
    /** Emitted when a new connection  is established.
     *  \param no The number of the \c Connection.
     */
    SigC::Signal1<void, int> connection_opened;

    /** Emitted when a connection is closed.
     *  \param id The number of the \c Connection.
     */
    SigC::Signal1<void, int> connection_closed;

    const uC::PluginNode& drivers() const {
      return(dm_.get_drivers());
    }
    std::list<std::string> loaded_drivers() {
      return(dm_.get_registered_drivers());
    }
    
    SigC::Tunnel *requests() { return(requests_); }
    SigC::Tunnel *signals() { return(signals_); }
    
    SigC::Signal0<void> destroy;
  private:
    void do_open_connection(std::string url, 
                            std::string user, 
                            std::string passwd);
    void do_close_connection(Connection *conn);
    void do_emit_error(std::string e) { ErrorHandler::emit_error(e); }
    
    SigC::Tunnel *requests_, *signals_;
    bool mt_;
    
    std::vector<ConnectionInfo> connections_;
    std::vector<std::string> paths_;
    GQL::DriverManager dm_;
};

typedef std::vector<Session::ConnectionInfo> ConnectionList;

}

#endif

