#include <iostream>

#include <sigc++/thread_tunnel.h>

#include "gql++/database-metadata.h"
#include "gql++/result-set.h"
#include "gql++/object.h"
#include "gql++/statement.h"

#include "gdbi++/session.h"
#include "gdbi++/connection.h"
#include "gdbi++/table.h"

namespace gdbi
{

using SigC::slot;
using SigC::pack;

using namespace std;

class ConnectionThread : public SigC::Threads::Thread,
                         public SigC::StandardDispatcher
{
  public:
    ConnectionThread() {
      tunnel_ = 0;
    }
    void set_tunnel(SigC::Tunnel *tunnel) {
      tunnel_ = tunnel;
    }
    virtual void *main(void *arg) {
      run();
      cout << "exiting conn thread" << endl;

      //delete tunnel_;
      delete this;
      return(0);
    }
  private:
    SigC::Tunnel *tunnel_;
};

Connection::Connection(Session *session, GQL::Connection *conn,
                       SigC::Tunnel *tunnel, bool mt)
    : ErrorHandler(session), conn_(conn), types_("types")
{
  mt_ = mt;
  
  if (tunnel && mt_)
  {
    // Requests are handled in seperate thread, signals in main thread
    ConnectionThread *thread = new ConnectionThread;
    requests_ = new SigC::ThreadTunnel(thread);
    signals_ = tunnel;
    thread->set_tunnel(requests_);
    thread->start();
  }
  else if (tunnel)
  {
    // All things are handled in main thread, but use tunnels
    requests_ = tunnel; 
    signals_ = 0;
  }
  else
  {
    requests_ = signals_ = 0;
  }
    
  GQL::ResultSet *rs;
  GQL::SQLObject *obj = conn_->create_object();
  vector<string> types(1);
  Table *table;
  string str;

  types[0] = "TABLE";
  rs = conn_->get_meta_data()->get_tables("", "", "", types);
  
  while (rs->next())
  {
    rs->get(2, obj);
    str = obj->to_string();
    table = new Table(conn_, str);
    table->set_parent(this);
    tables_.push_back(table);
  }

  // Populate types_
  types_.set_subschema("unknown",
                       manage(new PropertySchema("Unknown Type", true)));
  types_.set_subschema("number",
                       manage(new PropertySchema("Number", true)));
  types_.set_subschema("string",
                       manage(new PropertySchema("String", true)));
  types_.set_subschema("bool",
                       manage(new PropertySchema("Boolean Value", true)));
  types_.set_subschema("date",
                       manage(new PropertySchema("Date", true)));
}

struct ShutdownInfo
{
    GQL::Connection *connection;
    SigC::ThreadTunnel *tunnel;
    SigC::Signal0<void> destroy_signal;
};

static void do_shutdown(ShutdownInfo *info)
{
  cout << "do_shutdown" << endl;
  
  delete info->connection;

  // Tunnel is passed only in MT mode
  if (info->tunnel)
  {
    cout << "conn closed, exiting dispatcher" << endl;
    info->tunnel->dispatcher()->exit();
  }
  info->destroy_signal.emit();

  delete info;
}

Connection::~Connection()
{
  ShutdownInfo *info = new ShutdownInfo;
  
  SigC::ThreadTunnel *tunnel = dynamic_cast<SigC::ThreadTunnel *>(mt_ ? requests_ : 0);

  info->connection = conn_;
  info->tunnel = tunnel;
  info->destroy_signal = destroy;
  
  // Defer real shutdown to thread if any
  pack(slot(&do_shutdown), info)->tunnel(requests_);
}

const PropertySchema *Connection::type_schema(const string& name) const
{
  const PropertySchema *schema = types_.subschema(name);

  if (!schema)
    schema = types_.subschema("unknown");
  
  return schema;
}

list<string> Connection::types() const
{
  return types_.subschemata();
}

void Connection::do_add_table(Table *table)
{
  tables_.push_back(table);
  table->set_parent(this);

  table->do_insert_into(conn_);
  pack(table_added.slot(), (int)tables_.size() - 1)->tunnel(signals_);
}

void Connection::do_remove_table(int pos)
{
  Table *table = tables_[pos];
  
  if (table)
  {
    table->do_drop();
    tables_.erase(tables_.begin() + pos);
    pack(table_removed.slot(), pos)->tunnel(signals_);
  }
}

}

