/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  sataos.c: Support functions to predict satellite passes.

  Copyright (C)  2001-2003  Alexandru Csete.

  Authors:   Alexandru Csete <csete@users.sourceforge.net>
             John Magliacane (original author of tracking code).

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <math.h>
#include "defaults.h"
#include "satdata.h"
#include "qth.h"
#include "util.h"
#include "satlog.h"
#include "sataos.h"

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

extern GtkWidget *app;
extern qth_struc qth;          /* qth.c */

/* function prototypes */
static void aos_precalc (sat_t *, gdouble);
static void aos_calc (sat_t *, gdouble);




gdouble aos_find_aos (sat_t sat, gdouble start)
{
	/* This function finds and returns the time of AOS (aostime).
	   Stolen from Predict. 
	*/
	gdouble aostime,daynum;

	aostime=0.0;
	daynum = start;

	if ( sat.hasaos && !sat.geostat && !sat.decayed)
	{
		aos_calc (&sat, daynum);

		/* Get the satellite in range */

		while ( sat.el < -1.0 )
		{
			daynum -= 0.00035*(sat.el*(((sat.alt/8400.0)+0.46))-2.0);

			/* Technically, this should be:

			   daynum-=0.0007*(elevation*(((ak/8400.0)+0.46))-2.0);

			   but it sometimes skipped passes for
			   satellites in highly elliptical orbits. */

			aos_calc (&sat, daynum);
		}

		/* Find AOS */

		/** Users using Keplerian data to track the Sun MAY find
		    this section goes into an infinite loop when tracking
		    the Sun if their QTH is below 30 deg N! **/

		while ( aostime == 0.0 )
		{
			if ( fabs (sat.el) < 0.03 )
				aostime=daynum;
			else
			{
				daynum -= sat.el*sqrt(sat.alt)/530000.0;
				aos_calc (&sat, daynum);
			}
		}
	}

	return aostime;
}


gdouble aos_find_los (sat_t sat, gdouble start)
{
	gdouble lostime,daynum;

	lostime=0.0;
	daynum = start;

	if ( !sat.geostat && sat.hasaos && !sat.decayed )
	{
		/* FindLOS code */
		aos_calc (&sat, daynum);
		do
		{
			daynum += sat.el*sqrt(sat.alt)/502500.0;
			aos_calc( &sat, daynum );
			if ( fabs (sat.el) < 0.03 ) 
				lostime = daynum;

		} while (lostime==0.0);
	}

	return lostime;
}


gchar **aos_detailed_list (sat_t sat, guint step)
{
	/* This function finds the next upcoming pass for the satellite "sat".
	   It returns a newly allocated vector containing a line for each
	   timestep "step". The fields in each line are separated by ";".
	   The vector has to be freed by the calling function after use!
	   The returned data is NOT filtered against the user defined
	   bitmask. The first line in the returned vector is the orbit
	   number and the number of data lines.
	*/
	gdouble time,dstep; /* dstep is the time step in daynums */
	gchar *times,*line,*list=NULL,*listbuff,**listv;
	guint i;

	switch (sat.status) {
	case SAT_STATUS_DECAYED:
	case SAT_STATUS_GEOSTAT:
	case SAT_STATUS_NOAOS:
		return NULL;
		break;
	default:
		time = (sat.el > 0) ? sat.los+0.005 : sat.aos;
		/* find next AOS/LOS times */
		sat.aos = aos_find_aos (sat, time);
		sat.los = aos_find_los (sat, sat.aos+0.005);

		dstep = ((gdouble)step)/(3600.0*24.0);
/*		n = round((sat.los-sat.aos)/dstep);*/
		aos_precalc (&sat, sat.aos);
		i = 0;
		for (time=sat.aos; (time<=sat.los) || (sat.el>=0.0); time+=dstep) {
			aos_calc (&sat,time);
			times = dnum2fstr (time,TFORM_SATLIST);
			line = g_strdup_printf ("%s;%7.2f;%7.2f;%7.2f;%7.2f;%5d;%5d;%5d;%5d",
						times, sat.az, sat.el, sat.lat, sat.lon,
						(guint)sat.alt, (guint)sat.range,
						(guint)sat.vel, (guint)sat.fp);

			if (i!=0) {
				listbuff = g_strdup (list);
				g_free (list);
				list = g_strjoin ("?",listbuff,line,NULL);
				g_free (listbuff);
			}
			else
				list = g_strdup (line);
			
			/* clean up memory */
			g_free (times);
			g_free (line);
			i++;
		}
		/* first line is orbit number and number of lines */
		listbuff = g_strdup (list);
		g_free (list);
		line = g_strdup_printf ("%ld;%d", sat.orbit, i);
		list = g_strjoin ("?",line,listbuff,NULL);
		g_free (line);
		g_free (listbuff);
		listv = g_strsplit (list,"?",i);
		g_free (list);
		return listv;
		break;
	}
	return NULL;
}



gchar **aos_detailed_list_new (sat_t sat, guint step, gdouble aos)
{
	/* This function does practically the same as the previous
	   function except that it takes an extra argument specifying
	   the time of AOS so that it can be used to get details about
	   any upcoming pass. If "aos" is 0, the first upcoming pass
	   is calculated.
	*/
	gdouble time,dstep; /* dstep is the time step in daynums */
	gchar *times,*line,*list=NULL,*listbuff,**listv;
	guint i;

	switch (sat.status) {
	case SAT_STATUS_DECAYED:
	case SAT_STATUS_GEOSTAT:
	case SAT_STATUS_NOAOS:
		return NULL;
		break;
	default:
		if (!aos) {
			time = (sat.el > 0) ? sat.los+0.005 : sat.aos;
			/* find next AOS/LOS times */
			sat.aos = aos_find_aos (sat, time);
			sat.los = aos_find_los (sat, sat.aos+0.005);
		}
		else {
			sat.aos = aos;
			sat.los = aos_find_los (sat, sat.aos+0.005);
		}
		dstep = ((gdouble)step)/(3600.0*24.0);
/*		n = round((sat.los-sat.aos)/dstep);*/
		aos_precalc (&sat, sat.aos);
		i = 0;
		for (time=sat.aos; (time<=sat.los) || (sat.el>=0.0); time+=dstep) {
			aos_calc (&sat,time);
			times = dnum2fstr (time,TFORM_SATLIST);
			line = g_strdup_printf ("%s;%7.2f;%7.2f;%7.2f;%7.2f;%5d;%5d;%5d;%5d",
						times, sat.az, sat.el, sat.lat, sat.lon,
						(guint)sat.alt, (guint)sat.range,
						(guint)sat.vel, (guint)sat.fp);

			if (i!=0) {
				listbuff = g_strdup (list);
				g_free (list);
				list = g_strjoin ("?",listbuff,line,NULL);
				g_free (listbuff);
			}
			else
				list = g_strdup (line);
			
			/* clean up memory */
			g_free (times);
			g_free (line);
			i++;
		}
		/* first line is orbit number and number of lines */
		listbuff = g_strdup (list);
		g_free (list);
		line = g_strdup_printf ("%ld;%d", sat.orbit, i);
		list = g_strjoin ("?",line,listbuff,NULL);
		g_free (line);
		g_free (listbuff);
		listv = g_strsplit (list,"?",i);
		g_free (list);
		return listv;
		break;
	}
	return NULL;
}



gchar **aos_simple_list (sat_t sat, guint n , const gchar *format)
{
	/* This function finds AOS/LOS times for the upcoming "n" passes
	   for the satellite "sat". It returns a vector containing one
	   line for each pass. The strings are formatted using "format".
	   The AOS and LOS times are separated with a ";". The vector has
	   to be freed by the calling function. If "sat" has no AOS "NULL"
	   is returned.
	*/

	gchar *list=NULL,**listv,*listbuff,*buff,*buff1,*buff2;
	gdouble time;
	guint i;


	switch (sat.status) {
	case SAT_STATUS_DECAYED:
	case SAT_STATUS_GEOSTAT:
	case SAT_STATUS_NOAOS:
		return NULL;
		break;
	default:
		time = (sat.el > 0) ? sat.los+0.005 : sat.aos;

		/* Get AOS times and add them to list */
		for ( i=0; i<n; i++ ) {
			sat.aos = aos_find_aos (sat, time);
			sat.los = aos_find_los (sat, sat.aos+0.005);
			buff1 = dnum2fstr (sat.aos, format); /* returns newly allocated str */
			buff2 = dnum2fstr (sat.los, format);
			buff = g_strdup_printf ("%s;%s", buff1, buff2);
			if (i==0) {
				list = g_strdup (buff);
			} else {
				listbuff = g_strdup (list);
				g_free (list);
				list = g_strjoin ("?",listbuff,buff,NULL);
				g_free (listbuff);
			}
			/* clean up memory */
			g_free (buff1);
			g_free (buff2);
			g_free (buff);
			/* timestep */
			time = sat.los + 0.005;
		}
		if (list) {
			listv = g_strsplit (list,"?",n);
			g_free (list);
			return listv;
		}
	}

	/* The code should never make it this far */
	satlog_log (SAT_LOG_CRITICAL, _("aos_verbose_list(): Internal error!"));
	return NULL;
}



gchar **aos_simple_list_new (sat_t sat, guint n , const gchar *format)
{
	/* This function does the same as the previous one except
	   that it appends an extra column containing the aos time
	   in "daynum" format. This can be used to call the
	   aos_get_detailed_list_new function to get detailed info
	   about a specific pass.
	*/

	gchar *list=NULL,**listv,*listbuff,*buff,*buff1,*buff2;
	gdouble time;
	guint i;


	switch (sat.status) {
	case SAT_STATUS_DECAYED:
	case SAT_STATUS_GEOSTAT:
	case SAT_STATUS_NOAOS:
		return NULL;
		break;
	default:
		time = (sat.el > 0) ? sat.los+0.005 : sat.aos;

		/* Get AOS times and add them to list */
		for ( i=0; i<n; i++ ) {
			sat.aos = aos_find_aos (sat, time);
			sat.los = aos_find_los (sat, sat.aos+0.005);
			buff1 = dnum2fstr (sat.aos, format); /* returns newly allocated str */
			buff2 = dnum2fstr (sat.los, format);
			buff = g_strdup_printf ("%s;%s;%f", buff1, buff2, sat.aos);
			if (i==0) {
				list = g_strdup (buff);
			} else {
				listbuff = g_strdup (list);
				g_free (list);
				list = g_strjoin ("?",listbuff,buff,NULL);
				g_free (listbuff);
			}
			/* clean up memory */
			g_free (buff1);
			g_free (buff2);
			g_free (buff);
			/* timestep */
			time = sat.los + 0.005;
		}
		if (list) {
			listv = g_strsplit (list,"?",n);
			g_free (list);
			return listv;
		}
	}

	/* The code should never make it this far */
	satlog_log (SAT_LOG_CRITICAL, _("aos_verbose_list(): Internal error!"));
	return NULL;
}




static void
aos_precalc (sat_t *sat, gdouble dnum)
{
	/* This function performs preliminary calculations
	   prior to tracking or prediction. The code is similar
	   to PreCalc in engine.c but this one uses a pointer to
	   sat_struct.
	   Shamelessly stolen from Predict (but modified as necessary).
	*/
 
	sat->daynum = dnum;

	sat->epoch=DayNum(1,0,sat->epyear)+sat->epday;
	sat->age=sat->daynum-sat->epoch;
	sat->yr=(gfloat)sat->epyear;

	/* Do the Y2K thing... */

	if ( sat->yr <= 50.0)
		sat->yr += 100.0;

	sat->t1 = sat->yr - 1.0;
	sat->df = 366.0+floor(365.25*(sat->t1-80.0)) - 
		floor(sat->t1/100.0)+floor(sat->t1/400.0+0.75);
	sat->t1 = (sat->df+29218.5)/36525.0;
	sat->t1 = 6.6460656+sat->t1*(2400.051262+sat->t1*2.581e-5);
	sat->se = sat->t1/24.0-sat->yr;
	sat->n0 = sat->meanmo+(sat->age*sat->drag);
	sat->sma = 331.25*pow((1440.0/sat->n0),(2.0/3.0));
	sat->e2 = 1.0-(sat->eccn*sat->eccn); 
	sat->e1 = sqrt(sat->e2);
	/**** BIG TIME BUG??? ****/
	sat->k2 = 9.95*(exp(log(R0/sat->sma)*3.5))/sat->e2*sat->e2;
	/*************************/
	sat->s1 = sin(sat->incl*DEG2RAD); 
	sat->c1 = cos(sat->incl*DEG2RAD);
	sat->l8 = qth.lat*DEG2RAD; 
	sat->s9 = sin(sat->l8); 
	sat->c9 = cos(sat->l8);
	sat->s8 = sin(-qth.lon*DEG2RAD); 
	sat->c8 = cos(qth.lon*DEG2RAD);
	sat->r9 = R0*(1.0+(FF/2.0)*(cos(2.0*sat->l8)-1.0))+(qth.alt/1000.0);
	sat->l8 = atan((1.0-FF)*(1.0-FF)*sat->s9/sat->c9);
	sat->z9 = sat->r9*sin(sat->l8);
	sat->x9 = sat->r9*cos(sat->l8)*sat->c8;
	sat->y9 = sat->r9*cos(sat->l8)*sat->s8;
	sat->apogee = sat->sma*(1.0+sat->eccn)-R0;
	sat->perigee = sat->sma*(1.0-sat->eccn)-R0;

}


gdouble c[4][3];

static void
aos_calc (sat_t *sat , gdouble dnum)
{
	/* This is the same stuff as in engine.c but uses
	   a pointer to a sat structure.
	   Stolen from Predict - modified as necessary.
	*/

	sat->age = dnum-sat->epoch;
	sat->o = DEG2RAD*(sat->raan-(sat->age)*sat->k2*sat->c1);
	sat->s0 = sin(sat->o); 
	sat->c0 = cos(sat->o);
	sat->w = DEG2RAD*(sat->argper+(sat->age)*sat->k2*
				   (2.5*sat->c1*sat->c1-0.5));
	sat->s2 = sin(sat->w); 
	sat->c2 = cos(sat->w);
	c[1][1] = sat->c2*sat->c0-sat->s2*sat->s0*sat->c1;
	c[1][2] =-sat->s2*sat->c0-sat->c2*sat->s0*sat->c1;
	c[2][1] = sat->c2*sat->s0+sat->s2*sat->c0*sat->c1;
	c[2][2] =-sat->s2*sat->s0+sat->c2*sat->c0*sat->c1;
	c[3][1] = sat->s2*sat->s1;
	c[3][2] = sat->c2*sat->s1;
	sat->q0 = (sat->meanan/360.0)+sat->eporb;
	sat->phase = sat->n0*sat->age+sat->q0; 
	sat->orbit = (glong)floor(sat->phase);
	sat->phase = sat->phase-floor(sat->phase);
	sat->m = sat->phase*TP;
	sat->e = sat->m+sat->eccn*
		(sin(sat->m)+0.5*sat->eccn*sin(sat->m*2.0));
	do   /* Kepler's Equation */
	{
		sat->s3 = sin(sat->e); 
		sat->c3 = cos(sat->e); 
		sat->r3 = 1.0-sat->eccn*sat->c3;
		sat->m1 = sat->e-sat->eccn*sat->s3; 
		sat->m5 = sat->m1 - sat->m;
		sat->e  = sat->e - sat->m5/sat->r3;
	} while ( fabs(sat->m5) >= 1.0e-6 );
	sat->x0 = sat->sma*(sat->c3-sat->eccn); 
	sat->yzero = sat->sma*sat->e1*sat->s3;
	sat->r = sat->sma*sat->r3; 
	sat->x1 = sat->x0*c[1][1]+sat->yzero*c[1][2];
	sat->yone = sat->x0*c[2][1]+sat->yzero*c[2][2];
	sat->z1 = sat->x0*c[3][1]+sat->yzero*c[3][2];
	sat->g7 = (dnum-sat->df)*1.0027379093+sat->se;
	sat->g7 = TP*(sat->g7-floor(sat->g7));
	sat->s7 = -sin(sat->g7); 
	sat->c7 = cos(sat->g7);
	sat->x = sat->x1*sat->c7-sat->yone*sat->s7; 
	sat->y = sat->x1*sat->s7+sat->yone*sat->c7;
	sat->z = sat->z1; 
	sat->x5 = sat->x-sat->x9; 
	sat->y5 = sat->y-sat->y9; 
	sat->z5 = sat->z-sat->z9;
	sat->range2 = sat->x5*sat->x5+sat->y5 *	sat->y5+sat->z5*sat->z5; 
	sat->z8 = sat->x5*sat->c8*sat->c9+sat->y5 * sat->s8*sat->c9+sat->z5*sat->s9;
	sat->x8 =-sat->x5*sat->c8*sat->s9-sat->y5 * sat->s8*sat->s9+sat->z5*sat->c9;
	sat->y8 = sat->y5*sat->c8-sat->x5*sat->s8; 
	sat->alt = sat->r-R0;
	sat->el = atan(sat->z8/sqrt(sat->range2-sat->z8 * sat->z8))/DEG2RAD;
	sat->az = atan(sat->y8/sat->x8)/DEG2RAD;

	if ( sat->x8 < 0.0 )
		sat->az += 180.0;

	if ( sat->az < 0.0 )
		sat->az += 360.0;

		/*ma256=(int)256.0*phase;*/
	sat->range = sqrt(sat->range2); 
	sat->vel = 3.6*sqrt(3.98652e+14*((2.0/(sat->r*1000.0)) - 1.0/(sat->sma*1000.0)));
	sat->fp = 12756.33*acos(R0/sat->r);
	sat->lat = atan(sat->z/sqrt(sat->r*sat->r - sat->z*sat->z))/DEG2RAD;
	sat->lon =-atan(sat->y/sat->x)/DEG2RAD;

	if ( sat->x < 0.0 )
		sat->lon += 180.0;

	if ( sat->lon < 0.0 )
		sat->lon += 360.0;
	sat->range = sqrt(sat->range2);
		
	
}

