/***************************************************************************
                          pg_connection.c  -  description
                             -------------------
    begin                : Sat May 20 2000
    copyright            : (C) 2000 by Thierry Florac
    email                : tflorac@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "gpc_utils.h"
#include "support.h"

#include "libpq-fe.h"
#include "pg_connection.h"
#include "pg_connection_intf.h"
#include "pg_query.h"


/** This 'abstract' structure is used to store internal connection 
    parameters                                                             */
struct _pgConnection {
  PGconn   *_handle;      /* PostgreSQL connection handle                  */

  gboolean  _intrans;     /* Is a transaction already active ?             */

  gchar    *_hostname;    /* Postmaster hostname                           */
  gchar	   *_port;        /* Postmaster port                               */
  gchar	   *_database;    /* Postmaster Database Name                      */
  gchar	   *_username;    /* Postmaster Database user name                 */
  gchar    *_password;    /* Postmaster Database password                  */
};


/** Internal function */
void
pg_connection_dialog_set_connection (pgConnection *connection);


/** Allocate a new pgConnection structure
    Result: NULL if memory can't be allocated
            A valid pointer otherwise
            Default values can be fetched from package configuration file  */
pgConnection* 
pg_connection_new (void)
{
  pgConnection *connection;

  connection = g_new0 (pgConnection, 1);
  connection->_handle = NULL;
  connection->_hostname = gnome_config_get_string ("/" PACKAGE "/pgConnection/hostname");
  connection->_port = gnome_config_get_string ("/" PACKAGE "/pgConnection/port");
  connection->_database = gnome_config_get_string ("/" PACKAGE "/pgConnection/database");
  connection->_username = gnome_config_get_string ("/" PACKAGE "/pgConnection/username");
  if (gnome_config_get_bool ("/" PACKAGE "/options/store_password=true"))
    connection->_password = gnome_config_private_get_string ("/" PACKAGE "/pgConnection/password");
  if (!connection->_username)
    connection->_username = getenv ("USER");
  connection->_password = NULL;
  connection->_intrans = FALSE;
  return connection;
}


/** Allocate a new pgConnection structure
    Parameters are initialized using a connection dialog box
    Result: NULL if memory can't be allocated
            A valid pointer otherwise                                      */
pgConnection* 
pg_connection_new_with_dialog (void)
{
  pgConnection *connection;

  connection = pg_connection_new ();
  pg_connection_dialog_set_connection (connection);
  gnome_dialog_run_and_close (GNOME_DIALOG (create_pg_connection_dialog ()));
  if (modal_result == GPC_MODAL_RESULT_OK)
    return connection;
  else
  {
    pg_connection_free (connection);
    connection = NULL;
    return NULL;
  }
}


/** Allocate a new pgConnection structure with specified parameters
    Result: NULL if memory can't be allocated
            A valid pointer otherwise                                      */
pgConnection* 
pg_connection_new_with_params (gchar *hostname,
                               gchar *port,
                               gchar *database,
                               gchar *username,
                               gchar *password)
{
  pgConnection *connection;

  connection = pg_connection_new ();
  connection->_hostname = g_strdup (hostname);
  connection->_port = g_strdup (port);
  connection->_database = g_strdup (database);
  connection->_username = g_strdup (username);
  connection->_password = g_strdup (password);
  return connection;
}


/** Retrieve PostgreSQL handle of a specified pgConnection
    - @connection: the connection for which we need handle
    Result: NULL if connection is not defined
            A valid pointer otherwise                                      */
PGconn* 
pg_connection_get_handle (pgConnection *connection)
{
  if (!connection)
    return NULL;
  return connection->_handle;
}


/** These functions retrieve several connection properties
    - @connection: the connection for which we need properties
    Result: NULL if connection is not defined
            A valid char pointer otherwise                                 */
gchar* 
pg_connection_get_hostname (pgConnection *connection)
{
  if (!connection || !connection->_hostname)
    return NULL;
  return g_strdup (connection->_hostname);
}


gchar* 
pg_connection_get_port (pgConnection *connection)
{
  if (!connection || !connection->_port)
    return NULL;
  return g_strdup (connection->_port);
}


gchar* 
pg_connection_get_database (pgConnection *connection)
{
  if (!connection || !connection->_database)
    return NULL;
  return g_strdup (connection->_database);
}


gchar* 
pg_connection_get_username (pgConnection *connection)
{
  if (!connection || !connection->_username)
    return NULL;
  return g_strdup (connection->_username);
}


gchar* 
pg_connection_get_password (pgConnection *connection)
{
  if (!connection || !connection->_password)
    return NULL;
  return g_strdup (connection->_password);
}


/** These functions set several connection properties
    - @connection: the connection for which we set properties
    - @param: the new property value                                       */
void 
pg_connection_set_hostname (pgConnection *connection, 
			    gchar *hostname)
{
  if (connection) {
    if  (pg_connection_is_active (connection) && strcmp (connection->_hostname,hostname))
      pg_connection_close (connection);
    if (connection->_hostname)
      g_free (connection->_hostname);
    connection->_hostname = g_strdup (hostname);
  }
}


void 
pg_connection_set_port (pgConnection *connection, 
			gchar *port)
{
  if (connection) {
    if  (pg_connection_is_active (connection) && strcmp (connection->_port,port))
      pg_connection_close (connection);
    if (connection->_port)
      g_free (connection->_port);
    connection->_port = g_strdup (port);
  }
}


void 
pg_connection_set_database (pgConnection *connection, 
			    gchar *database)
{
  if (connection) {
    if (pg_connection_is_active (connection) && strcmp (connection->_database,database))
      pg_connection_close (connection);
    if (connection->_database)
      g_free (connection->_database);
    connection->_database = g_strdup (database);
  }
}


void 
pg_connection_set_username (pgConnection *connection, 
			    gchar *username)
{
  if (connection) {
    if (pg_connection_is_active (connection) && strcmp (connection->_username,username))
      pg_connection_close (connection);
    if (connection->_username)
      g_free (connection->_username);
    connection->_username = g_strdup (username);
  }
}


void 
pg_connection_set_password (pgConnection *connection, 
			    gchar *password)
{
  if (connection) {
    if (pg_connection_is_active (connection) && strcmp (connection->_password,password))
      pg_connection_close (connection);
    if (connection->_password)
      g_free (connection->_password);
    connection->_password = g_strdup (password);
  }
}


/** Open a previously allocated connection
    - @connection: the connection that may be activated
    Result: TRUE if connection was correctly activated
            FALSE if an error was encountered                              */
gboolean 
pg_connection_open (pgConnection *connection)
{
  gboolean result;

  if (!connection)
    return FALSE;
  if (pg_connection_is_active (connection))
    return TRUE;
  connection->_handle = PQsetdbLogin (connection->_hostname,
	                              connection->_port,
	                              NULL,
	                              NULL,
	                              connection->_database,
	                              connection->_username,
	                              connection->_password);
  result = pg_connection_is_active (connection);
  if (!result) {
    PQfinish (connection->_handle);
    connection->_handle = NULL;
  }
  return result;
}


/** Close a previously activated connection
    - @connection: the connection that may be closed
    Result: TRUE if connection was correctly closed
            FALSE if an error was encountered                              */
gboolean 
pg_connection_close (pgConnection *connection)
{
  if (!connection || !connection->_handle)
    return FALSE;
  if  (connection->_intrans)
    pg_connection_rollback (connection);
  PQfinish (connection->_handle);
  connection->_handle = NULL;
  return TRUE;
}


/** Check to know if the specified connection is active
    - @connection: the connection that may be checked
    Result: TRUE if connection is currently active
            FALSE otherwise                                                */
gboolean 
pg_connection_is_active (pgConnection *connection)
{
  if (!connection || !connection->_handle)
    return FALSE;
  return (PQstatus (connection->_handle) == CONNECTION_OK);
}


/** Start a new transaction on the specified connection
    - @connection: the connection on which we start transaction
    Result: TRUE if transaction was correctly started
            FALSE otherwise                                                */
gboolean 
pg_connection_start (pgConnection *connection)
{
  PGresult *res;
  gboolean result;

  if (!connection || !pg_connection_is_active (connection))
    return FALSE;
  if (connection->_intrans)
    return TRUE;
  res = PQexec (connection->_handle, "BEGIN");
  result = (res && (PQresultStatus (res) == PGRES_COMMAND_OK));
  PQclear (res);
  return result;
}


/** Check to know if an explicit transaction is actually started
    - @connection: the connection that may be checked
    Result: TRUE if a transaction is actually started
            FALSE otherwise                                                */
gboolean 
pg_connection_in_trans (pgConnection *connection)
{
  if (!connection || !pg_connection_is_active (connection))
    return FALSE;
  return (connection->_intrans);
}


/** Execute a command query on the specified connection
    - @connection: the connection on which we want to execute the command
    - @sql: SQL code of the command that may be executed
    Result: TRUE if the command was successfully processed
            FALSE otherwise                                                */
gboolean 
pg_connection_execute (pgConnection *connection, 
		       gchar *sql)
{
  PGresult *res;
  gboolean result;

  if (!connection || !pg_connection_is_active (connection))
    return FALSE;
  res = PQexec (connection->_handle, sql);
  result = (res && (PQresultStatus (res) == PGRES_COMMAND_OK));
  PQclear (res);
  return result;
}


/** Execute a select query on the specified connection
    - @connection: the connection on which we want to execute the command
    - @sql: SQL code of the 'SELECT' statement that may be executed
    Result: a pointer to a newly allocated pgQuery structure
            NULL if an error was encountered                               */
void* 
pg_connection_query (pgConnection *connection, 
		     gchar *sql)
{
  if (!connection || !pg_connection_is_active (connection))
    return NULL;
  return pg_query_new_with_sql (connection, sql);
}


/** Commit changes applied to the connection in an explicit transaction context
    - @connection: the connection on which we want to apply modifications
    Result: TRUE if changes were applied successfully
            FALSE if an error was encountered                              */
gboolean 
pg_connection_commit (pgConnection *connection)
{
  PGresult *res;
  gboolean result;

  if (!connection || !pg_connection_is_active (connection))
    return FALSE;
  res = PQexec (connection->_handle, "COMMIT");
  result = (res && (PQresultStatus (res) == PGRES_COMMAND_OK));
  PQclear (res);
  connection->_intrans = FALSE;
  return result;
}


/** Cancel changes applied to the connection in an explicit transaction context
    - @connection: the connection on which we want to rollback modifications
    Result: TRUE if changes were canceled successfully
            FALSE if an error was encountered                              */
gboolean 
pg_connection_rollback (pgConnection *connection)
{
  PGresult *res;
  gboolean result;

  if (!connection || !pg_connection_is_active (connection))
    return FALSE;
  res = PQexec (connection->_handle, "ROLLBACK");
  result = (res && (PQresultStatus (res) == PGRES_COMMAND_OK));
  PQclear (res);
  connection->_intrans = FALSE;
  return result;
}


/** Display a dialog box showing the last PostgreSQL error message
    - @connection: the connection for which we want to look at error messages
    - @start: message to display before PostgreSQL internal error message
    - @end: message to display after PostgreSQL internal error message     */
void 
pg_connection_show_error (pgConnection *connection, 
			  gchar *start, 
			  gchar *end)
{
  gchar *message;

  if (!connection || !connection->_handle)
    message = g_strdup_printf ("%s\n\n%s", start, end);
  else
    message = g_strdup_printf  ("%s\n\n  %s\n\n%s", start, PQerrorMessage  (connection->_handle), end);
  gnome_dialog_run_and_close (GNOME_DIALOG (gnome_error_dialog (message)));
  g_free (message);
}


/** Release memory and ressources allocated to a specified connection
    - @connection: the connection that may be released                    */
void 
pg_connection_free (pgConnection *connection)
{
  if (!connection)
    return;
  if (pg_connection_is_active (connection))
    pg_connection_close (connection);
  if (connection->_hostname) {
    gnome_config_set_string ("/" PACKAGE "/pgConnection/hostname", connection->_hostname);
    g_free (connection->_hostname);
    connection->_hostname = NULL;
  }
  if (connection->_port) {
    gnome_config_set_string ("/" PACKAGE "/pgConnection/port", connection->_port);
    g_free (connection->_port);
    connection->_port = NULL;
  }
  if (connection->_database) {
    gnome_config_set_string ("/" PACKAGE "/pgConnection/database", connection->_database);
    g_free (connection->_database);
    connection->_database = NULL;
  }
  if (connection->_username) {
    gnome_config_set_string ("/" PACKAGE "/pgConnection/username", connection->_username);
    g_free (connection->_username);
    connection->_username = NULL;
  }
  if (connection->_password) {
    if (gnome_config_get_bool ("/" PACKAGE "/options/store_password=true"))
      gnome_config_private_set_string ("/" PACKAGE "/pgConnection/password", connection->_password);
    else
      gnome_config_private_clean_key ("/" PACKAGE "/pgConnection/password");  
    g_free (connection->_password);
    connection->_password = NULL;
  }
  gnome_config_sync ();
  g_free (connection);
  connection = NULL;
}


/***************************************************************************/
/**     pg_connection_dialog callbacks implementation                      */
/***************************************************************************/

pgConnection* 
pgConnectionDialogConnection;


void
pg_connection_dialog_set_connection (pgConnection *connection)
{
  if (connection)
    pgConnectionDialogConnection = connection;
}


void
on_pg_connection_dialog_show (GtkWidget       *widget,
                              gpointer         user_data)
{
  GtkWidget   *dialog;
  GtkEntry    *entry;
  gchar	      *value;

  modal_result = GPC_MODAL_RESULT_UNKNOWN;
  if (pgConnectionDialogConnection) {
    dialog = widget;
    value = pg_connection_get_hostname(pgConnectionDialogConnection);
    if (value) {
      entry = GTK_ENTRY (lookup_widget (dialog, "hostname_entry"));
      gtk_entry_set_text (entry, value);
      g_free (value);
    }
    value = pg_connection_get_port (pgConnectionDialogConnection);
    if (value) {
      entry = GTK_ENTRY (lookup_widget (dialog, "portnumber_entry"));
      gtk_entry_set_text (entry, value);
      g_free (value);
    }
    value = pg_connection_get_database (pgConnectionDialogConnection);
    if (value) {
      entry = GTK_ENTRY (lookup_widget (dialog, "databasename_entry"));
      gtk_entry_set_text (entry, value);
      g_free (value);
    }
    value = pg_connection_get_username (pgConnectionDialogConnection);
    if (value) {
      entry = GTK_ENTRY (lookup_widget (dialog, "username_entry"));
      gtk_entry_set_text (entry, value);
      g_free (value);
    }
    value = pg_connection_get_password (pgConnectionDialogConnection);
    if (value) {
      entry = GTK_ENTRY (lookup_widget (dialog, "password_entry"));
      gtk_entry_set_text (entry, value);
      g_free (value);
    }
  }
}


void
on_pg_connection_dialog_ok_button_clicked (GtkButton       *button,
                                           gpointer         user_data)
{
  GtkWidget   *dialog;
  GtkEntry    *entry;

  if (pgConnectionDialogConnection) {
    dialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
    entry = GTK_ENTRY (lookup_widget (dialog, "hostname_entry"));
    pg_connection_set_hostname (pgConnectionDialogConnection, gtk_entry_get_text (entry));
    entry = GTK_ENTRY (lookup_widget (dialog, "portnumber_entry"));
    pg_connection_set_port (pgConnectionDialogConnection, gtk_entry_get_text (entry));
    entry = GTK_ENTRY (lookup_widget (dialog, "databasename_entry"));
    pg_connection_set_database (pgConnectionDialogConnection, gtk_entry_get_text (entry));
    entry = GTK_ENTRY (lookup_widget (dialog, "username_entry"));
    pg_connection_set_username (pgConnectionDialogConnection, gtk_entry_get_text (entry));
    entry = GTK_ENTRY (lookup_widget (dialog, "password_entry"));
    pg_connection_set_password (pgConnectionDialogConnection, gtk_entry_get_text (entry));
    gnome_dialog_close (GNOME_DIALOG (dialog));
  }
  modal_result = GPC_MODAL_RESULT_OK;
}


void
on_pg_connection_dialog_cancel_button_clicked (GtkButton       *button,
					       gpointer         user_data)
{
  gnome_dialog_close (GNOME_DIALOG (gtk_widget_get_toplevel (GTK_WIDGET (button))));
  modal_result = GPC_MODAL_RESULT_CANCEL;
}

