/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __ASSERT_MACROS_H__
#define __ASSERT_MACROS_H__

#include <check.h>
#include <math.h>
#include <glib.h>

#define ASSERT_EQUAL_INT(i1, i2) {                              \
    gchar *msg;                                                 \
    msg = g_strdup_printf("%d != %d", (int) i1, (int) i2);      \
    fail_unless(i1 == i2, msg);                                 \
    g_free(msg);                                                \
}

#define FLOAT_EPSILON (0.000001)
#define ASSERT_EQUAL_FLOAT(f1, f2) {                                       \
    gchar *msg;                                                            \
    msg = g_strdup_printf("%g != %g(epsilon: %g)", f1, f2, FLOAT_EPSILON); \
    fail_unless((FLOAT_EPSILON > fabsf(f1 - f2)), msg);                    \
    g_free(msg);                                                           \
}

#define ASSERT_EQUAL_POINTER(p1, p2) {          \
    gchar *msg;                                 \
    msg = g_strdup_printf("%p != %p", p1, p2);  \
    fail_unless(p1 == p2, msg);                 \
    g_free(msg);                                \
}

#define ASSERT_EQUAL_STRING(s1, s2) {                   \
    if (s1 == NULL) {                                   \
        fail(#s1 " is NULL");                           \
    }                                                   \
    else if (s2 == NULL) {                              \
        fail(#s2 " is NULL");                           \
    }                                                   \
    else {                                              \
        gchar *msg;                                     \
        msg = g_strdup_printf("'%s' != '%s'", s1, s2);  \
        fail_unless(strcmp(s1, s2) == 0, msg);          \
        g_free(msg);                                    \
    }                                                   \
}

#define ASSERT_EQUAL_ENUM(e1, e2) {             \
    fail_unless(e1 == e2, #e1 " != " #e2);      \
}

#define ASSERT_EQUAL_MEMORY(m1, m2, bytes) {                    \
    fail_unless(memcmp(m1, m2, bytes) == 0, #m1 " != " #m2);    \
}

#define ASSERT_NULL(exp) {				\
    fail_unless(exp == NULL, #exp " is not NULL");	\
}

#define ASSERT_NOT_NULL(exp) {			\
    fail_unless(exp != NULL, #exp " is NULL");	\
}

#define ASSERT_TRUE(exp) {			\
    fail_unless(exp, #exp " = FALSE");		\
}

#define ASSERT_FALSE(exp) {			\
    fail_unless(!exp, #exp " = TRUE");		\
}

#endif /* #ifndef __ASSERT_MACROS_H__ */
