/* 
    gy_utils.c: all utility/misc functions

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2, or (at
    your option) any later version.

    This program is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <termios.h>
#include <unistd.h>
#include <assert.h>
#include <readline/readline.h>
#include <guile/gh.h>
#include <sys/stat.h>
#include <time.h>

#include <config.h>
#include "gy-utils.h"
#include "gnuyahoo.h"
#include "interpreter.h"

tcflag_t old_lflag;
cc_t old_vtime;

struct termios term;

void
get_terminal_attributes (void)
{
  if (tcgetattr (STDIN_FILENO, &term) < 0)
    {
      perror ("tcgetattr()");
      exit (EXIT_FAILURE);
    }
  old_lflag = term.c_lflag;
  old_vtime = term.c_cc[VTIME];
}

void
set_terminal_attributes (void)
{
  term.c_lflag = old_lflag;
  term.c_cc[VTIME] = old_vtime;
  if (tcsetattr (STDIN_FILENO, TCSANOW, &term) < 0)
    {
      perror ("tcsetattr()");
      exit (EXIT_FAILURE);
    }
}

void
bell (void)
{
  if (get_bell_state ())
#if defined (HAVE_RL_DING)
    rl_ding ();
#else
#if defined (HAVE_DING)
    ding ();
#else
    /* don't worry, 'else' will never happen. configure script exits if both
       functions are missing */
    assert (0);
#endif
#endif
}

/* Trims the leading and trailing white spaces of a string  */
char *
stripwhite (char *string)
{
  register char *s, *t;

  for (s = string; whitespace (*s); s++)
    ;

  if (*s == 0)
    return s;

  t = s + strlen (s) - 1;
  while (t > s && whitespace (*t))
    t--;
  *++t = '\0';

  return s;
}


char *
get_token (char **line)
{
  char *command;
  while (1)
    {
      command = (char *) strsep (line, " ");
      if (!command)
	break;
      if (*(command))
	break;
    }
  return command;
}

char *
get_token_with_strdelim (char **line, char *delim)
{
  char *token = (char *) NULL;
  char *tmp_str = (char *) NULL;
  int token_length = 0;

  if (!(*line) || !delim)
    return token;

  tmp_str = strstr (*line, delim);
  if (tmp_str)
    token_length = strlen (*line) - strlen (tmp_str);
  else
    token_length = strlen (*line);

  token = (char *) malloc (sizeof (char) * (token_length + 1));
  strncpy (token, *line, token_length);
  token[token_length] = (char) NULL;
  *line = tmp_str;

  return token;
}

char *
get_token_with_strdelim_i (char **line, char *delim)
{
  char *token = (char *) NULL;
  char *tmp_str = (char *) NULL;
  int token_length = 0;

  if (!(*line) || !delim)
    return token;

  tmp_str = strcasestr (*line, delim);
  if (tmp_str)
    token_length = strlen (*line) - strlen (tmp_str);
  else
    token_length = strlen (*line);

  token = (char *) malloc (sizeof (char) * (token_length + 1));
  strncpy (token, *line, token_length);
  token[token_length] = (char) NULL;
  *line = tmp_str;

  return token;
}

char *
filter_message (char *message)
{
  char *str_strip = (char *) NULL;
  char *str_temp = (char *) NULL;
  int message_length = 0, term_flag = 0;

  str_strip = (char *) malloc (strlen (message) + 1);
  str_temp = str_strip;
  message_length = strlen (message);

  while (*message)
    {
      if (*message == 6) // terminate if ^F found
	{
	  *str_strip = '\0';
	  term_flag = !term_flag;
	  break;
	}
      if (*message == 27) // skip all ^[[*m
	{
	  while (*message != 109)
	    message++;
	  message++;
	}
      if (*message != 6 && *message != 27) // skip <font> tag
	{
	  if (!strncmp (message, "<font ", strlen ("<font ")))
	    message = strstr (message, ">") + 1;
	  *str_strip++ = *message++;
	}
    }

  if (!term_flag)
    str_temp [message_length] = '\0';
  return str_temp;
}

char *
get_utc_time (char *time_secs)
{
  time_t t;
  struct tm *utc_time;
  char *time_fstr;

  time_fstr = (char *) malloc (UTC_TIME_STR_SIZE);
  t = atoi (time_secs);
  utc_time = gmtime (&t);
  if (strftime (time_fstr, UTC_TIME_STR_SIZE, UTC_TIME_STR_FORMAT, utc_time) <= 0)
    time_fstr = time_secs;

  return time_fstr;
}

void
show_version (void)
{
#ifdef VERSION
  printf ("GNU Messenger %s for Yahoo\n", VERSION);
#else
  printf ("GNU Messenger for Yahoo%s\n");
#endif
  //  FIXME: we haven't still signed the copyright papers
  //  printf("Copyright (C) 2001 Free Software Foundation, Inc.\n");

  fputs ("GNU Yahoo Messenger comes with ABSOLUTELY NO WARRANTY.\n"
	 "You may redistribute copies of GNU Yahoo\n"
	 "under the terms of the GNU General Public License.\n"
	 "For more information about these matters, see the file"
	 "named COPYING.\n", stdout);
}

char *
get_home_directory (void)
{
  struct passwd *current_passwd;
  uid_t user_id;
  setpwent ();
  user_id = getuid ();
  while ((current_passwd = getpwent ()))
    {
      if (current_passwd->pw_uid == user_id)
	return current_passwd->pw_dir;
    }
  return NULL;
}

char *
get_config_directory (void)
{
  char *config_directory;
  int length =
    strlen (get_home_directory ()) + strlen (GY_CONFIG_DIRECTORY) + 2;

  config_directory = (char *) calloc (length, sizeof (char));
  sprintf (config_directory, "%s/" GY_CONFIG_DIRECTORY,
	   get_home_directory ());

  return config_directory;
}

char *
get_config_filename (void)
{
  char *config_filename;
  int length = strlen (get_home_directory ()) + strlen (GY_CONFIG_FILE) + 2;

  config_filename = (char *) calloc (length, sizeof (char));
  sprintf (config_filename, "%s/" GY_CONFIG_FILE, get_home_directory ());
  return config_filename;
}

char *
get_download_filename (void)
{
  char *download_filename = NULL;
  int length =
    strlen (get_config_directory ()) + strlen (GY_DOWNLOADS_URL_FILE) + 2;

  download_filename = (char *) calloc (length, sizeof (char));
  sprintf (download_filename, "%s/" GY_DOWNLOADS_URL_FILE,
	   get_config_directory ());
  return download_filename;
}

char *
get_global_extensions_directory (void)
{
  char *global_extensions_directory;
  int length = strlen (GY_GLOBAL_EXTENSIONS_DIRECTORY) + 2;

  global_extensions_directory = (char *) calloc (length, sizeof (char));
  sprintf (global_extensions_directory, "%s/",
	   GY_GLOBAL_EXTENSIONS_DIRECTORY);

  return global_extensions_directory;
}

char *
get_local_extensions_directory (void)
{
  char *local_extensions_directory;
  int length = strlen (get_home_directory ()) +
    strlen (GY_LOCAL_EXTENSIONS_DIRECTORY) + 2;

  local_extensions_directory = (char *) calloc (length, sizeof (char));
  sprintf (local_extensions_directory, "%s/%s/", get_home_directory (),
	   GY_LOCAL_EXTENSIONS_DIRECTORY);

  return local_extensions_directory;
}

void
gy_load (char *filename)
{
  char *extension_filepath;
  struct stat buf;
  int length;

  if (stat (filename, &buf) == 0)
    {
      gh_eval_file_with_standard_handler (filename);
      return;
    }

  length = strlen (get_local_extensions_directory ()) + strlen (filename) + 1;
  extension_filepath = (char *) calloc (length, sizeof (char));
  sprintf (extension_filepath, "%s%s", get_local_extensions_directory (),
	   filename);
  if (stat (extension_filepath, &buf) == 0)
    {
      gh_eval_file_with_standard_handler (extension_filepath);
      return;
    }
  free (extension_filepath);

  length =
    strlen (get_global_extensions_directory ()) + strlen (filename) + 1;
  extension_filepath = (char *) calloc (length, sizeof (char));
  sprintf (extension_filepath, "%s%s", get_global_extensions_directory (),
	   filename);
  if (stat (extension_filepath, &buf) == 0)
    {
      gh_eval_file_with_standard_handler (extension_filepath);
      return;
    }
}
