/* 
   gnuyahoo.c: main initialization file
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <guile/gh.h>
#include <getopt.h>
#include <config.h>

#include "messenger.h"
#include "extension.h"
#include "gy-utils.h"
#include "yahoo-wrapper.h"
#include "interpreter.h"
#include "gnuyahoo.h"

/* private members REALLY. extern'ing these variables are strongly
   discouraged. instead use exported methods to access these variables
*/
char *default_login_id;
char *default_password;
char *default_prompt = GY_PROMPT_STRING;
int setup_mode = 0;

typedef void (*sighandler_t) (int);

static struct option long_options[] = {
  {"user", 1, 0, 0},
  {"status", 1, 0, 0},
  {"version", 0, 0, 0},
  {"help", 0, 0, 0},
  {0, 0, 0, 0}
};

char options_string[] = "u:vhs:";

char *
get_default_login_id (void)
{
  return default_login_id;
}

void
set_default_login_id (char *default_login_id_value)
{
  if (default_login_id == NULL)
    default_login_id = default_login_id_value;
}

char *
get_default_password (void)
{
  return default_password;
}

void
set_default_password (char *default_password_value)
{
  if (default_password == NULL)
    default_password = default_password_value;
}

char *
get_default_prompt (void)
{
  return default_prompt;
}

void
set_default_prompt (char *default_prompt_value)
{
  if (default_prompt == NULL)
    default_prompt = default_prompt_value;
}

int
get_setup_mode (void)
{
  return setup_mode;
}

void
set_setup_mode (int setup_mode_value)
{
  setup_mode = setup_mode_value;
}

// for --usage
void
show_usage (void)
{
  fputs ("This is GNU Yahooo\n\n" "\t " PACKAGE " [options]\n\n", stdout);
}

// for --help
void
show_options (void)
{
  show_usage ();
  fputs ("Options:\n"
	 "--user=YAHOOID       Yahoo login name\n"
	 "--status=STATUS      Default login status number\n"
	 "--version            Version information\n"
	 "--help               Help on options\n\n"
	 "For more information, type \"?help\" from within " PACKAGE ", "
	 "or consult the \n" PACKAGE
	 " manual (available as on-line info or a " "printed manual).\n"
	 "Report bugs to \"" GY_BUG_REPORT_EMAIL "\"\n\n", stdout);
}


// parse messenger specific options
void
parse_options (int argc, char **argv)
{
  int option_char;
  int option_index = 0;
  char *status_str = NULL;
  char *user;

  while (1)
    {
      option_char =
	getopt_long (argc, argv, options_string, long_options, &option_index);

      if (option_char == -1)
	break;

      switch (option_char)
	{
	case 0:
	  if (strcmp (long_options[option_index].name, "user") == 0)
	    {
	      // FIXME: gives a "implicit declaration" warning even if "string.h" is included !!
	      // user = (char *) strndup ((const char *) optarg, USERNAME_LENGTH * sizeof (char));

	      user = strdup (optarg);
	      if (!user)
		{
		  fprintf (stderr, "Insufficient memory\n");
		  exit (EXIT_FAILURE);
		}
	      set_default_login_id (user);
	    }
	  if (strcmp (long_options[option_index].name, "status") == 0)
	    {
	      // FIXME: gives a "implicit declaration" warning even if "string.h" is included !!
	      // status_str = (char *) strndup (optarg, 3);
	      status_str = (char *) strdup (optarg);
	      if (!status_str)
		{
		  fputs ("Insufficient memory\n", stderr);
		  exit (EXIT_FAILURE);
		}
	      set_current_status (atoi (status_str));
	    }
	  if (strcmp (long_options[option_index].name, "version") == 0)
	    {
	      show_version ();
	      exit (EXIT_SUCCESS);
	    }
	  if (strcmp (long_options[option_index].name, "help") == 0)
	    {
	      show_options ();
	      exit (EXIT_SUCCESS);
	    }
	  break;

	case 'u':
	  // FIXME: gives a "implicit declaration" warning even if "string.h" is included !!
	  // user = (char *) strndup (optarg, USERNAME_LENGTH * sizeof (char));
	  user = strdup (optarg);
	  if (!user)
	    {
	      fprintf (stderr, "Insufficient memory\n");
	      exit (EXIT_FAILURE);
	    }
	  set_default_login_id (user);
	  break;

	case 'v':
	  show_version ();
	  exit (EXIT_SUCCESS);

	case 'h':
	  show_options ();
	  exit (EXIT_SUCCESS);

	case 's':
	  status_str = strdup (optarg);
	  if (!status_str)
	    {
	      fputs ("Insufficient memory\n", stderr);
	      exit (EXIT_FAILURE);
	    }
	  set_current_status (atoi (status_str));
	  break;

	case '?':
	  fputs ("Use `" PACKAGE
		 " --help' for a complete list of options\n", stderr);
	  exit (EXIT_FAILURE);

	default:
	  fprintf (stderr, "?? getopt returned character code 0%o ??\n",
		   option_char);
	}
    }

  if (optind < argc)
    {
      fputs ("Use `" PACKAGE " --help' for a complete list of"
	     " options\n", stderr);

      exit (EXIT_FAILURE);
    }
}

void
running_for_first_time (void)
{
  struct stat *config_file_stat = NULL;
  int config_file_fd;
  char *config_directory = get_config_directory ();
  char *config_filename = get_config_filename ();

  if (mkdir (config_directory, GY_CONFIG_DIRECTORY_MODE) != 0)
    {
      // mkdir  succeeded
      // means config dir is there already
      return;
    }
  else
    {
      printf ("\nRunning " PACKAGE " for first time!!\n");
      printf ("[%s] created\n", config_directory);
    }

  // see if config file is there
  if (stat (config_filename, config_file_stat) != 0)
    {
      // config file missing, so touch it
      config_file_fd = open (config_filename, O_CREAT);
      if (config_file_fd == -1)
	{
	  perror ("open[config_file]");
	  exit (EXIT_FAILURE);
	}
      if (fchmod (config_file_fd, GY_CONFIG_DIRECTORY_MODE) != 0)
	{
	  perror ("fchmod[" GY_CONFIG_FILE "]");
	  exit (EXIT_FAILURE);
	}
      close (config_file_fd);
      printf ("[%s] created\n\n", config_filename);
    }

  // this is required for delayed, higher level setup procedures
  // that needs to be executed after guile initialization
  // of after sign-in.
  set_setup_mode (1);

  free (config_directory);
  free (config_filename);
}


void
inner_main (int argc, char **argv)
{
  /* your guile environment is ready :)
     you are free to eval any guile code */

  /* >>> GUILE INITIALIZATION STARTS */

  /* interface our gnu messenger env to guile by hooks and new
     procedures */
  guile_env_init ();

  /* gy guile env is ready. you can now call gnu messenger routines
     from with in scheme */

  /* load and evaluate init.scm from well known path */
  gy_load (GY_INIT_FILE);

  // load and evaluate your config file, which contains scheme code
  gh_eval_file_with_standard_handler (get_config_filename ());

  /* <<< GUILE INITIALIZATION ENDS */


  // ACTUAL MAIN STARTS FROM HERE
  messenger_main (argc, argv);

  /* control can never reach here. messenger_main calls exits
     internally */
}

int
main (int argc, char *argv[])
{
  running_for_first_time ();

  get_terminal_attributes ();

  /* command line arguments overrides all default settings, including
     the config file */
  parse_options (argc, argv);

  gh_enter (argc, argv, inner_main);

  /* control can never reach here. gh_enter calls exit internally */
  return 0;
}
