/* -*- mode:ObjC -*-
   XGContext - Drawing context using the XGPS Library.

   Copyright (C) 1998,1999 Free Software Foundation, Inc.

   Written by:  Adam Fedor <fedor@boulder.colorado.edu>
   Date: Nov 1998
   
   This file is part of the GNU Objective C User Interface Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */

#include "config.h"
#include "gnustep/xgps/XGContext.h"
#include "gnustep/xgps/XGGState.h"
#include "SharedX/xrtools.h"
#include "SharedX/XGInputServer.h"
#include <AppKit/AppKitExceptions.h>
#include <AppKit/NSAffineTransform.h>
#include <AppKit/NSColor.h>
#include <AppKit/NSView.h>
#include <AppKit/NSWindow.h>
#include <Foundation/NSException.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSData.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSString.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSDebug.h>

#include "gnustep/xgps/XGContextPrivate.h"
#include "gnustep/xgps/XGStreamContext.h"

#if HAVE_XFT
#include "XftFontInfo.h"
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>


#define GSI_ARRAY_TYPES       GSUNION_OBJ

#if     GS_WITH_GC == 0
#define GSI_ARRAY_RELEASE(A, X)    [(X).obj release]
#define GSI_ARRAY_RETAIN(A, X)     [(X).obj retain]
#else
#define GSI_ARRAY_RELEASE(A, X)
#define GSI_ARRAY_RETAIN(A, X)     (X).obj
#endif

#ifdef GSIArray
#undef GSIArray
#endif
#include <base/GSIArray.h>

/* Error macros */
#define CHECK_NULL_OUTPUT(outvar) \
  if (outvar == NULL) \
    DPS_ERROR(DPSnulloutput, @"NULL output variable specified")

#define CHECK_INVALID_FONT(ident) \
  if (ident >= [fontid count]) \
    DPS_ERROR(DPSinvalidfont, @"Cannot find indicated font")

#define CHECK_STACK_UNDERFLOW(stack) \
  if (GSIArrayCount((GSIArray)stack) == 0) \
    DPS_ERROR(DPSstackunderflow, @"Attempt to pop from empty stack")

#if 0
#define CHECK_TYPECHECK(obj, kind) \
  if ([kind class] != Nil && !GSObjCIsKindOf(GSObjCClass(obj), [kind class])) \
    DPS_ERROR(DPStypecheck, @"Invalid object")
#else
#define CHECK_TYPECHECK(obj,kind)
#endif

#define ctxt_pop(object, stack, kind) \
  do { \
    CHECK_STACK_UNDERFLOW(stack); \
    object = (GSIArrayLastItem((GSIArray)stack)).obj; \
    CHECK_TYPECHECK(object, kind); \
    AUTORELEASE(RETAIN(object)); \
    GSIArrayRemoveLastItem((GSIArray)stack); \
  } while (0)

#define ctxt_push(object, stack) \
  GSIArrayAddItem((GSIArray)stack, (GSIArrayItem)object)


static NSObject *mark = nil;
static Class NSAffineTransform_class = Nil;
static NSMutableDictionary *globalfontdir = nil;
// FIXME This is used nowhere!
Atom			WM_STATE;

extern int XGErrorHandler(Display *display, XErrorEvent *err);

@interface XGContext (Window)
- (void) _setupRootWindow;
@end

@interface XGContext (Private)
- (void) setupRunLoopInputSourcesForMode: (NSString*)mode; 
@end

#define XDPY (((RContext *)context)->dpy)
#define XSCR (((RContext *)context)->screen_number)

@interface GSBackend : XGContext
{}
@end

@implementation GSBackend
@end

/**
   <unit>
   <heading>XGContext</heading>
   <p>
   This class is a reasonable attempt at mapping PostScript operators to an
   Xlib library implementation. Don't even begin to think that this
   is a full PostScript implementation, however. Many, many operators are
   not implemented or do not work as expected. Before using an operator,
   check to make sure it is implemented, otherwise the opstack may 
   be screwed up and nothing else will work!. No errors are returned
   for non-implemneted operators.

   The documentation below mostly describes methods that are specific to 
   this backend and wouldn't necessarily be used in other backends. The methods
   that this class does implement that would need to be in every backend are
   the methods of its NSGraphicContext superclass. See the documentation
   for NSGraphicContext for more information.
   </p>
   </unit>
*/
@implementation XGContext 

+ (void) initialize
{
  if (self == [XGContext class])
    {
      globalfontdir = [[NSMutableDictionary alloc] init];
      mark = [NSObject new];
      NSAffineTransform_class = [NSAffineTransform class];
      [self setVersion: 1];
    }
}

/* Initialize AppKit backend */
+ (void)initializeBackend
{
  Class fontClass = Nil;

  NSDebugLog(@"Initializing GNUstep GUI X/GPS backend.\n");

  [NSGraphicsContext setDefaultContextClass: [XGContext class]];
  [GSFontEnumerator setDefaultClass: [XGFontEnumerator class]];

#if HAVE_XFT
  if ([[NSUserDefaults standardUserDefaults] boolForKey: @"GSFontAntiAlias"])
    {
      fontClass = [XftFontInfo class];
    }
#endif
  if (fontClass == Nil)
    {
      fontClass = [XGFontInfo class];
    }
  [GSFontInfo setDefaultClass: fontClass];
}

/**
   Returns a pointer to the current X-Windows display variable for
   the current context.
*/
+ (Display*) currentXDisplay
{
  return [(XGContext*)GSCurrentContext() xDisplay];
}

- (RContextAttributes *) _getXDefaults
{
  int dummy;
  RContextAttributes *attribs;

  NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
  attribs = (RContextAttributes *)malloc(sizeof(RContextAttributes));

  attribs->flags = 0;
  if ([defaults boolForKey: @"NSDefaultVisual"])
    attribs->flags |= RC_DefaultVisual;
  if ((dummy = [defaults integerForKey: @"NSDefaultVisual"]))
    {
      attribs->flags |= RC_VisualID;
      attribs->visualid = dummy;
    }
  if ((dummy = [defaults integerForKey: @"NSColorsPerChannel"]))
    {
      attribs->flags |= RC_ColorsPerChannel;
      attribs->colors_per_channel = dummy;
    }

  return attribs;
}

- _initXContext
{
  Display		*dpy;
  int			screen_number;
  NSString		*display_name;
  RContext		*rcontext;
  RContextAttributes	*attribs;
  XColor		testColor;
  unsigned char		r, g, b;
  
  display_name = [context_info objectForKey: @"DisplayName"];
  if (display_name == nil)
    {
      NSString	*host;
      NSString	*dnum = @"0.0";

      host = [[NSUserDefaults standardUserDefaults] stringForKey: @"NSHost"];
      if (host == nil)
	{
	  NSString	*d = [[[NSProcessInfo processInfo] environment]
	    objectForKey: @"DISPLAY"];

	  if (d == nil)
	    {
	      host = @"";
	    }
	  else
	    {
	      if ([d hasPrefix: @":"] == YES)
		{
		  host = @"";	// local host
		}
	      else
		{
		  NSArray	*a = [d componentsSeparatedByString: @":"];

		  if ([a count] != 2)
		    {
		      NSLog(@"X DISPLAY environment variable has bad format"
			@" assuming local X server (DISPLAY=:0.0)");
		      host = @"";
		    }
		  else
		    {
		      host = [a objectAtIndex: 0];
		      dnum = [a lastObject];
		      if ([dnum isEqual: @"0"] == NO
			&& [dnum hasPrefix: @"0."] == NO)
			{
			  NSLog(@"Only one display per host fully supported.");
			}
		    }
		}
	      if ([host isEqual: @""] == NO)
		{
		  /**
		   * If we are using the DISPLAY environment variable to
		   * determine where to display, set the NSHost default
		   * so that other parts of the system know where we are
		   * displaying.
		   */
		  [[NSUserDefaults standardUserDefaults] registerDefaults:
		    [NSDictionary dictionaryWithObject: host
						forKey: @"NSHost"]];
		}
	    }
	}
      if ([host isEqual: @""] == NO)
	{
	  /**
	   * If the NSHost default told us to display somewhere, we need
	   * to generate a display name for X from the host name and the
	   * default display and screen numbers (zero).
	   */
	  display_name = [NSString stringWithFormat: @"%@:%@", host, dnum];
	}
    }

  if (display_name)
    {
      dpy = XOpenDisplay([display_name cString]);
    }
  else
    { 
      dpy = XOpenDisplay(NULL);
      display_name = [NSString stringWithCString: XDisplayName(NULL)];
    }

  /* Use the fact that the screen number is specified like an extension
     e.g. hostname:0.1 */
  screen_number = [[display_name pathExtension] intValue];

  if (dpy == NULL)
    {
      char *dname = XDisplayName([display_name cString]);
      DPS_FATAL(DPSconfigurationerror, @"Unable to connect to X Server `%s'", 
		dname);
    }
  else
    NSDebugLog(@"Opened display %@", display_name);

  /* Get the visual information */
  attribs = NULL;
  //attribs = [self _getXDefaults];
  rcontext = RCreateContext(dpy, screen_number, attribs);
  context  = (void *)rcontext;

  /*
   * If we have shared memory available, only use it when the XGPS-Shm
   * default is set to YES
   */
  if (rcontext->attribs->use_shared_memory == True
    && [[NSUserDefaults standardUserDefaults] boolForKey: @"XGPS-Shm"] != YES)
    rcontext->attribs->use_shared_memory = False;

  /*
   *	Crude tests to see if we can accelerate creation of pixels from
   *	8-bit red, green and blue color values.
   */
  if (rcontext->depth == 12 || rcontext->depth == 16)
    {
      drawMechanism = XGDM_FAST16;
      r = 8;
      g = 9;
      b = 7;
      testColor.pixel = (((r << 5) + g) << 6) + b;
      XQueryColor(rcontext->dpy, rcontext->cmap, &testColor);
      if (((testColor.red >> 11) != r)
	|| ((testColor.green >> 11) != g)
	|| ((testColor.blue >> 11) != b))
	{
	  NSLog(@"WARNING - XGContext is unable to use the "
	    @"fast algorithm for writing to a 16-bit display on "
	    @"this host - perhaps you'd like to adjust the code "
	    @"to work ... and submit a patch.");
	  drawMechanism = XGDM_PORTABLE;
	}
    }
  else if (rcontext->depth == 15)
    {
      drawMechanism = XGDM_FAST15;
      r = 8;
      g = 9;
      b = 7;
      testColor.pixel = (((r << 5) + g) << 5) + b;
      XQueryColor(rcontext->dpy, rcontext->cmap, &testColor);
      if (((testColor.red >> 11) != r)
	|| ((testColor.green >> 11) != g)
	|| ((testColor.blue >> 11) != b))
	{
	  NSLog(@"WARNING - XGContext is unable to use the "
	    @"fast algorithm for writing to a 15-bit display on "
	    @"this host - perhaps you'd like to adjust the code "
	    @"to work ... and submit a patch.");
	  drawMechanism = XGDM_PORTABLE;
	}
    }
  else if (rcontext->depth == 24 || rcontext->depth == 32)
    {
      drawMechanism = XGDM_FAST32;
      r = 32;
      g = 33;
      b = 31;
      testColor.pixel = (((r << 8) + g) << 8) + b;
      XQueryColor(rcontext->dpy, rcontext->cmap, &testColor);
      if (((testColor.red >> 8) == r)
        && ((testColor.green >> 8) == g)
        && ((testColor.blue >> 8) == b))
	{
	  drawMechanism = XGDM_FAST32;
	}
      else if (((testColor.red >> 8) == b)
	&& ((testColor.green >> 8) == g)
	&& ((testColor.blue >> 8) == r))
	{
	  drawMechanism = XGDM_FAST32_BGR;
	}
      else
	{
	  NSLog(@"WARNING - XGContext is unable to use the "
	    @"fast algorithm for writing to a 32-bit display on "
	    @"this host - perhaps you'd like to adjust the code "
	    @"to work ... and submit a patch.");
	  drawMechanism = XGDM_PORTABLE;
	}
    }
  else
    {
      NSLog(@"WARNING - XGContext is unable to use a "
	@"fast algorithm for writing to the display on "
	@"this host - perhaps you'd like to adjust the code "
	@"to work ... and submit a patch.");
      drawMechanism = XGDM_PORTABLE;
    }

  XSetErrorHandler(XGErrorHandler);

  if (GSDebugSet(@"XSynchronize") == YES)
    XSynchronize(dpy, True);

  [self _setupRootWindow];
  inputServer = [[XIMInputServer allocWithZone: [self zone]] 
		  initWithDelegate: nil display: dpy name: @"XIM"];
  return self;
}

/**
   Opens the X display (using a helper method) and sets up basic
   display mechanisms, such as visuals and colormaps.
*/
- (id) initWithContextInfo: (NSDictionary *)info
{
  NSString *contextType;
  NSZone   *z = [self zone];

  contextType = [info objectForKey: 
		  NSGraphicsContextRepresentationFormatAttributeName];
  if (contextType && [contextType isEqual: NSGraphicsContextPSFormat])
    {
      /* Don't call self, which expects the display to be initialized */
      [super dealloc];
      return [[XGStreamContext allocWithZone: z] initWithContextInfo: info];
    }

  /* Initialize lists and stacks */
  opstack =  NSZoneMalloc(z, sizeof(GSIArray_t));
  GSIArrayInitWithZoneAndCapacity((GSIArray)opstack, z, 2);
  gstack =  NSZoneMalloc(z, sizeof(GSIArray_t));
  GSIArrayInitWithZoneAndCapacity((GSIArray)gstack, z, 2);
  ulist  = [[NSMutableArray allocWithZone: z] initWithCapacity: 32];

  // Set up font cache
  fontid = [NSMutableArray arrayWithCapacity: 32];

  [self _initXContext];

  /* Create a default gstate */
  gstate = [[XGGState allocWithZone: z] initWithDrawContext: self];

  [super initWithContextInfo: info];

  [self setupRunLoopInputSourcesForMode: NSDefaultRunLoopMode]; 
  [self setupRunLoopInputSourcesForMode: NSConnectionReplyMode]; 
  [self setupRunLoopInputSourcesForMode: NSModalPanelRunLoopMode]; 
  [self setupRunLoopInputSourcesForMode: NSEventTrackingRunLoopMode]; 

  WM_STATE = XInternAtom(XDPY, "WM_STATE", False);
  return self;
}

/**
   Closes all X resources, the X display and dealloc other ivars.
*/
- (void) dealloc
{
  NSDebugLog(@"Destroying X Context");
  GSIArrayEmpty((GSIArray)opstack);
  NSZoneFree([self zone], opstack);
  GSIArrayEmpty((GSIArray)gstack);
  NSZoneFree([self zone], gstack);
  DESTROY(ulist);
  DESTROY(inputServer);
  [self _destroyContextWindows];
  XCloseDisplay(XDPY);
  [super dealloc];
}

/**
   Returns the XGDrawMechanism, which roughly describes the depth of
   the screen and how pixels should be drawn to the screen for maximum
   speed.
*/
- (XGDrawMechanism) drawMechanism
{
  return drawMechanism;
}

/**
   Returns YES, since this is a display context.
*/
- (BOOL)isDrawingToScreen
{
  return YES;
}

/**
   Returns a pointer to a structure which describes aspects of the
   X windows display 
*/
- (void *) xrContext
{
  return context;
}

/**
   Returns the current XGGState object
*/
- (XGGState *) xrCurrentGState
{
  return gstate;
}

/*
  Returns a pointer to the X windows display variable
*/
- (Display *) xDisplay
{
  return XDPY;
}

- (void) setXDisplay: (Display *)xdisplay
{
  NSLog(@"Warning: resetting X display\n");
  XDPY = xdisplay;
  /* FIXME: Should update visualinfo, colormaps, etc? */
}

/**
   Returns the root window of the display 
*/
- (Window) xDisplayRootWindow
{
  return RootWindow(XDPY, XSCR);
}

/**
   Returns the application root window, which is used for many things
   such as window hints 
*/
- (Window) xAppRootWindow
{
  return generic.appRootWindow;
}

/**
   Returns the closest color in the current colormap to the indicated
   X color
*/
- (XColor)xColorFromColor: (XColor)color
{
  Status ret;
  RColor rcolor;
  Colormap colormap = XDefaultColormap(XDPY, XSCR);
  XAllocColor(XDPY, colormap, &color);
  rcolor.red   = color.red / 256;
  rcolor.green = color.green / 256;
  rcolor.blue  = color.blue / 256;
  ret = RGetClosestXColor((RContext *)context, &rcolor, &color);
  if (ret == False)
    NSLog(@"Failed to alloc color (%d,%d,%d)\n",
          (int)rcolor.red, (int)rcolor.green, (int)rcolor.blue);
  return color;
}

- (void) flush
{
  XFlush(XDPY);
}

@end

/* Common graphics functions */
@implementation XGContext (NSGraphics) 
//
// Read the Color at a Screen Position
//
- (NSColor *) NSReadPixel: (NSPoint) location
{
  return nil;
}


//
// Copy an image
//
- (void) NSCopyBitmapFromGState: (int) srcGstate:  (NSRect) srcRect 
			       : (NSRect) destRect
{
}

- (void) NSCopyBits: (int) srcGstate : (NSRect) srcRect : (NSPoint) destPoint
{
  float x, y, w, h;

  x = NSMinX(srcRect);
  y = NSMinY(srcRect);
  w = NSWidth(srcRect);
  h = NSHeight(srcRect);

  DPScomposite(self, x, y, w, h, srcGstate, destPoint.x, destPoint.y,
	       NSCompositeCopy);
}

//
// Imaging Functions
//
- (void) NSDrawBitmap: (NSRect) rect : (int) pixelsWide : (int) pixelsHigh
		     : (int) bitsPerSample : (int) samplesPerPixel 
		     : (int) bitsPerPixel : (int) bytesPerRow : (BOOL) isPlanar
		     : (BOOL) hasAlpha : (NSString *) colorSpaceName
		     : (const unsigned char *const [5]) data
{
  NSAffineTransform *trans;
  NSSize scale;

  // Compute the transformation matrix
  scale = NSMakeSize(NSWidth(rect) / pixelsWide, 
		     NSHeight(rect) / pixelsHigh);
  trans = [NSAffineTransform transform];
  [trans translateToPoint: rect.origin];
  [trans scaleBy: scale.width : scale.height];

  /* This does essentially what the DPS...image operators do, so
     as to avoid an extra method call */
  [[self xrCurrentGState] DPSimage: trans 
			  : pixelsWide : pixelsHigh 
			  : bitsPerSample : samplesPerPixel 
			  : bitsPerPixel : bytesPerRow 
			  : isPlanar
			  : hasAlpha : colorSpaceName
			  : data];
}

/* Context helper wraps */
static unsigned int unique_index = 0;

- (unsigned int) GSWDefineAsUserObj
{
  if(gstate)
    {
      DPSsendint(self, ++unique_index);
      DPSexch(self);
      DPSdefineuserobject(self);
      return unique_index;
    }
  return 0;
}

- (void) GSWSetViewIsFlipped: (BOOL) flipped
{
  if(gstate)
    gstate->viewIsFlipped = flipped;
}

- (BOOL) GSWViewIsFlipped
{
  if(gstate)
    return gstate->viewIsFlipped;
  return NO;
}

@end

@implementation XGContext (Ops)

/* ----------------------------------------------------------------------- */
/* Color operations */
/* ----------------------------------------------------------------------- */

- (void)DPScurrentcmykcolor: (float *)c : (float *)m : (float *)y : (float *)k 
{
  [gstate DPScurrentcmykcolor:c :m :y :k];
}

- (void)DPSsetcmykcolor: (float)c : (float)m : (float)y : (float)k 
{
  [gstate DPSsetcmykcolor:c :m :y :k];
}

- (void)DPScurrentrgbcolor: (float *)r : (float *)g : (float *)b 
{
  CHECK_NULL_OUTPUT(r);
  CHECK_NULL_OUTPUT(g);
  CHECK_NULL_OUTPUT(b);
  [gstate DPScurrentrgbcolor:r :g :b];
}

- (void)DPSsetrgbcolor: (float)r : (float)g : (float)b 
{
  [gstate DPSsetrgbcolor:r :g :b];
}

- (void)DPScurrenthsbcolor: (float *)h : (float *)s : (float *)b 
{
  CHECK_NULL_OUTPUT(h);
  CHECK_NULL_OUTPUT(s);
  CHECK_NULL_OUTPUT(b);
  [gstate DPScurrenthsbcolor:h :s :b];
}

- (void)DPSsethsbcolor: (float)h : (float)s : (float)b 
{
  [gstate DPSsethsbcolor:h :s :b];
}

- (void)DPScurrentgray: (float *)gray 
{
  CHECK_NULL_OUTPUT(gray);
  [gstate DPScurrentgray: gray];
}

- (void)DPSsetgray: (float)gray 
{
  [gstate DPSsetgray: gray];
}

- (void)DPScurrenthalftone 
{
}

- (void)DPScurrenthalftonephase: (float *)x : (float *)y 
{
}

- (void)DPSsethalftone 
{
}

- (void)DPSsethalftonephase: (float)x : (float)y 
{
}

/* ----------------------------------------------------------------------- */
/* Data operations */
/* ----------------------------------------------------------------------- */

- (void)DPSclear 
{
  GSIArrayEmpty((GSIArray)opstack);
  GSIArrayInitWithZoneAndCapacity((GSIArray)opstack, [self zone], 2);
}

- (void)DPScleartomark 
{
  id object = nil;

  while (object != mark)
    {
      CHECK_STACK_UNDERFLOW(opstack);
      object = (GSIArrayLastItem((GSIArray)opstack)).obj;
      CHECK_TYPECHECK(object, kind);
      GSIArrayRemoveLastItem((GSIArray)opstack);
    }
}

- (void)DPScopy: (int)n 
{
  unsigned count = GSIArrayCount((GSIArray)opstack);
  int i;

  for (i = 0; i < n; i++)
    {
      NSObject *obj = (GSIArrayItemAtIndex((GSIArray)opstack, count - n + i)).obj;

      ctxt_push(obj, opstack);
    }
}

- (void)DPScount: (int *)n 
{
  CHECK_NULL_OUTPUT(n);
  *n = GSIArrayCount((GSIArray)opstack);
}

- (void)DPScounttomark: (int *)n 
{
  unsigned count = GSIArrayCount((GSIArray)opstack);
  unsigned pos = count;

  CHECK_NULL_OUTPUT(n);
  while (pos > 0)
    {
      NSObject *obj = (GSIArrayItemAtIndex((GSIArray)opstack, --pos)).obj;

      if (obj == mark)
	{
	  *n = count - pos - 1;
	}
    }
  *n = count;
}

- (void)DPSdup 
{
  NSObject *obj = (GSIArrayLastItem((GSIArray)opstack)).obj;

  ctxt_push(obj, opstack);
}

- (void)DPSexch 
{
  unsigned count = GSIArrayCount((GSIArray)opstack);

  if (count < 2)
    DPS_ERROR(DPSstackunderflow, @"Attempt to exch in empty stack");
  GSIArrayInsertItem((GSIArray)opstack, 
		 GSIArrayLastItem((GSIArray)opstack), count-2);
  GSIArrayRemoveLastItem((GSIArray)opstack);
}

- (void)DPSexecstack 
{
}

- (void)DPSget 
{
}

- (void)DPSindex: (int)i 
{
  unsigned count = GSIArrayCount((GSIArray)opstack);
  NSObject *obj = (GSIArrayItemAtIndex((GSIArray)opstack, count - i)).obj;

  ctxt_push(obj, opstack);
}

- (void)DPSmark 
{
  ctxt_push(mark, opstack);
}

- (void)DPSmatrix 
{
  ctxt_push([NSAffineTransform_class transform], opstack);
}

- (void)DPSnull 
{
}

- (void)DPSpop 
{
  id obj;
  ctxt_pop(obj, opstack, NSObject);
}

- (void)DPSput 
{
}

- (void)DPSroll: (int)n : (int)j 
{
}

/* ----------------------------------------------------------------------- */
/* Font operations */
/* ----------------------------------------------------------------------- */

- (void)DPSdefineresource: (const char *)category 
{
  if (strcmp(category, "Font"))
    DPS_ERROR(DPSundefined, [NSString stringWithCString: category]);
  else
    [self DPSdefinefont];
}

- (void)DPSfindresource: (const char *)key : (const char *)category 
{
  if (strcmp(category, "Font"))
    DPS_ERROR(DPSundefined, @"Could not find key %s", key);
  else 
    [self DPSfindfont: key];
}

- (void)DPSFontDirectory 
{
}

- (void)DPSISOLatin1Encoding 
{
}

- (void)DPSSharedFontDirectory 
{
}

- (void)DPSStandardEncoding 
{
}

- (void)DPScurrentcacheparams 
{
}

- (void)DPScurrentfont 
{
  int ident = 0;
  NSFont *font = [gstate currentFont];

  if (font != nil)
    ident = [fontid indexOfObject: font];
  
  ctxt_push([NSNumber numberWithLong: ident], opstack);
}

- (void)DPSdefinefont 
{
  NSNumber *obj;
  NSString *key;

  ctxt_pop(obj, opstack, NSNumber);
  ctxt_pop(key, opstack, NSString);
  [globalfontdir setObject: obj forKey: key];
  ctxt_push(obj, opstack);
}

- (void)DPSfindfont: (const char *)font_name 
{
  NSNumber *obj;
  NSString *name = [NSString stringWithCString: font_name];

  obj = [globalfontdir objectForKey: name];

  if (obj == nil)
    {
      int ident;
      NSFont *font = [NSFont fontWithName: name
			     size: 0];

      if (font != nil)
        {
	  ident = [fontid count];
	  [fontid addObject: font];
	  obj = [NSNumber numberWithLong: ident];
	}
      else
	{
	  DPS_ERROR(DPSundefinedresource, @"Could not find font resource %@", name);
	}
    }

  ctxt_push(obj, opstack);
}

- (void)DPSmakefont 
{
  NSNumber *obj;
  long ident;
  NSFont *font;
  NSString *name;
  float fmatrix[6];
  NSAffineTransform *matrix;

  // Take a matrix from the stack
  ctxt_pop(matrix, opstack, NSAffineTransform_class);
  [matrix getMatrix: fmatrix];

  ctxt_pop(obj, opstack, NSNumber);
  ident = [obj longValue];
  CHECK_INVALID_FONT(ident);
  font = [fontid objectAtIndex: ident];
  name = [font fontName];

  font = [NSFont fontWithName: name 
		 matrix: fmatrix];
  if (font != nil)
    {
      ident = [fontid count];
      [fontid addObject: font];
    }
  ctxt_push([NSNumber numberWithLong: ident], opstack);
}

- (void)DPSscalefont: (float)size 
{
  NSNumber *obj;
  NSFont *font;
  NSString *name;
  long ident;

  ctxt_pop(obj, opstack, NSNumber);
  ident = [obj longValue];
  CHECK_INVALID_FONT(ident);
  font = [fontid objectAtIndex: ident];
  name = [font fontName];

  /* Try to find a font that matches this size */
  font = [NSFont fontWithName: name
		 size: size];
  if (font != nil)
    {
      ident = [fontid count];
      [fontid addObject: font];
    }
  ctxt_push([NSNumber numberWithLong: ident], opstack);
}

- (void)DPSselectfont: (const char *)font_name : (float)scale 
{
  NSString *name = [NSString stringWithCString: font_name];
  NSFont *font = [NSFont fontWithName: name
			 size: scale];

  [gstate setFont: font];
}

- (void)DPSsetfont: (int)ident 
{
  NSFont *font;

  CHECK_INVALID_FONT(ident);
  font = [fontid objectAtIndex: ident];
  [gstate setFont: font];
}

- (void)DPSundefinefont: (const char *)name 
{
  [globalfontdir removeObjectForKey: [NSString stringWithCString: name]];
}

- (void) setFont: (NSFont*) font
{
  [gstate setFont: font];
}

/* ----------------------------------------------------------------------- */
/* Gstate operations */
/* ----------------------------------------------------------------------- */

- (void)DPSconcat: (const float *)m 
{
  [gstate DPSconcat: m];
}

- (void)DPScurrentdash 
{
}

- (void)DPScurrentflat: (float *)flatness 
{
  CHECK_NULL_OUTPUT(flatness);
  [gstate DPScurrentflat: flatness];
}

- (void)DPScurrentgstate: (int)gst 
{
  if (gst)
    {
      /* Associate/copy current gstate with gst */
      ctxt_push([NSNumber numberWithInt: gst], opstack);
      ctxt_push(gstate, opstack);
      [self DPSdefineuserobject];
      [self DPSexecuserobject: gst];
    }
}

- (void)DPScurrentlinecap: (int *)linecap 
{
  [gstate DPScurrentlinecap: linecap];
}

- (void)DPScurrentlinejoin: (int *)linejoin 
{
  [gstate DPScurrentlinejoin: linejoin];
}

- (void)DPScurrentlinewidth: (float *)width 
{
  [gstate DPScurrentlinewidth: width];
}

- (void)DPScurrentmatrix 
{
  float m[6];
  id obj;

  ctxt_pop(obj, opstack, NSAffineTransform_class);
  [gstate DPScurrentmatrix: m];
  [obj setMatrix: m];
  ctxt_push(obj, opstack);
}

- (void)DPScurrentmiterlimit: (float *)limit 
{
  CHECK_NULL_OUTPUT(limit);
  [gstate DPScurrentmiterlimit: limit];
}

- (void)DPScurrentpoint: (float *)x : (float *)y 
{
  CHECK_NULL_OUTPUT(x);
  CHECK_NULL_OUTPUT(y);
  [gstate DPScurrentpoint:x :y];
}

- (void)DPScurrentscreen 
{
  [self DPSsendint: XSCR]; 
}

- (void)DPScurrentstrokeadjust: (int *)b 
{
  CHECK_NULL_OUTPUT(b);
  [gstate DPScurrentstrokeadjust: b];
}

- (void)DPScurrenttransfer 
{
}

- (void)DPSdefaultmatrix 
{
}

- (void)DPSgrestore 
{
  if (GSIArrayCount((GSIArray)gstack) == 0)
    return;
  RELEASE(gstate);
  gstate = (GSIArrayLastItem((GSIArray)gstack)).obj;
  /* If the gstate is part of a save object, set it, but don't pop
     it off the stack */
  if ([gstate saveObject] == nil)
    ctxt_pop(gstate, gstack, XGGState);
  RETAIN(gstate);
}

- (void)DPSgrestoreall 
{
  if (GSIArrayCount((GSIArray)gstack) == 0)
    return;
  RELEASE(gstate);
  while (GSIArrayCount((GSIArray)gstack) > 0)
    {
      gstate = (GSIArrayLastItem((GSIArray)gstack)).obj;
      if ([gstate saveObject] != nil)
	{
	  /* Set this gstate but don't pop it */
	  break;
	}
      else
	ctxt_pop(gstate, gstack, XGGState);
    }
  RETAIN(gstate);
}

- (void)DPSgsave 
{
  ctxt_push(gstate, gstack);
  AUTORELEASE(gstate);
  gstate = [gstate copy];
}

- (void)DPSgstate 
{
  ctxt_push(AUTORELEASE([gstate copy]), opstack);
}

- (void)DPSinitgraphics 
{
  [gstate DPSinitgraphics];
}

- (void)DPSinitmatrix 
{
  [gstate DPSinitmatrix];
}

- (void)DPSrotate: (float)angle 
{
  [gstate DPSrotate: angle];
}

- (void)DPSscale: (float)x : (float)y 
{
  [gstate DPSscale:x :y];
}

- (void)DPSsetdash: (const float *)pat : (int)size : (float)offset 
{
  [gstate DPSsetdash: pat : size : offset]; 
}

- (void)DPSsetflat: (float)flatness 
{
  [gstate DPSsetflat: flatness];
}

- (void)DPSsetgstate: (int)gst 
{
  if (gst)
    {
      [self DPSexecuserobject: gst];
      RELEASE(gstate);
      ctxt_pop(gstate, opstack, XGGState);
      RETAIN(gstate);
    }
  else
    DESTROY(gstate);
}

- (void)DPSsetlinecap: (int)linecap 
{
  [gstate DPSsetlinecap: linecap];
}

- (void)DPSsetlinejoin: (int)linejoin 
{
  [gstate DPSsetlinejoin: linejoin];
}

- (void)DPSsetlinewidth: (float)width 
{
  [gstate DPSsetlinewidth: width];
}

- (void)DPSsetmatrix 
{
  float matrix[6];
  NSAffineTransform *obj;

  ctxt_pop(obj, opstack, NSAffineTransform_class);
  [obj getMatrix: matrix];
  [gstate DPSsetmatrix: matrix];
}

- (void)DPSsetmiterlimit: (float)limit 
{
  [gstate DPSsetmiterlimit: limit];
}

- (void)DPSsetscreen 
{
// Should pop a number and change the X screen
}

- (void)DPSsetstrokeadjust: (int)b 
{
  [gstate DPSsetstrokeadjust: b];
}

- (void)DPSsettransfer 
{
}

- (void)DPStranslate: (float)x : (float)y 
{
  [gstate DPStranslate:x :y];
}

/* ----------------------------------------------------------------------- */
/* Flush operation */
/* ----------------------------------------------------------------------- */
- (void)DPSflush
{
}

/* ----------------------------------------------------------------------- */
/* Matrix operations */
/* ----------------------------------------------------------------------- */

- (void)DPSconcatmatrix
{
  NSAffineTransform *obj1, *obj2, *obj3;

  ctxt_pop(obj3, opstack, NSAffineTransform_class);
  ctxt_pop(obj2, opstack, NSAffineTransform_class);
  ctxt_pop(obj1, opstack, NSAffineTransform_class);
  [obj2 appendTransform: obj1];
  obj3 = [obj2 copy];
  ctxt_push(obj3, opstack);
  RELEASE(obj3);
}

- (void)DPSdtransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: NSAffineTransform_class])
    {
      ctxt_push(obj, opstack);
      point = [gstate deltaPointInMatrixSpace: NSMakePoint(x1, y1)];
    }
  else
    {
      point = [obj deltaPointInMatrixSpace: NSMakePoint(x1, y1)];
    }
  *x2 = point.x;
  *y2 = point.y;
}

- (void)DPSidentmatrix 
{
  NSAffineTransform *obj;

  ctxt_pop(obj, opstack, NSAffineTransform_class);
  obj = [NSAffineTransform_class transform];
  ctxt_push(obj, opstack);
}

- (void)DPSidtransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: NSAffineTransform_class])
    {
      float m[6];

      ctxt_push(obj, opstack);
      // Optimization: We could get the current transformation from the GState
      [gstate DPScurrentmatrix: m];
      obj = [NSAffineTransform_class matrixFrom: m];
    }
  [obj invert];
  point.x = x1; point.y = y1;
  point = [obj deltaPointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

- (void)DPSinvertmatrix 
{
  NSAffineTransform *obj, *obj2;

  ctxt_pop(obj, opstack, NSAffineTransform_class);
  ctxt_pop(obj2, opstack, NSAffineTransform_class);
  [obj invert];
  ctxt_push(obj, opstack);
}

- (void)DPSitransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: NSAffineTransform_class])
    {
      float m[6];

      ctxt_push(obj, opstack);
      // Optimization: We could get the current transformation from the GState
      [gstate DPScurrentmatrix: m];
      obj = [NSAffineTransform_class matrixFrom: m];
    }
  [obj invert];
  point.x = x1; point.y = y1;
  point = [obj pointInMatrixSpace: point];
  *x2 = point.x;
  *y2 = point.y;
}

- (void)DPStransform: (float)x1 : (float)y1 : (float *)x2 : (float *)y2 
{
  NSAffineTransform *obj;
  NSPoint point;

  ctxt_pop(obj, opstack, Nil);
  if (![obj isKindOfClass: NSAffineTransform_class])
    {
      ctxt_push(obj, opstack);
      point = [gstate pointInMatrixSpace: NSMakePoint(x1, y1)];
    }
  else
    {
      point = [obj pointInMatrixSpace: NSMakePoint(x1, y1)];
    }
  *x2 = point.x;
  *y2 = point.y;
}

/* ----------------------------------------------------------------------- */
/* Opstack operations */
/* ----------------------------------------------------------------------- */

- (void)DPSdefineuserobject
{
  int n;
  id obj;
  NSNumber *number;
  ctxt_pop(obj, opstack, NSObject);
  ctxt_pop(number, opstack, NSNumber);
  n = [number intValue];
  if (n < 0)
    DPS_ERROR(DPSinvalidparam, @"Invalid userobject index");
  else if (n < [ulist count])
    [ulist replaceObjectAtIndex: n withObject: obj];
  else
    {
      int i = n - [ulist count];
      while (i--)
	[ulist addObject: AUTORELEASE([[NSObject alloc] init])];
      [ulist addObject: obj];
    }
}

- (void)DPSexecuserobject: (int)index
{
  if (index < 0 || index >= [ulist count])
    DPS_ERROR(DPSinvalidparam, @"Invalid userobject index");
  ctxt_push([ulist objectAtIndex: index], opstack);
}

- (void)DPSundefineuserobject: (int)index
{
  if (index < 0 || index >= [ulist count])
    DPS_ERROR(DPSinvalidparam, @"Invalid userobject index");
  [ulist replaceObjectAtIndex: index
                   withObject: AUTORELEASE([[NSObject alloc] init])];
}

- (void)DPSgetboolean: (int *)it 
{
  NSNumber *number;
  CHECK_NULL_OUTPUT(it);
  ctxt_pop(number, opstack, NSNumber);
  *it = [number boolValue];
}

- (void)DPSgetchararray: (int)size : (char *)s 
{
  NSMutableData *data;
  CHECK_NULL_OUTPUT(s);
  ctxt_pop(data, opstack, NSMutableData);
  memcpy(s, [data bytes], size*sizeof(char));
}

- (void)DPSgetfloat: (float *)it 
{
  NSNumber *number;
  CHECK_NULL_OUTPUT(it);
  ctxt_pop(number, opstack, NSNumber);
  *it = [number floatValue];
}

- (void)DPSgetfloatarray: (int)size : (float *)a 
{
  NSMutableData *data;
  CHECK_NULL_OUTPUT(a);
  ctxt_pop(data, opstack, NSMutableData);
  memcpy(a, [data bytes], size*sizeof(float));
}

- (void)DPSgetint: (int *)it 
{
  NSNumber *number;
  CHECK_NULL_OUTPUT(it);
  ctxt_pop(number, opstack, NSNumber);
  *it = [number intValue];
}

- (void)DPSgetintarray: (int)size : (int *)a 
{
  NSMutableData *data;
  CHECK_NULL_OUTPUT(a);
  ctxt_pop(data, opstack, NSMutableData);
  memcpy(a, [data bytes], size*sizeof(int));
}

- (void)DPSgetstring: (char *)s 
{
  NSString *str;
  CHECK_NULL_OUTPUT(s);
  ctxt_pop(str, opstack, NSString);
  strcpy(s, [str cString]);
}

- (void)DPSsendboolean: (int)it 
{
  ctxt_push([NSNumber numberWithBool: it], opstack);
}

- (void)DPSsendchararray: (const char *)s : (int)size 
{
  NSMutableData *data;
  data = [NSMutableData dataWithBytes: s length: size*sizeof(char)];
  ctxt_push(data, opstack);
}

- (void)DPSsendfloat: (float)it 
{
  ctxt_push([NSNumber numberWithFloat: it], opstack);
}

- (void)DPSsendfloatarray: (const float *)a : (int)size 
{
  NSMutableData *data;
  data = [NSMutableData dataWithBytes: a length: size*sizeof(float)];
  ctxt_push(data, opstack);
}

- (void)DPSsendint: (int)it 
{
  ctxt_push([NSNumber numberWithInt: it], opstack);
}

- (void)DPSsendintarray: (const int *)a : (int)size 
{
  NSMutableData *data;
  data = [NSMutableData dataWithBytes: a length: size*sizeof(int)];
  ctxt_push(data, opstack);
}

- (void)DPSsendstring: (const char *)s 
{
  ctxt_push([NSString stringWithCString: s], opstack);
}

/* ----------------------------------------------------------------------- */
/* Paint operations */
/* ----------------------------------------------------------------------- */

- (void)DPSashow: (float)x : (float)y : (const char *)s 
{
  [gstate DPSashow: x : y : s];
}

- (void)DPSawidthshow: (float)cx : (float)cy : (int)c : (float)ax : (float)ay : (const char *)s 
{
  [gstate DPSawidthshow: cx : cy : c : ax : ay : s];
}

- (void)DPScopypage 
{
}

- (void)DPSeofill 
{
  [gstate DPSeofill];
}

- (void)DPSerasepage 
{
}

- (void)DPSfill 
{
  [gstate DPSfill];
}

- (void)DPSimage 
{
  NSAffineTransform *trans; 
  int pixelsWide;
  int pixelsHigh;
  int bitsPerSample;
  const unsigned char *data[5];
  id obj;

  ctxt_pop(obj, opstack, NSString);
  if ([obj respondsToSelector: @selector(cString)])
    data[0] = (unsigned char*)[obj cString];
  else
    data[0] = (unsigned char*)[obj pointerValue];

  ctxt_pop(trans, opstack, NSAffineTransform_class);
  ctxt_pop(obj, opstack, NSNumber);
  bitsPerSample = [obj intValue];
  ctxt_pop(obj, opstack, NSNumber);
  pixelsHigh = [obj intValue];
  ctxt_pop(obj, opstack, NSNumber);
  pixelsWide = [obj intValue];

  [gstate DPSimage: trans 
	  : pixelsWide : pixelsHigh 
	  : bitsPerSample : 1
	  : 0 : 0 
	  : NO : NO 
	  : NSCalibratedWhiteColorSpace
	  : data];
}

- (void)DPSimagemask 
{
}

- (void)DPScolorimage
{
  NSAffineTransform *trans; 
  int pixelsWide;
  int pixelsHigh;
  int bitsPerSample;
  int samplesPerPixel; 
  BOOL isPlanar;
  BOOL hasAlpha;
  NSString *colorSpaceName;
  const unsigned char *data[5];
  id obj;
  int i, n;

  ctxt_pop(obj, opstack, NSNumber);
  n = [obj intValue];
  samplesPerPixel = n;

  if (n == 1)
    colorSpaceName = NSDeviceWhiteColorSpace;
  else if (n == 3)
    colorSpaceName = NSDeviceRGBColorSpace;
  else if (n == 4)
    colorSpaceName = NSDeviceCMYKColorSpace;

  for (i = 0; i < n; i++)
    {
      ctxt_pop(obj, opstack, NSString);
      if ([obj respondsToSelector: @selector(cString)])
	data[i] = (unsigned char*)[obj cString];
      else
	data[i] = (unsigned char*)[obj pointerValue];
    }

  ctxt_pop(trans, opstack, NSAffineTransform_class);
  ctxt_pop(obj, opstack, NSNumber);
  bitsPerSample = [obj intValue];
  ctxt_pop(obj, opstack, NSNumber);
  pixelsHigh = [obj intValue];
  ctxt_pop(obj, opstack, NSNumber);
  pixelsWide = [obj intValue];

  [gstate DPSimage: trans 
	  : pixelsWide : pixelsHigh 
	  : bitsPerSample : samplesPerPixel
	  : 0 : 0 
	  : YES : NO 
	  : colorSpaceName
	  : data];
}

- (void)DPSalphaimage
{
  NSAffineTransform *trans; 
  int pixelsWide;
  int pixelsHigh;
  int bitsPerSample;
  int samplesPerPixel; 
  BOOL isPlanar;
  BOOL hasAlpha;
  NSString *colorSpaceName;
  const unsigned char *data[5];
  id obj;
  int i, n;
  NSMutableDictionary *dict;

  ctxt_pop(obj, opstack, NSNumber);
  n = [obj intValue];
  samplesPerPixel = n;

  // FIXME - Shouldn't this be one more??
  if (n == 1)
    colorSpaceName = NSDeviceWhiteColorSpace;
  else if (n == 3)
    colorSpaceName = NSDeviceRGBColorSpace;
  else if (n == 4)
    colorSpaceName = NSDeviceCMYKColorSpace;

  for (i = 0; i < n; i++)
    {
      ctxt_pop(obj, opstack, NSString);
      if ([obj respondsToSelector: @selector(cString)])
	data[i] = (unsigned char*)[obj cString];
      else
	data[i] = (unsigned char*)[obj pointerValue];
    }

  ctxt_pop(trans, opstack, NSAffineTransform_class);
  ctxt_pop(obj, opstack, NSNumber);
  bitsPerSample = [obj intValue];
  ctxt_pop(obj, opstack, NSNumber);
  pixelsHigh = [obj intValue];
  ctxt_pop(obj, opstack, NSNumber);
  pixelsWide = [obj intValue];

  [gstate DPSimage: trans 
	  : pixelsWide : pixelsHigh 
	  : bitsPerSample : samplesPerPixel
	  : 0 : 0 
	  : YES : YES 
	  : colorSpaceName
	  : data];
}

- (void)DPSkshow: (const char *)s 
{
}

- (void)DPSrectfill: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectfill:x :y :w :h];
}

- (void)DPSrectstroke: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectstroke:x :y :w :h];
}

- (void)DPSshow: (const char *)s 
{
  [gstate DPSshow: s];
}

- (void)DPSshowpage 
{
}

- (void)DPSstroke 
{
  [gstate DPSstroke];
}

- (void)DPSstrokepath 
{
  [gstate DPSstrokepath];
}

- (void)DPSueofill: (const char *)nums : (int)n : (const char *)op : (int)l 
{
  XGGState *new_gstate = [gstate copy];

  [new_gstate DPSnewpath];
  // FIXME: Set the path from the given values

  [new_gstate DPSeofill];
  RELEASE(new_gstate);
}

- (void)DPSufill: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSustroke: (const char *)nums   : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSustrokepath: (const char *)nums : (int)n : (const char *)ops : (int)l 
{
}

- (void)DPSwidthshow: (float)x : (float)y : (int)c : (const char *)s 
{
  [gstate DPSwidthshow: x : y : c : s];
}

- (void)DPSxshow: (const char *)s : (const float *)numarray : (int)size 
{
  [gstate DPSxshow: s : numarray : size];
}

- (void)DPSxyshow: (const char *)s : (const float *)numarray : (int)size 
{
  [gstate DPSxyshow: s : numarray : size];
}

- (void)DPSyshow: (const char *)s : (const float *)numarray : (int)size 
{
  [gstate DPSyshow: s : numarray : size];
}

/* ----------------------------------------------------------------------- */
/* Path operations */
/* ----------------------------------------------------------------------- */

- (void)DPSarc: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
  [gstate DPSarc: x : y : r : angle1 : angle2];
}

- (void)DPSarcn: (float)x : (float)y : (float)r : (float)angle1 : (float)angle2 
{
  [gstate DPSarcn: x : y : r : angle1 : angle2];
}

- (void)DPSarct: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r 
{
  [gstate DPSarct: x1 : y1 : x2 : y2 : r];
}

- (void)DPSarcto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)r 
                : (float *)xt1 : (float *)yt1 : (float *)xt2 : (float *)yt2 
{
  [gstate DPSarcto: x1 : y1 : x2 : y2 : r : xt1 : yt1 : xt2 : yt2];
}

- (void)DPScharpath: (const char *)s : (int)b 
{
  [gstate DPScharpath: s : b];
}

- (void)DPSclip 
{
  [gstate DPSclip];
}

- (void)DPSclippath 
{
  [gstate DPSclippath];
}

- (void)DPSclosepath 
{
  [gstate DPSclosepath];
}

- (void)DPScurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3
{
  [gstate DPScurveto: x1 : y1 : x2 : y2 : x3 : y3];
}

- (void)DPSeoclip 
{
  [gstate DPSeoclip];
}

- (void)DPSeoviewclip 
{
  [gstate DPSeoviewclip];
}

- (void)DPSflattenpath 
{
  [gstate DPSflattenpath];
}

- (void)DPSinitclip 
{
  [gstate DPSinitclip];
}

- (void)DPSinitviewclip 
{
  [gstate DPSinitviewclip];
}

- (void)DPSlineto: (float)x : (float)y 
{
  [gstate DPSlineto: x : y];
}

- (void)DPSmoveto: (float)x : (float)y 
{
  [gstate DPSmoveto: x : y];
}

- (void)DPSnewpath 
{
  [gstate DPSnewpath];
}

- (void)DPSpathbbox: (float *)llx : (float *)lly : (float *)urx : (float *)ury 
{
  [gstate DPSpathbbox: llx : lly : urx : ury];
}

- (void)DPSpathforall 
{
  [gstate DPSpathforall];
}

- (void)DPSrcurveto: (float)x1 : (float)y1 : (float)x2 : (float)y2 : (float)x3 : (float)y3 
{
  [gstate DPSrcurveto: x1 : y1 : x2 : y2 : x3 : y3];
}

- (void)DPSrectclip: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectclip: x : y : w : h];
}

- (void)DPSrectviewclip: (float)x : (float)y : (float)w : (float)h 
{
  [gstate DPSrectviewclip: x : y : w : h];
}

- (void)DPSreversepath 
{
  [gstate DPSreversepath];
}

- (void)DPSrlineto: (float)x : (float)y 
{
  [gstate DPSrlineto: x : y];
}

- (void)DPSrmoveto: (float)x : (float)y 
{
  [gstate DPSrmoveto: x : y];
}

- (void)DPSsetbbox: (float)llx : (float)lly : (float)urx : (float)ury 
{
  [gstate DPSsetbbox: llx : lly : urx : ury];
}

- (void)DPSviewclip 
{
  [gstate DPSviewclip];
}

- (void)DPSviewclippath 
{
  [gstate DPSviewclippath];
}

/* ----------------------------------------------------------------------- */
/* System ops */
/* ----------------------------------------------------------------------- */

- (void)DPSrestore
{
  XRectangle	r;

  if (vmstackTop-1 >= VM_STACKSIZE)
    {
       vmstackTop--;
       return;
    }
  if (vmstackTop <= 0)
    {
      NSLog(@"DPSrestore: unbalanced save/restore");
      return;
    }
  // pop topmost rectangle from stack
  r = vcstack[--vmstackTop];
  if (r.width && viewclip)
    {
      XSubtractRegion(viewclip, viewclip, viewclip);
      XUnionRectWithRegion(&r, viewclip, viewclip);
      [gstate setClipMask];
    }
  else
    [self DPSinitviewclip];
  /* grestoreall and pop our gstate */
  [self DPSgrestoreall];
  ctxt_pop(gstate, gstack, XGGState);
  /* Already been retained by grestoreall */
  [gstate setSaveObject: nil];
}

- (void)DPSsave
{
  XRectangle	r;

  if (vmstackTop >= VM_STACKSIZE)
    {
      NSLog(@"DPSsave: stack limit exceeded !");
      vmstackTop++;
      return;
    }
  // add viewclip rectangle to stack
  if (viewclip)
    XClipBox(viewclip, &r);
  else
    r.width = r.height = 0;
  vcstack[vmstackTop++] = r;

  /* Save a gstate, but mark it as ours */
  [gstate setSaveObject: (id)vmstackTop];
  ctxt_push(gstate, gstack);
  AUTORELEASE(gstate);
  gstate = [gstate copy];
}

/* ----------------------------------------------------------------------- */
/* Window system ops */
/* ----------------------------------------------------------------------- */

- (void) DPScurrentdrawingfunction: (int *)function
{
}

- (void) DPScurrentgcdrawable: (void **)gc : (void **)draw : (int *)x : (int *)y
{
  if (gc)
    *gc = (void *)[gstate graphicContext];
  if (draw)
    *draw = (void *)[gstate drawable];
  if (x && y)
    {
      NSPoint offset = [gstate offset];
      *x = offset.x;
      *y = offset.y;
    }
}

- (void) DPScurrentgcdrawablecolor: (void **)gc : (void **)draw : (int *)x 
				  : (int *)y : (int *)colorInfo
{
  if (gc)
    *gc = (void *)[gstate graphicContext];
  if (draw)
    *draw = (void *)[gstate drawable];
  if (x && y)
    {
      NSPoint offset = [gstate offset];
      *x = offset.x;
      *y = offset.y;
    }
  if (colorInfo)
    *colorInfo = 0;
}

- (void) DPScurrentoffset: (int *)x : (int *)y
{
  if (x && y)
    {
      NSPoint offset = [gstate offset];
      *x = offset.x;
      *y = offset.y;
    }
}

- (void) DPSsetdrawingfunction: (int) function
{
  XGCValues values;
  values.function = function;
  [gstate setGCValues: values withMask: (GCFunction | GCForeground)];
}

- (void) DPSsetgcdrawable: (void *)gc : (void *)draw : (int)x : (int)y
{
  [gstate setGraphicContext: (GC)gc];
  [gstate setDrawable: (Drawable)draw];
  [gstate setOffset: NSMakePoint(x, y)];
}

- (void) DPSsetgcdrawablecolor: (void *)gc : (void *)draw : (int)x : (int)y
				  : (const int *)colorInfo
{
  [gstate setGraphicContext: (GC)gc];
  [gstate setDrawable: (Drawable)draw];
  [gstate setOffset: NSMakePoint(x, y)];
}

- (void) DPSsetoffset: (short int)x : (short int)y
{
  [gstate setOffset: NSMakePoint(x,y)];
}

- (void) DPSsetrgbactual: (double)r : (double)g : (double)b : (int *)success
{
  [gstate DPSsetrgbcolor: r : g : b];
  *success = 1;
}

/* Add the current gstate to the gstate index (if not there already)
   and return its index */
- (void) DPScapturegstate: (int *)gst
{
  CHECK_NULL_OUTPUT(gst);
  if ([ulist indexOfObject: gstate] == NSNotFound)
    [ulist addObject: gstate];
  *gst = [ulist indexOfObject: gstate];
}

/*-------------------------------------------------------------------------*/
/* Graphics Extension Ops */
/*-------------------------------------------------------------------------*/
- (void) DPScomposite: (float)x : (float)y : (float)w : (float)h 
		     : (int)gstateNum : (float)dx : (float)dy : (int)op
{
  NSRect rect;
  NSPoint p;
  XGGState *g = gstate;

  if (gstateNum)
    {
      Drawable source;
      [self DPSexecuserobject: gstateNum];
      ctxt_pop(g, opstack, XGGState);
      source = [g drawable];
      if (!source) 
        DPS_ERROR(DPSundefined, @"Composite drawable not realized\n");
    }

  rect = NSMakeRect(x, y, w, h);
  p    = NSMakePoint(dx, dy);

  [gstate compositeGState: g fromRect: rect toPoint: p op: op];
}

- (void) DPScompositerect: (float)x : (float)y : (float)w : (float)h : (int)op
{
  [gstate  compositerect: NSMakeRect(x, y, w, h) op: op];
}

- (void) DPSdissolve: (float)x : (float)y : (float)w : (float)h : (int)gstateNum
		    : (float)dx : (float)dy : (float)delta
{
  NSRect rect;
  NSPoint p;
  XGGState *g = gstate;

  if (gstateNum)
    {
      Drawable source;
      [self DPSexecuserobject: gstateNum];
      ctxt_pop(g, opstack, XGGState);
      source = [g drawable];
      if (!source) 
        DPS_ERROR(DPSundefined, @"Composite drawable not realized\n");
    }

  rect = NSMakeRect(x, y, w, h);
  p    = NSMakePoint(dx, dy);

  [gstate dissolveGState: g fromRect: rect toPoint: p delta: delta];
}

- (void) DPSreadimage
{
}

- (void) DPSsetalpha: (float)a
{
  [gstate DPSsetalpha: a];
}

- (void) DPScurrentalpha: (float *)alpha
{
  [gstate DPScurrentalpha: alpha];
}

/* ----------------------------------------------------------------------- */
/* Client functions */
/* ----------------------------------------------------------------------- */
- (void) DPSPrintf: (char *)fmt : (va_list)args
{
  /* Do nothing. We can't parse PostScript */
}

- (void) DPSWriteData: (char *)buf : (unsigned int)count
{
  /* Do nothing. We can't parse PostScript */
}

@end

void
GSWinitcontext(NSGraphicsContext *ctxt, int window_number, GC xgc, 
	       Drawable drawable, int xoff, int yoff)
{
  [[(XGContext *)ctxt xrCurrentGState] setWindow: window_number];
  DPSsetgcdrawable(ctxt, xgc, (void *)drawable, xoff, yoff);
  DPSinitmatrix(ctxt);
  DPSinitclip(ctxt);
}
