/*
**  MimeTypeManager.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import <AppKit/AppKit.h>

#import "GNUMailConstants.h"
#import "MimeTypeManager.h"
#import "MimeType.h"
#import "NSStringExtensions.h"
#import "Utilities.h"

NSString *PathToMimeTypes()
{
  return [NSString stringWithFormat: @"%@/%@",
		   GNUMailUserLibraryPath(), @"Mime-Types"];
}

@implementation MimeTypeManager

- (id) init
{
  [super init];

  [self setMimeTypes: [NSArray array]];

  return self;
}

- (void) dealloc
{
  RELEASE(mimeTypes);

  [super dealloc];
}


- (BOOL) synchronize
{
  return [NSArchiver archiveRootObject: self toFile: PathToMimeTypes()];
}

//
// NSCoding protocol
//

- (void) encodeWithCoder: (NSCoder *) theCoder
{
  [theCoder encodeObject: [self mimeTypes]];
}

- (id) initWithCoder: (NSCoder *) theCoder
{
  self = [super init];

  [self setMimeTypes: [theCoder decodeObject]];

  return self;
}


//
// access/mutation methods
//

- (MimeType *) mimeTypeAtIndex: (int) theIndex
{
  return [mimeTypes objectAtIndex: theIndex];
}

- (void) addMimeType: (MimeType *) theMimeType
{
  [mimeTypes addObject: theMimeType];
}

- (void) removeMimeType: (MimeType *) theMimeType;
{
  [mimeTypes removeObject: theMimeType];
}

- (NSArray *) mimeTypes
{
  return mimeTypes;
}

- (void) setMimeTypes: (NSArray *) theMimeTypes
{
  if ( theMimeTypes )
    {
      NSMutableArray *newMimeTypes;

      newMimeTypes = [[NSMutableArray alloc] initWithArray: theMimeTypes];
      RELEASE(mimeTypes);
      mimeTypes = newMimeTypes;
    }
  else
    {
      RELEASE(mimeTypes);
      mimeTypes = nil;
    }
}


//
// FIXME: This method should be 'table driven'.
//
- (MimeType *) bestMimeTypeForFileExtension: (NSString *) theFileExtension
{
  MimeType *aMimeType;
  
  // We first search from the MIME types specified by the user
  // If we haven't found it, we try to guess the best one we could use
  aMimeType = [self mimeTypeForFileExtension: theFileExtension];

  if (! aMimeType )
    {
      aMimeType = [[MimeType alloc] init];
      
      // video/x-msvideo
      if ( [theFileExtension caseInsensitiveCompare: @"avi"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"video/x-msvideo"];
	}
      // text/plain
      else if ( [theFileExtension caseInsensitiveCompare: @"c"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"cc"] == NSOrderedSame || 
		[theFileExtension caseInsensitiveCompare: @"h"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"hh"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"m"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"txt"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"text/plain"];
	}
      // application/x-csh
      else if ( [theFileExtension caseInsensitiveCompare: @"csh"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"application/x-csh"];
	}
      // application/msword
      else if ( [theFileExtension caseInsensitiveCompare: @"doc"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"application/msword"];
	}
      // application/postscript
      else if ( [theFileExtension caseInsensitiveCompare: @"ai"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"eps"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"ps"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"application/postscript"];
	}
      // image/gif
      else if ( [theFileExtension caseInsensitiveCompare: @"gif"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"image/gif"];
	}
      // application/x-gzip
      else if ( [theFileExtension caseInsensitiveCompare: @"gz"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"application/x-gzip"];
	}
      // text/html
      else if ( [theFileExtension caseInsensitiveCompare: @"htm"] == NSOrderedSame || 
		[theFileExtension caseInsensitiveCompare: @"html"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"text/html"];
	}
      // image/jpeg
      else if ( [theFileExtension caseInsensitiveCompare: @"jpe"] == NSOrderedSame || 
		[theFileExtension caseInsensitiveCompare: @"jpeg"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"jpg"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"image/jpeg"];
	}
      // audio/midi
      else if ( [theFileExtension caseInsensitiveCompare: @"mid"] == NSOrderedSame || 
		[theFileExtension caseInsensitiveCompare: @"midi"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"audio/midi"];
	}
      // video/quicktime
      else if ( [theFileExtension caseInsensitiveCompare: @"mov"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"qt"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"video/quicktime"];
	}
      // video/mpeg
      else if ( [theFileExtension caseInsensitiveCompare: @"mpe"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"mpeg"] == NSOrderedSame || 
		[theFileExtension caseInsensitiveCompare: @"mpg"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"video/mpeg"];
	}
      // audio/mpeg
      else if ( [theFileExtension caseInsensitiveCompare: @"mp2"] == NSOrderedSame ||
		[theFileExtension caseInsensitiveCompare: @"mp3"] == NSOrderedSame || 
		[theFileExtension caseInsensitiveCompare: @"mpga"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"audio/mpeg"];
	}
      // application/pdf
      else if ( [theFileExtension caseInsensitiveCompare: @"pdf"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"application/pdf"];
	}
      // image/png
      else if ( [theFileExtension caseInsensitiveCompare: @"png"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"image/png"];
	}
      // application/mspowerpoint
      else if ( [theFileExtension caseInsensitiveCompare: @"ppt"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"application/mspowerpoint"];
	}
      // text/rtf
      else if ( [theFileExtension caseInsensitiveCompare: @"rtf"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"text/rtf"];
	}
      // application/x-tar
      else if ( [theFileExtension caseInsensitiveCompare: @"tar"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"application/x-tar"];
	}
      // image/tiff
      else if ( [theFileExtension caseInsensitiveCompare: @"tif"] == NSOrderedSame || 
		[theFileExtension caseInsensitiveCompare: @"tiff"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"image/tiff"];
	}
      // audio/x-wav
      else if ( [theFileExtension caseInsensitiveCompare: @"wav"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"audio/x-wav"];
	}
      // text/xml
      else if ( [theFileExtension caseInsensitiveCompare: @"xml"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"text/xml"];
	}
      // application/zip
      else if ( [theFileExtension caseInsensitiveCompare: @"zip"] == NSOrderedSame )
	{
	  [aMimeType setMimeType: @"application/zip"];
	}
      // application/octet-stream (this is the default)
      else
	{
	  [aMimeType setMimeType: @"application/octet-stream"];
	}
      
      return aMimeType;
    }

  return aMimeType;
}

- (MimeType *) mimeTypeForFileExtension: (NSString *) theFileExtension
{
  NSEnumerator *anEnumerator;
  MimeType *aMimeType;
  NSString *aString;
  int i;
  
  if (theFileExtension && [theFileExtension length] > 0)
    {
      for (i = 0; i < [[self mimeTypes] count]; i++)
	{
	  aMimeType = [[self mimeTypes] objectAtIndex: i];
	  anEnumerator = [aMimeType fileExtensions];
	  
	  while ( (aString = [anEnumerator nextObject]) )
	    {
	      if ([[aString stringByTrimmingWhiteSpaces] caseInsensitiveCompare: theFileExtension] == NSOrderedSame)
		{
		  return aMimeType;
		}
	    }
	}
    }

  return nil;
}

- (MimeType *) mimeTypeFromString: (NSString *) theString
{
  MimeType *aMimeType;
  int i;

  if (theString && [theString length] > 0)
    {
      for (i = 0; i < [[self mimeTypes] count]; i++) 
	{
	  aMimeType = [[self mimeTypes] objectAtIndex: i];

	  if ([[aMimeType mimeType] caseInsensitiveCompare: theString] == NSOrderedSame)
	    {
	      return aMimeType;
	    }
	}
    }

  return nil;
}

//
// class methods
//

+ (id) mimeTypesFromDisk
{
  return [NSUnarchiver unarchiveObjectWithFile: PathToMimeTypes()];
}

@end
