/*
**  MailcapWindowController.m
**
**  Copyright (c) 2001
**
**  Author: Vincent Ricard <vricard@wanadoo.fr> 
**          Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "MailcapWindowController.h"

#ifndef MACOSX
#import "MailcapWindow.h"
#endif

#import "MailcapParser.h"
#import "MimeType.h"
#import "GNUMailConstants.h"

@implementation MailcapWindowController

//
//
//
- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
#else
  MailcapWindow *mailcapWindow;
  
  // We create our view
  mailcapWindow = [[MailcapWindow alloc] initWithContentRect: NSMakeRect(100,100,500,265)
					 styleMask: NSTitledWindowMask
					 backing: NSBackingStoreRetained
					 defer: NO];
  
  self = [super initWithWindow: mailcapWindow];
  
  [mailcapWindow layoutWindow];
  [mailcapWindow setDelegate: self];
  
  // We link our outlets
  tableView = [mailcapWindow tableView];
  mimeTypesColumn = [mailcapWindow mimeTypesColumn];
  commandColumn = [mailcapWindow commandColumn];

  RELEASE(mailcapWindow);
#endif

  [[self window] setTitle: _(@"Import from mailcap file")];

  return self;
}


//
//
//
- (void) dealloc
{
  NSLog(@"MailcapWindowController: -dealloc");
  
  RELEASE(mimeTypes);
  RELEASE(chosenMimeTypes);

  [super dealloc];
}

//
// delegate methods
//

- (void) windowDidLoad
{
  chosenMimeTypes = [[NSMutableArray alloc] init];

  if (! mimeTypes)
    {
      mimeTypes = [[NSMutableArray alloc] init];
    }
}

//
// action methods
//
- (IBAction) importClicked: (id) sender
{
  NSEnumerator * anEnumerator;
  NSNumber *aRow;

  if ( [tableView numberOfSelectedRows] == 0 )
    {
      NSRunInformationalAlertPanel(_(@"Mailcap Import Error!"),
				   _(@"You must select at least one mailcap item to import!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }

  anEnumerator = [tableView selectedRowEnumerator];
    
  while ( (aRow = [anEnumerator nextObject]) )
    {
      [self chose: [aRow intValue]];
    }
  
  [NSApp stopModal];
  [self close];
}


- (IBAction) cancelClicked: (id) sender
{
  [NSApp abortModal];
  [self close];
}

//
// Data Source methods
//

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  if ( mimeTypes )
    {
      return [mimeTypes count];
    }
  else
    {
      return 0;
    }
}

- (id)           tableView: (NSTableView *)aTableView
 objectValueForTableColumn: (NSTableColumn *)aTableColumn 
		       row: (int)rowIndex
{
  if (aTableColumn == mimeTypesColumn)
    {
      return [[self mimeType: rowIndex] mimeType];
    }
  else
    {
      return [[self mimeType: rowIndex] dataHandlerCommand];
    }
}


//
// access / mutation methods
//

- (MimeType *) mimeType : (int) aIndex
{
  return [mimeTypes objectAtIndex: aIndex];
}

- (NSArray *) getChoices
{
  return [NSMutableArray arrayWithArray: chosenMimeTypes];
}

- (NSString *) filename
{
  return filename;
}

- (void) setFilename: (NSString *) theFilename
{
  if (! mimeTypes)
    {
      mimeTypes = [[NSMutableArray alloc] init];
    }
  else
    {
      [mimeTypes removeAllObjects];
    }

  if ( theFilename )
    {
      MailcapParser *aMailcapParser;
      NSArray *anArray;
      
      RETAIN(theFilename);
      RELEASE(filename);
      filename = theFilename;
      

      // We parse our mailcap file so that will provide data for our view (MailcapWindow)
      aMailcapParser = [[MailcapParser alloc] init];
      anArray = [aMailcapParser parseFile: filename];
      
      if (anArray)
	{
	  [mimeTypes addObjectsFromArray: anArray];
	}
     
      RELEASE(aMailcapParser);

      [tableView reloadData];
      [tableView setNeedsDisplay: YES];
    }
  else
    {
      RELEASE(filename);
      filename = nil;
    }
}

//
// Other methods
//

- (void) chose : (int) aIndex
{
  [chosenMimeTypes addObject: [self mimeType: aIndex]];
}


@end
