/*
**  BounceWindowController.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "BounceWindowController.h"

#ifndef MACOSX
#import "BounceWindow.h"
#endif

#import "Address.h"
#import "GNUMail.h"
#import "GNUMailConstants.h"
#import "NSStringExtensions.h"
#import "PasswordPanelController.h"
#import "Utilities.h"

#import <Pantomime/InternetAddress.h>
#import <Pantomime/Message.h>
#import <Pantomime/MimeUtility.h>
#import <Pantomime/NSDataExtensions.h>
#import <Pantomime/Parser.h>
#import <Pantomime/SMTP.h>
#import <Pantomime/Sendmail.h>

#import <time.h>

@implementation BounceWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
  
#else
  BounceWindow *bounceWindow;

  bounceWindow =
    [[BounceWindow alloc] initWithContentRect: NSMakeRect(100,100,514,165)
			  styleMask: NSTitledWindowMask|NSResizableWindowMask|NSClosableWindowMask
			  backing: NSBackingStoreBuffered
			  defer: NO];
  
  self = [super initWithWindow: bounceWindow];
  
  [bounceWindow layoutWindow];
  [bounceWindow setDelegate: self];
  
  // We link our outlets
  icon = [bounceWindow icon];
  sendButton = [bounceWindow sendButton];
  addressesButton = [bounceWindow addressesButton];
  cancelButton = [bounceWindow cancelButton];

  toField = [bounceWindow toField];
  ccField = [bounceWindow ccField];
  bccField = [bounceWindow bccField];

  personalProfilePopUpButton = [bounceWindow personalProfilePopUpButton];
  transportMethodPopUpButton = [bounceWindow transportMethodPopUpButton];

  RELEASE(bounceWindow);
#endif

  [[self window] setTitle: _(@"Bounce a message")];

  // We load our personal profiles and our transport methods
  [Utilities loadPersonalProfilesInPopUpButton: personalProfilePopUpButton];
  [Utilities loadTransportMethodsInPopUpButton: transportMethodPopUpButton];
 
  [self personalProfilesSelectionHasChanged: nil];
 
  return self;
}


- (void) dealloc
{
  NSLog(@"BounceWindowController: -dealloc");

  // We remove our observer for our two notifications
  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: PersonalProfilesHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: TransportMethodsHaveChanged
    object: nil];

  RELEASE(message);
  RELEASE(ports);
  RELEASE(connection);
  
  RELEASE(passwordCache);

  [super dealloc];
}


//
// action methods
//

//
//
//
- (IBAction) send : (id) sender
{
  NSDictionary *aDictionary;

  NSMutableData *aMessageAsData;
  NSRange aRange;

  NSData *rawSource, *aData;

  // We create our mutable string
  aMessageAsData = [[NSMutableData alloc] init];
  AUTORELEASE(aMessageAsData);

  // We get the raw source of the message
  rawSource = [[self message] rawSource];
  
  // We get our headers delimiter
  aRange = [rawSource rangeOfCString: "\n\n"];

  if (aRange.length == 0)
    {
      NSRunAlertPanel(_(@"Error!"),
  		      _(@"Unable to create a valid bounced message.\nPlease, report this as a bug."),
  		      _(@"OK"),
  		      NULL,
  		      NULL);
      return;
    }
  
  // We append the content of our headers
  aData = [rawSource subdataToIndex: aRange.location + 1];

  // If we have our "From " separator, we remove it since we don't want to send
  // this by SMTP.
  if ( [aData hasCPrefix: "From "] )
    {
      NSRange r;
      
      r = [aData rangeOfCString: "\n"];

      if (r.length > 0)
	{
	  aData = [aData subdataWithRange: NSMakeRange(r.location + 1, [aData length] - r.location - 1)];
	}
    }

  // We append all our headers
  [aMessageAsData appendData: aData];

  // We append our set of new headers
  if (! [self _appendAdditionalHeadersToData: aMessageAsData] )
    {
      NSRunAlertPanel(_(@"Error!"),
  		      _(@"Please, verify that the To/Cc/Bcc fields are correct. Also, you MUST have at least one\nvalid E-Mail address in the To field."),
  		      _(@"OK"),
  		      NULL,
  		      NULL);
      return;
    }

  // We sync our popup
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];

  // We first ask for our password and we cache it (if we need too)
  aDictionary = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"]
		  objectAtIndex: [transportMethodPopUpButton indexOfSelectedItem]];
  
  if ( [aDictionary objectForKey: @"SMTP_AUTH"] &&
       [[aDictionary objectForKey: @"SMTP_AUTH"] intValue] == NSOnState &&
       [[aDictionary objectForKey: @"TRANSPORT_METHOD"] intValue] == TRANSPORT_SMTP )
    {
      [self _passwordForServerName: [aDictionary objectForKey: @"SMTP_HOST"]
	    prompt: YES];
    }


  // We append our header delimiter
  [aMessageAsData appendCString: "\n"];

  // We finally append the content of our message
  [aMessageAsData appendData: [rawSource subdataFromIndex: aRange.location + 2]];

  // We disable our UI elements
  [sendButton setEnabled: NO];
  [addressesButton setEnabled: NO];
  [personalProfilePopUpButton setEnabled: NO];
  [transportMethodPopUpButton setEnabled: NO];

  // We start our animation timer
  animation_index = 1;
  animation =  [NSTimer scheduledTimerWithTimeInterval: 0.1
			target: self
			selector: @selector(updateAnimatedIcon:)
			userInfo: nil
			repeats: YES];
  RETAIN(animation);

  //
  // We detach our thread that will send the message using a SMTP
  // connection of a mailer.
  //
  [NSThread detachNewThreadSelector: @selector(_sendMessageUsingInfoFromDictionary:)
	    toTarget: self
	    withObject: [NSDictionary dictionaryWithObjectsAndKeys: aMessageAsData, @"MESSAGE",
				      ports, @"PORTS", nil]];
}


- (IBAction) personalProfilesSelectionHasChanged: (id) sender
{
  NSArray *anArray;
  int i;

  // We synchronize our from selection from both popups
  [personalProfilePopUpButton synchronizeTitleAndSelectedItem];
  [transportMethodPopUpButton synchronizeTitleAndSelectedItem];

  anArray = [[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"];
  
  for (i = 0; i < [anArray count]; i++)
    {
      NSDictionary *aDictionary;
      
      aDictionary = [anArray objectAtIndex: i];
      
      if ( [[aDictionary objectForKey: @"PERSONAL_PROFILE"] isEqualToString: 
							      [Utilities profileNameForItemTitle:
									   [personalProfilePopUpButton titleOfSelectedItem]]] )
	{
	  [transportMethodPopUpButton selectItemAtIndex: i];
	  return;
	}
    }
  
  [transportMethodPopUpButton selectItemAtIndex: 0];
}


//
//
//
- (IBAction) updateAnimatedIcon: (id) sender
{
  if (animation_index == 9)
    {
      animation_index = 1;
    }
  
  [icon setImage: [NSImage imageNamed: [NSString stringWithFormat: @"anim-logo-%d.tiff", animation_index]]];
  [[self window] update];
  
  animation_index += 1;
}

//
// delegate methods
//

- (BOOL) windowShouldClose: (id) sender
{
  // We first verify if the animation timer is running, if it is,
  // we are currently sending a mail so we inform the user about this.
  if ( animation && [animation isValid] )
    {
      NSRunInformationalAlertPanel(_(@"Closing..."),
				   _(@"A mail is currently being sent. Please wait."),
				   _(@"OK"), 
				   NULL, 
				   NULL);
      return NO;
    }
  
  return YES;
}

- (void) windowWillClose: (NSNotification *) theNotification
{
  if ( [GNUMail lastAddressTakerWindowOnTop] == self )
    {
      [GNUMail setLastAddressTakerWindowOnTop: nil];
    }

  AUTORELEASE(self);
}


- (void) windowDidBecomeMain: (NSNotification *) theNotification;
{
  [GNUMail setLastAddressTakerWindowOnTop: self];
}


- (void) windowDidLoad
{
  NSPort *port1, *port2;

#ifdef MACOSX
  [addressesButton setTarget: [NSApp delegate]];
  [addressesButton setAction: @selector(showAddressBook:)];
#endif

  // We initialize our password cache
  passwordCache = [[NSMutableDictionary alloc] init];
  
  // We initialize our NSConnection object and our ports
  port1 = [[NSPort alloc] init];
  port2 = [[NSPort alloc] init];
  
  connection = [[NSConnection alloc] initWithReceivePort: port1
				     sendPort: port2];
  [connection setRootObject: self];

  ports = [NSArray arrayWithObjects: port2, port1, nil];

  RETAIN(ports);
  RELEASE(port1);
  RELEASE(port2);

  // We add our observer for our two notifications
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_loadPersonalProfiles)
    name: PersonalProfilesHaveChanged
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_loadTransportMethods)
    name: TransportMethodsHaveChanged
    object: nil];
}

//
// access/mutation methods
//

- (Message *) message
{
  return message;
}

- (void) setMessage: (Message *) theMessage
{
  RETAIN(theMessage);
  RELEASE(message);
  message = theMessage;
}


//
// other methods
//

- (void) authenticationFailedForServer: (NSString *) theServer
{
  NSRunAlertPanel(_(@"Error!"),
		  _(@"An error occured during the authentication with the SMTP server (%@)."),
		  _(@"OK"),
		  NULL,
		  NULL,
		  theServer);
  
  // We enable our UI elements
  [sendButton setEnabled: YES];
  [addressesButton setEnabled: YES];
  [personalProfilePopUpButton setEnabled: YES];
  [transportMethodPopUpButton setEnabled: YES];

  // We stop our animation timer
  [animation invalidate];
  DESTROY(animation);
}

- (void) messageWasSent
{
  // We stop our animation timer
  [animation invalidate];
  DESTROY(animation);

  // We close our window
  [self close];
}

- (void) errorOccuredWhenSendingMessage
{
  NSRunAlertPanel(_(@"Error!"),
		  _(@"An error occured while sending the E-Mail. It might be a network\nproblem or an error in your sending preferences."),
		  _(@"OK"),
		  NULL,
		  NULL);

  // We enable our UI elements
  [sendButton setEnabled: YES];
  [addressesButton setEnabled: YES];
  [personalProfilePopUpButton setEnabled: YES];
  [transportMethodPopUpButton setEnabled: YES];
}

//
//
//
- (void) takeToAddress: (Address *) theAddress
{
  [Utilities appendAddress: theAddress
	     toTextField: toField];
}


//
//
//
- (void) takeCcAddress: (Address *) theAddress
{
  [Utilities appendAddress: theAddress
	     toTextField: ccField];
}


//
//
//
- (void) takeBccAddress: (Address *) theAddress
{
  [Utilities appendAddress: theAddress
	     toTextField: bccField];
}

@end

//
// private methods
//
@implementation BounceWindowController (Private)

- (BOOL) _appendAdditionalHeadersToData: (NSMutableData *) theData
{
  InternetAddress *anInternetAddress;
  NSCalendarDate *aCalendarDate;
  NSDictionary *aProfile;

  NSDictionary *aLocale;

  // We first verify if at least a recipient To has been define
  if (! [[[toField stringValue] stringByTrimmingWhiteSpaces] length] )
    {
      return NO;
    }
  
  // We get our locale in English
#ifndef MACOSX
  aLocale = [NSDictionary dictionaryWithContentsOfFile: [NSBundle pathForGNUstepResource: @"English"
								  ofType: nil
								  inDirectory: @"Resources/Languages"] ];
#else
  aLocale = [NSDictionary dictionaryWithContentsOfFile: [[NSBundle bundleForClass:[NSObject class]]
							  pathForResource: @"English"
							  ofType: nil
							  inDirectory: @"Languages"] ];
#endif

  // We set the Resent-Date
#ifndef MACOSX
  tzset();
  
  aCalendarDate = [[[NSDate alloc] init] dateWithCalendarFormat:@"%a, %d %b %Y %H:%M:%S %z"
					 timeZone: [NSTimeZone timeZoneWithAbbreviation: 
								 [NSString stringWithCString: tzname[1]]] ];
#else
  aCalendarDate = [[[NSDate alloc] init] dateWithCalendarFormat:@"%a, %d %b %Y %H:%M:%S %z"
					 timeZone: [NSTimeZone systemTimeZone] ];
#endif
  
  [theData appendCFormat: @"Resent-Date: %@\n", [aCalendarDate descriptionWithLocale: aLocale]];
  
  // We get our profile
  aProfile = [[[NSUserDefaults standardUserDefaults] objectForKey: @"PERSONAL"]
	       objectForKey: [Utilities profileNameForItemTitle: [personalProfilePopUpButton titleOfSelectedItem]] ];
  
  // We set the Resent-From
  anInternetAddress = [[InternetAddress alloc] initWithPersonal: [aProfile objectForKey: @"NAME"]
					       andAddress: [aProfile objectForKey: @"EMAILADDR"]];
  
  [theData appendCString: "Resent-From: "];
  [theData appendData: [anInternetAddress dataValue]];
  [theData appendCString: "\n"];
  RELEASE(anInternetAddress);
  
  // We set the Resent-To
  [theData appendCString: "Resent-To: "];
  [theData appendData: [[toField stringValue] dataUsingEncoding: NSASCIIStringEncoding]];
  [theData appendCString: "\n"];
    
  // We set the ReSent-Cc, if we need to.
  if ( [[[ccField stringValue] stringByTrimmingWhiteSpaces] length] )
    {
      [theData appendCString: "Resent-Cc: "];
      [theData appendData: [[ccField stringValue] dataUsingEncoding: NSASCIIStringEncoding]];
      [theData appendCString: "\n"];
    }

  // We set the ReSent-Bcc, if we need to.
  if ( [[[bccField stringValue] stringByTrimmingWhiteSpaces] length] )
    {
      [theData appendCString: "Resent-Bcc: "];
      [theData appendData: [[bccField stringValue] dataUsingEncoding: NSASCIIStringEncoding]];
      [theData appendCString: "\n"];
    }

  // We set the ReSent-Message-ID
  [theData appendCString: "Resent-Message-ID: <"];
  [theData appendData: [MimeUtility generateOSID]];
  [theData appendCString: ">\n"];
  
  return YES;
}

- (void) _loadPersonalProfiles
{
  [Utilities loadPersonalProfilesInPopUpButton: personalProfilePopUpButton];
}

- (void) _loadTransportMethods
{
  [Utilities loadTransportMethodsInPopUpButton: transportMethodPopUpButton];
}


//
//
//
- (NSString *) _passwordForServerName: (NSString *) theName
			       prompt: (BOOL) aBOOL
{
  NSDictionary *aDictionary;
  NSString *password;
  
  //
  // We get our right transport method
  // We don't need to verify if it's a SMTP transport because this method isn't called
  // if it's NOT SMTP.
  //
  aDictionary = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"]
		  objectAtIndex: [transportMethodPopUpButton indexOfSelectedItem]];

  if ( aDictionary )
    {    
      password = nil;

      // We verify in the user defaults
      if ( [[aDictionary objectForKey: @"REMEMBERPASSWORD"] intValue] == NSOnState )
	{
	  password = [Utilities decryptPassword: [aDictionary objectForKey: @"SMTP_PASSWORD"] 
				withKey: theName];
	}

      // We verify in our cache
      if (! password )
	{
	  password = [passwordCache objectForKey: theName];
	}
      
      // If we must prompt for the password
      if (! password && aBOOL )
	{
	  PasswordPanelController *theController;
	  int result;
	  
	  theController = [[PasswordPanelController alloc] initWithWindowNibName: @"PasswordPanel"];
	  [[theController window] setTitle: theName];
      
	  result = [NSApp runModalForWindow: [theController window]];
	  
	  // If the user has entered a password...
	  if (result == NSRunStoppedResponse)
	    {
	      password = [theController password];
	      
	      // Let's cache this password...
	      [passwordCache setObject: password
			     forKey: theName];
	    }
	  else
	    {
	      password = nil;
	    }
	  
	  RELEASE(theController);
	}
    }
  else
    {
      password = nil;
    }
  
  return password;
}


//
//
//
- (void) _sendMessageUsingInfoFromDictionary: (NSDictionary *) theDictionary
{  
  NSConnection *serverConnection;
  NSDictionary *aDictionary;
  NSAutoreleasePool *pool;
  
  BOOL messageWasSent;

  pool = [[NSAutoreleasePool alloc] init];
  
  // We first get our connection to our main thread
  serverConnection = [[NSConnection alloc] initWithReceivePort: [[theDictionary objectForKey: @"PORTS"]
								  objectAtIndex: 0]
					   sendPort: [[theDictionary objectForKey: @"PORTS"]
						       objectAtIndex: 1]];

  messageWasSent = NO;

  // We send our message using the right delivery method!
  aDictionary = [[[NSUserDefaults standardUserDefaults] objectForKey: @"SENDING"]
		  objectAtIndex: [transportMethodPopUpButton indexOfSelectedItem]];
       
  // We send our message!
  if ( [[aDictionary objectForKey: @"TRANSPORT_METHOD"] intValue]  == TRANSPORT_SMTP  )
    {
      SMTP *aSMTP;
      
      aSMTP = [[SMTP alloc] initWithName: [aDictionary objectForKey: @"SMTP_HOST"] 
			    port: 25];
      
      // We verify if we need to authenticate ourself to the SMTP sever
      if ( [aDictionary objectForKey: @"SMTP_AUTH"] &&
	   [[aDictionary objectForKey: @"SMTP_AUTH"] intValue] == NSOnState )
	{
	  NSString *password;
	  BOOL aBOOL;

	  password = [self _passwordForServerName: [aDictionary objectForKey: @"SMTP_HOST"]
			   prompt: NO];
	  aBOOL = NO;
	  
	  if ( password )
	    {
	      aBOOL = [aSMTP authenticateWithUsername: [aDictionary objectForKey: @"SMTP_USERNAME"]
			     password: password
			     mechanism: [aDictionary objectForKey: @"SMTP_AUTH_MECHANISM"]];
	    }
	  
	  if ( !aBOOL )
	    {
	      [(id)[serverConnection rootProxy] authenticationFailedForServer: [aDictionary objectForKey: @"SMTP_HOST"]];
	      [aSMTP close];
	      RELEASE(aSMTP);
	      goto done;
	    }
	}

      if ( aSMTP )
	{
	  messageWasSent = [aSMTP sendMessageFromRawSource: [theDictionary objectForKey: @"MESSAGE"]];
	  [aSMTP close];
	}
      else
	{
	  messageWasSent = NO;
	}
    }
  else 
    {
      Sendmail *aSendmail;
      
      aSendmail = [[Sendmail alloc] initWithPathToSendmail:  [aDictionary objectForKey: @"MAILER_PATH"]];
      messageWasSent = [aSendmail sendMessageFromRawSource: [theDictionary objectForKey: @"MESSAGE"]];
    }

  if ( messageWasSent )
    {
      [(id)[serverConnection rootProxy] messageWasSent];
    }
  else 
    {
      [(id)[serverConnection rootProxy] errorOccuredWhenSendingMessage];
    }
  
 done:
  
  RELEASE(serverConnection);
  RELEASE(pool);

  [NSThread exit];
}

@end
