/*
**  AddressBook.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**          Jonathan B. Leffert <jonathan@leffert.net>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#import "AddressBook.h"

#import "Address.h"
#import "Group.h"
#import "GNUMailConstants.h"
#import "Utilities.h"

static int currentVersion = 1;

NSString *PathToAddressBook()
{
  return [NSString stringWithFormat: @"%@/%@",
		   GNUMailUserLibraryPath(), @"AddressBook"];
}

@implementation AddressBook

- (id) init
{
  self = [super init];

  [AddressBook setVersion: currentVersion];

  [self setGroups: [NSArray array]];

  return self;
}

- (void) dealloc
{
  RELEASE(allGroups);

  [super dealloc];
}

- (BOOL) synchronize
{
  return [NSArchiver archiveRootObject: self toFile: PathToAddressBook()];
}

//
// NSCoding protocol
//

- (void) encodeWithCoder: (NSCoder *) theCoder
{
  [AddressBook setVersion: currentVersion];

  [theCoder encodeObject: [self allGroups]];
}

- (id) initWithCoder: (NSCoder *) theCoder
{
  int version;
  
  version = [theCoder versionForClassName: NSStringFromClass([self class])];;
  
  self = [super init];

  // Initial version of the serialized filter object
  if ( version == 0 )
    {
      NSArray *anArray;
      Group *aGroup;
      int i;

      aGroup = [[Group alloc] initWithName: _(@"General")];

      anArray = (NSArray *)[theCoder decodeObject];
      
      for (i = 0; i < [anArray count]; i++)
	{
	  [aGroup addAddress: [anArray objectAtIndex: i]];
	}
      
      [self setGroups: [NSArray arrayWithObject: aGroup]];
      RELEASE(aGroup);
    }
  // Version 1
  else
    {
      [self setGroups: [theCoder decodeObject]];
    }

  return self;
}


//
// access/mutation methods
//

- (void) addGroup: (Group *) theGroup
{
  [allGroups addObject: theGroup];
}

- (void) removeGroup: (Group *) theGroup;
{
  [allGroups removeObject: theGroup];
}

- (NSArray *) allGroups
{
  return allGroups;
}

- (void) setGroups: (NSArray *) theGroups
{
  if ( theGroups )
    {
      NSMutableArray *aMutableArray;

      aMutableArray = [[NSMutableArray alloc] initWithArray: theGroups];
      RELEASE(allGroups);
      allGroups = aMutableArray;
    }
  else
    {
      RELEASE(allGroups);
      allGroups = nil;
    }
}


- (Group *) groupAtIndex: (int) theIndex
{
  return [[self allGroups] objectAtIndex: theIndex];
}


- (Group *) groupForName: (NSString *) theName
{
  int i;

  for (i = 0; i < [allGroups count]; i++)
    {
      if ( [[(Group *)[allGroups objectAtIndex: i] name] caseInsensitiveCompare: theName] == NSOrderedSame)
	{
	  return [allGroups objectAtIndex: i];
	}
    }
  
  return nil;
}


- (unsigned int) indexOfGroup: (Group *) theGroup
{
  return [[self allGroups] indexOfObject: theGroup];
}


- (void) sort
{
  [allGroups sortUsingSelector: @selector(compareAccordingToName:)];
}

//
// class methods
//

+ (id) addressBook
{
  return AUTORELEASE([[self alloc] init]);
}

+ (id) addressBookFromDisk
{
  return [NSUnarchiver unarchiveObjectWithFile: PathToAddressBook()];
}

@end
