#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# Labels.py
#
# DESCRIPTION:
# Implements a basic report template
#
# NOTES:
# While functional, the primary purpose of this wizard is
# as a "learning-by-example" tool.


from gnue.designer.reports.TemplateSupport import *
import string

# NOTE: It is VERY important that in any references to a
#    "<insert type here> wizard", the word "wizard" must be
#    in lower case, as many proper names such as "Foobar Wizard"
#    have been trademarked. (No, I'm not kidding :)
#


class SimpleReportTemplate(ReportTemplate):

  # The first step in our wizard.
  # The template parser will initially
  # call GetStep(FIRST_STEP).
  FIRST_STEP = '0'


  ###############
  #
  # Initialize any runtime variables
  #
  def Start(self, report, current):
    self.report = report
    self.current = current


  ###############
  #
  # Return the markup for a specific page
  #
  def GetStep(self, stepCode):

    #
    # Step #1 / Get Title, et al
    #
    if stepCode == '0':
      return   { 'title': 'Basic Report Information',
                 'content': (WizardText('Welcome to the labels wizard.'),
                             WizardText('To create your report, I need to know some basic information.\n\n'
                                             'First, what connection should this report use to connect to the database?'),
                             WizardInput('connection',label='Connection:', required=1,
                                         set=self.GetAvailableConnections()),
                             WizardText('You may be asked to login to this connection.'),
                            ),
                 'prev': None,
                 'next': '1' }


    #
    # Step #2 / Get Base Table
    #
    elif stepCode == '1':

      # TODO: If the connection changed between steps,
      # TODO: variables['table'] and variables['fields']
      # TODO: should be cleared.

      return   { 'title': 'Select Base Table/Source',
                 'content': (WizardText('Now, please select the base table for your report.'),
                             WizardInput('table', label='Base Table:', required=1, lines=5,
                                         set=self.GetAvailableSources(self.variables['connection'])), ),
                 'prev': '0',
                 'next': '2' }


    #
    # Step #3 / Get Columns to Include
    #
    elif stepCode == '2':

      # TODO: If the table changed between steps,
      # TODO: variables['fields'] should be cleared.

      return   { 'title': 'Select Fields to Include',
                 'content': (WizardText('Which fields shall I include in your report?'),
                             WizardInput('fields', label='Columns:', required=1,
                                         maxSelections=-1, orderable=1,
                                         set=self.GetAvailableFields( \
                                                self.variables['connection'],
                                                self.variables['table'])), ),
                 'prev': '1',
                 'next': None }



  ###############
  #
  # Verify contents of current step
  # Return None if no problems, otherwise
  # return a tuple of error message strings
  #
  def ValidateStep(self, stepCode):

    # The Simple wizard uses basic "required"
    # settings in the page markup, so doesn't
    # need any special validation.
    return None



  ###############
  #
  # We have all the data, so generate our report. This
  # is called after the user clicks the "Finish" button.
  # No more user input is allowed at this point.
  #
  def Finalize(self):
    self.instance.wizardName = TemplateInformation['Name']

    # We will use the table name as the basis for all our
    # object names. We will add a prefix based on the object
    # type to the table name. Capitalize the first letter and
    # strip all spaces...
    if len(self.variables['table']) == 1:
      tableKey = string.upper(self.variables['table'])
    else:
      tableKey = string.upper(self.variables['table'][0]) + \
                    string.replace(self.variables['table'][1:],' ','_')


    # Get the common report objects
    report = self.report
    sources = self.current['sources']
    layout = self.current['layout']


    # Create a single datasource based on user's input
    datasource = self.AddElement('datasource', sources,
       {  'database': self.variables['connection'],
          'table': self.variables['table'],
          'type': 'object',
          'name': 'dts%s' %  tableKey })

    schema = self.GetSourceSchema(self.variables['connection'],
                         self.variables['table'])

    # Make a map of all the field schemas we will need
    # We will not actually create the entries at this point
    # because we want to keep in the order that the user
    # specified
    fields = {}
    for field in schema.fields():
      if field.name in self.variables['fields']:
        fields[field.name] = field

    out_labels = self.AddElement('out:labels', layout, {})
    section = self.AddElement('section', out_labels, {'source': datasource.name})
    out_label = self.AddElement('out:label', section, {})
    out_region = self.AddElement('out:region', out_label, {'name':"primary"})

    # Add each field
    for name in self.variables['fields']:
      field = fields[name]
      out_line = self.AddElement('out:line', out_region, {})
      self.AddElement('field', out_line, {'name':field.name})

    # That's it... we're done.
    return 1


############
#
# Basic information about this template
#
TemplateInformation = {
    'Product': 'reports',
    'ProductStyle': 'GNUe:Reports:Labels',
    'BaseID' : 'Labels',
    'BaseClass' : SimpleReportTemplate,
    'Name' : 'Labels wizard',
    'Description' : 'Creates a labels report.',
    'Version' : '0.0.1',
    'Author' : 'The GNUe Designer Team',
    'Behavior': WIZARD
}



