#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# DirectoryServer.py
#
# DESCRIPTION:
# Class that provides a Directory of RPC Services.
#
# NOTES:
#


from gnue.common.apps import errors
from gnue.common.rpc.drivers import Base
import string

# =============================================================================
# Exceptions
# =============================================================================

class MethodNotFoundError (errors.ApplicationError):
  def __init__ (self, method):
    msg = u_("The requested method '%s' does not exist") % method
    errors.ApplicationError.__init__ (self, msg)

##############################################################################
#
# ServerAdapter
#
class DirectoryServer(Base.Server):
  def __init__(self, rpcdef, bindings, params):
    Base.Server.__init__ (self, rpcdef, bindings, params)

    # Keep a directory of all methods we expose and
    # various attributes of those methods (such as
    # "signatures" [as defined by the XML-RPC docs]
    # and help/description information.
    self.directory = {}

    self.addStandardMethods()
    # Add all the grpc defined methods
    # to our internal "service directory"
    self.mapObjects(rpcdef,bindings)

    gDebug (8, 'XML-RPC Service Directory:\n * %s' % \
            string.join(self.directory.keys(),'\n * '))
    gDebug (8,'XML-RPC Service Directory:\n%s' % \
            self.directory.keys())
    
    self._dispatchers=[self.methodDispatcher]
    
  # add some standard methods to the service directory

  def addStandardMethods(self):
    self.directory['system.listMethods'] = \
         {'signature': ('string',), 
          'binding':   self.system__listMethods,
          'help':      'Returns an array of all supported methods' }
    
    self.directory['system.methodHelp'] = \
         { 'signature': ('string','string'),
           'binding':   self.system__methodHelp,
           'help':      'Returns an string with documentation for ' \
           + 'the specified function' }
    self.directory['system.methodSignature'] = \
         { 'signature': ('string',),
           'binding':   self.system__methodSignature,
           'help':      'Returns an array containing the method signature'}
           
    
  #
  # Create an internal "service directory"
  #
  def mapObjects(self, object, bindings, parent=None):

    # For servicable objects, maintain a complete "path" for reference
    if object._type in ('RpService','RpMethod','RpObject'):
      if parent and hasattr(parent,'_path'):
        object._path = "%s.%s" % (parent._path, object.name)
      else:
        object._path = object.name


    ##    
    ## Add binding informations to the objects
    ##
    ## services are static objects and
    ## objects  are dynamic ones
    ##
    if hasattr(object,'binding'):      

      # the direct mapping
      if bindings.has_key(object.binding):
        gDebug (8, 'GNURPC Binding Information:');
        gDebug (8, ' * %s is bound to \n * %s' % \
                         (object.binding,bindings[object.binding]))

        # for services create an "static" object 
        if object._type == 'RpService':
          object._realbinding=bindings[object.binding]()
        # TODO: Clearup this "static" object

        else:
          
          # in all other cases just save the binding information
          object._realbinding=bindings[object.binding]          
        
        
      else:
        # RpObject normaly don't need binding information, because
        # they are bound dynamicly
        if object._type != 'RpObject':
          
          print o(u_("Missing Binding information. Please add binding "
                     "information for %s") % object.binding)

        
    # care for bindings in all Services
    if object._type == 'RpService':
      if hasattr(object,'_realbinding'):
        pass  # nothing to do
      else:
        if parent._type == 'RpService':
          try:
            object._realbinding=getattr(parent._realbinding,\
                                         object.name)

            gDebug (8, '* %s is bound to \n * %s' % \
                             (object._path,object._realbinding))
          except:
            tmsg = u_("GNURPC cannot bind service '%(name)s' to service "
                      "'%(destination)s'") \
                   % {'name'       : object.name,
                      'destination': parent.name}
            raise AttributeError, tmsg
        elif parent._type == 'RpGnuRpc':
          pass
        else:
          tmsg = u_("GNURPC cannot bind service '%(name)s' to service "
                    "'%(destination)s'") \
                 % {'name'       : object.name,
                    'destination': parent._type}
          raise AttributeError, tmsg
          

    # Compute binding for methods and for attributs
    # both are direct lins to the specific object
    # 
    # the dispatcher has to distinguish methods and
    # objects by testing if they are callable
    if (object._type == 'RpMethod') or \
       (object._type == 'RpAttribute'):
      
      # check for the binding
      if hasattr(object,'_realbinding'):
        bindto=object._realbinding
      else:
        if parent._type == 'RpService':
          try:
            bindto=getattr(parent._realbinding,object.name)
            gDebug (8,'* %s is bound to \n * %s' % \
                             (object._path,bindto))
          except:
            tmsg = u_("GNURPC cannot bind method/attribut '%(name)s' to "
                      "service '%(service)s'") \
                   % {'name'   : object.name,
                      'service': parent.name}
            raise AttributeError, tmsg
            pass
        else:
          bindto=None

    if object._type == 'RpMethod':    
      self.addRpMethod(object,parent,bindto)

    #
    # Add all attribute methods to our directory..
    # XML-RPC doesn't support "Attributes", so an
    # attribute is exposed as a pair of get_<name>
    # and set_<name> methods.
    #
    if object._type == 'RpAttribute':
      self.addRpAttribut(object,parent,bindto)


    # Now, map our children
    for child in object._children:
      self.mapObjects(child, bindings, object)


  ## add an method to the directory
  def addRpMethod(self,object,parent,binding):

    # Build the signature list...
    signature = []
    
    # The first element is the return type (can be None)
    if hasattr(object,'return'):
      ret=getattr(object,'return') # trick to get attribut "return"
      if len(ret):
        signature.append(ret)
      else:
        signature.append("None")
    else:
      signature.append("None")
      
    # Then add the argument datatypes...
    for arg in object._arguments:
      signature.append(arg.type)

    # Check for an description of the object
    helptext=""
    if hasattr(object,"helptext"):
      helptext=object.helptext
        
    # Add the directory entry
    self.directory[object._path] = \
                                 { 'signature': tuple(signature),
                                   'help': helptext,
                                   'binding': binding  }

  def addRpAttribut(self,object,parent,bindto):

    helptext=""
    if hasattr(object,"helptext"):
      helptext=object.helptext
      
    if not object.writeonly:
      # Add the get_* directory entry
      self.directory['%s.get_%s' % (parent._path, object.name)] = \
                                 { 'signature': (object.type,),
                                   'help': helptext,       
                                   'binding': None   } # TODO
      
    if not object.readonly:
      # Add the set_* directory entry
      self.directory['%s.set_%s' % (parent._path, object.name)] = \
                                 { 'signature': (None, object.type),
                                   'help': helptext,        
                                   'binding': None   } # TODO


  def getMethodDirEntry (self, method):
    if not self.directory.has_key (method):
      raise MethodNotFoundError, method

    gDebug (8, 'GNURPC Directory entry:\n (%s,%s,%s)' % \
                     (self.directory [method]['signature'],\
                      self.directory [method]['help'],\
                      self.directory [method]['binding']))
    
    return self.directory [method]



  ##########################################################
  #
  #  method / call dispatching
  #
  ##########################################################


  #
  # Call the requested method
  #
  def call(self, method, params):
    if self._loglevel>0:
      print _("Dispatching: "), method, params

    for i in self._dispatchers:
      (result,rtype,method,params) = i(method,params)

      if rtype:
        # check for empty results (not allowed for XMLRPC)
        if (result==None) or (result==[None]):
          gDebug (8,'Transform result None into 1')
          result=1
          return result
    
        return result

  #
  # dispatch methods
  #
  def methodDispatcher(self, method, params):

    # call to a service method or a helper call (get/set) for
    # a service attribut
    try:
      direntry = self.getMethodDirEntry(method)
      server_method = direntry['binding']
      server_attribute = None
      
      # check if it is an real method (binding -> method)      
      # or an get/set method for an attribut (binding-> attribut)
      if (type(server_method)!=type(self.call)):
        server_attribute = server_method
        server_method=None
          
      signature=direntry['signature']

      if (server_method==None) and (server_attribute==None):
        tmsg = u_("Server XML-RPC method '%s' is not bound to real method") \
               % method
        raise AttributeError, tmsg
      
    except KeyError:
      tmsg = u_("Server does not have XML-RPC procedure %s") % method
      raise AttributeError, tmsg
    
    self.checkParamsAgainstSignature(signature,params)
    
    # check if it is an real method (binding -> method)
    # or an get/set method for an attribut (binding-> attribut)

    if (server_method!=None):
            
      # call method with params
      result=server_method(*params)
      
    else:
      # simulate a get_X / set_X method
      result=self.emulate_get_set_method(self, server_attribute,
                                         method, params)
    return (result,signature[0],method,params)


  #
  # Call the requested method
  #
  def emulate_get_set_method(self, real_attribute, methodname, params):
      
    ## check wether its the set or the get method for the attribut
    mparts=string.splitfields(methodname,'.')
    mparts.reverse()
    calltype=mparts[0]
    calltype=calltype[:4]
    gDebug (8,'method %s has calling type %s' % (method,calltype))
    
    if calltype=='set_':
      # setAttribut method
      server_attribute=params[0]
      return None
    elif calltype=='get_':
      # getAttribut method
      return server_attribute
    else:
      tmsg = u_("Internal Server XML-RPC error: method type (get/set "
                "attribute) couldn't be detected (method %s)") % method
      raise AttributeError, tmsg


  #
  # check if the requested method has correct parameters and
  #   correct parameter types
  #
  def checkParamsAgainstSignature(self, method, params):
    try:
      # TODO:  Compare submitted attributs with signature
      pass
    except KeyError:
        tmsg = u_("Server XML-RPC procedure %(method)s accepts just %(attr)s "
                  "as attributs") \
               % {'method': method,
                  'attr'  : attr}
        raise AttributeError, tmsg
    



####### Introspection support

  def system__listMethods(self):
    return self.directory.keys()


  def system__methodHelp(self, method):
    if self.directory.has_key(method):
      try:
        return self.directory[method]['help']
      except KeyError:
        return u_("No help available for %s") % method
    else:
      # TODO: Is this right? If the requested method is not available?
      self.raiseException(_('InvalidMethodName'),
                          _('Requested method does not exist'))

    
  def system__methodSignature(self, method):
    if self.directory.has_key(method):
      try:
        return (self.directory[method]['signature'],)
      except KeyError:
        return None
    else:
      # TODO: Is this right? If the requested method is not available?
      self.raiseException(_('InvalidMethodName'),
                          _('Requested method does not exist'))
  
