/*
 * This file is part of Gnome Subtitles.
 * Copyright (C) 2006-2009 Pedro Castro
 *
 * Gnome Subtitles is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Gnome Subtitles is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

using GnomeSubtitles.Core.Command;
using GnomeSubtitles.Dialog;
using GnomeSubtitles.Execution;
using GnomeSubtitles.Ui;
using Gtk;
using Mono.Unix;
using SubLib.Core.Domain;
using System;
using System.IO;
using System.Text;

namespace GnomeSubtitles.Core {

/* Delegates */
public delegate void DocumentHandler (Document document);
public delegate void VideoLoadedHandler (Uri videoUri);
public delegate void TimingModeChangedHandler (TimingMode timingMode);
public delegate void BasicEventHandler ();

public class Base {
	private static Glade.XML glade = null;
	
	private static MainUi ui = null;
	private static ExecutionContext executionContext = null;
	private static EventHandlers handlers = null;
	private static CommandManager commandManager = null;
	private static Clipboards clipboards = null;
	private static DragDrop dragDrop = null;
	private static Config config = null;
	private static Dialogs dialogs = null;
	private static SpellLanguages spellLanguages = null;
	
	private static Document document = null;
	private static Uri videoUri = null;
	private static TimingMode timingMode = TimingMode.Times;

	/* Events */
	public static event BasicEventHandler InitFinished;
	public static event DocumentHandler DocumentLoaded;
	public static event DocumentHandler DocumentUnloaded;
	public static event BasicEventHandler TranslationLoaded;
	public static event BasicEventHandler TranslationUnloaded;
	public static event VideoLoadedHandler VideoLoaded;
	public static event BasicEventHandler VideoUnloaded;
	public static event TimingModeChangedHandler TimingModeChanged;
	
	
	/* Public properties */
	
	public static MainUi Ui {
		get { return ui; }
	}
	
	public static ExecutionContext ExecutionContext {
		get { return executionContext; }
	}
	
	public static EventHandlers Handlers {
		get { return handlers; }
	}
	
	public static CommandManager CommandManager {
		get { return commandManager; }
	}
	
	public static Clipboards Clipboards {
		get { return clipboards; }
	}

	public static DragDrop DragDrop {
		get { return dragDrop; }
	}
	
	public static Config Config {
		get { return config; }
	}
	
	public static Dialogs Dialogs {
		get { return dialogs; }
	}
	
	public static SpellLanguages SpellLanguages {
		get { return spellLanguages; }
	}
	
	public static Document Document {
		get { return document; }
	}
	
	public static Uri VideoUri {
		get { return videoUri; }
	}
	
	public static bool IsDocumentLoaded {
		get { return document != null; }
	}
	
	public static bool IsVideoLoaded {
		get { return videoUri != null; }
	}
	
	public static TimingMode TimingMode {
		get { return timingMode; }
		set {
			if (timingMode != value) {
				timingMode = value;
				EmitTimingModeChangedEvent();
			}		
		}
	}

	public static bool TimingModeIsFrames {
		get { return timingMode == TimingMode.Frames; }
	}

	public static bool TimingModeIsTimes {
		get { return timingMode == TimingMode.Times; }
	}


	/* Public methods */
	
	/// <summary>Runs the main GUI, after initialization.</summary> 
	public static void Run (ExecutionContext executionContext) {
		Init(executionContext);
		
		ui.Start();
		executionContext.RunApplication();
	}
	
	/// <summary>Quits the program.</summary>
	public static void Quit () {
		ui.Video.Quit();
		executionContext.QuitApplication();
	}
	
	public static void Kill () {
		clipboards.WatchPrimaryChanges = false;
		ui.Kill();
		executionContext.QuitApplication();
	}
	
	public static void NewDocument (string path) {
		if (IsDocumentLoaded)
			CloseDocument();
			
		document = new Document(path);
		EmitDocumentLoadedEvent();
		
		if (document.Subtitles.Count == 0)
			commandManager.Execute(new InsertFirstSubtitleCommand());
	}
	
	public static void OpenDocument (string path, Encoding encoding) {
		if (IsDocumentLoaded)
			CloseDocument();
	
		document = new Document(path, encoding);
		TimingMode = document.TextFile.TimingMode;
		EmitDocumentLoadedEvent();
	}
	
	public static void CloseDocument () {
		if (!IsDocumentLoaded)
			return;
		
		if (document.IsTranslationLoaded)
			CloseTranslation();
	
		document.Close();
		CommandManager.Clear();		
		EmitDocumentUnloadedEvent();
		
		document = null;
	}
	
	public static void OpenVideo (Uri uri) {
		if (uri == null)
			return;
	
		if (IsVideoLoaded)
			CloseVideo();

		ui.Video.Open(uri);
	}
	
	public static void UpdateFromVideoLoaded (Uri uri) {
		videoUri = uri;
	
		EmitVideoLoadedEvent();
	}
	
	public static void CloseVideo () {
		ui.Video.Close();
		videoUri = null;
		
		EmitVideoUnloadedEvent();
	}
	
	public static void Open (string path, Encoding encoding, Uri videoUri) {
		OpenDocument(path, encoding);
		OpenVideo(videoUri);
	}
	
	public static void OpenTranslation (string path, Encoding encoding) {
		if (document.IsTranslationLoaded)
			CloseTranslation();
	
		document.OpenTranslation(path, encoding);
		EmitTranslationLoadedEvent();
	}
	
	public static void NewTranslation () {
		if (document.IsTranslationLoaded)
			CloseTranslation();
	
		document.NewTranslation();
		EmitTranslationLoadedEvent();
	}
	
	public static void CloseTranslation () {
		document.CloseTranslation();
		EmitTranslationUnloadedEvent();
	}
	
	public static Widget GetWidget (string name) {
		return glade.GetWidget(name);
	}

	/* Private members */
	
	/// <summary>Initializes the base program structure.</summary>
	/// <remarks>Nothing is done if initialization has already occured. The core value is checked for this,
	/// if it's null then initialization hasn't occured yet.</remarks>
	private static void Init (ExecutionContext newExecutionContext) {
		if ((executionContext != null) && (executionContext.Initialized))
			throw new Exception("The Base environment was already initialized.");

		executionContext = newExecutionContext;
		executionContext.InitApplication();
		
		/* Initialize Command manager */
		commandManager = new CommandManager();
		
		/* Initialize handlers */
		handlers = new EventHandlers();
		
		/* Initialize misc */
		clipboards = new Clipboards();
		dragDrop = new DragDrop();
		config = new Config();
		dialogs = new Dialogs();
		spellLanguages = new SpellLanguages();

		/* Initialize the GUI */
		ui = new MainUi(handlers, out glade);
		clipboards.WatchPrimaryChanges = true;
		Catalog.Init(ExecutionContext.TranslationDomain, ConfigureDefines.LocaleDir);
		
		EmitInitFinishedEvent();
	}
	
	/* Event members */
	
	private static void EmitInitFinishedEvent () {
		if (InitFinished != null)
			InitFinished();
	}
	
	private static void EmitDocumentLoadedEvent () {
		if (DocumentLoaded != null)
			DocumentLoaded(document);
	}
	
	private static void EmitDocumentUnloadedEvent () {
		if (DocumentUnloaded != null)
			DocumentUnloaded(document);
	}
	
	private static void EmitTranslationLoadedEvent () {
		if (TranslationLoaded != null)
			TranslationLoaded();
	}
	
	private static void EmitTranslationUnloadedEvent () {
		if (TranslationUnloaded != null)
			TranslationUnloaded();
	}
	
	private static void EmitVideoLoadedEvent () {
		if (VideoLoaded != null)
			VideoLoaded(videoUri);
	}
	
	private static void EmitVideoUnloadedEvent () {
		if (VideoUnloaded != null)
			VideoUnloaded();
	}
	
	private static void EmitTimingModeChangedEvent () {
		if (TimingModeChanged != null)
			TimingModeChanged(timingMode);
	}

}

}
