#include <config.h>
#include "gnome-cmd-includes.h"


void debug_print (const gchar *fmt, ...)
{
#ifdef __DEBUG__
    va_list     argptr;
    char        string[1024];

    va_start (argptr,fmt);
    vsnprintf (string,1024,fmt,argptr);
    va_end (argptr);
    g_printerr("DEBUG: %s", string);
#endif //__DEBUG__
}


void warn_print (const gchar *fmt, ...)
{
    va_list     argptr;
    char        string[1024];

    va_start (argptr,fmt);
    vsnprintf (string,1024,fmt,argptr);
    va_end (argptr);
    g_printerr("WARNING: %s", string);
}


void run_command (const gchar *command)
{
	gnome_execute_shell (NULL, command);
}


void run_shell_command_indir (const gchar *dir, const gchar *command)
{
	gnome_execute_shell (dir, command);
}


void create_error_dialog (const gchar *msg, ...)
{
    va_list     argptr;
    gchar        string[1024];

    va_start (argptr, msg);
    vsprintf (string, msg, argptr);
    va_end (argptr);

    gnome_error_dialog (string);
}


void create_warning_dialog (const gchar *msg, ...)
{
    va_list     argptr;
    gchar        string[1024];

    va_start (argptr, msg);
    vsprintf (string, msg, argptr);
    va_end (argptr);

    gnome_warning_dialog (string);
}


void create_question_dialog (const gchar *msg, gpointer callback)
{
    gnome_question_dialog_modal (msg, (GnomeReplyCallback)callback, NULL);
}



gboolean string2int (const gchar *s, gint *i)
{
    int ret;
    ret = sscanf (s, "%d", i);
    return (ret == 1);
}


gboolean string2uint (const gchar *s, guint *i)
{
    int ret;
    ret = sscanf (s, "%d", i);
    return (ret == 1);
}


gboolean string2short (const gchar *s, gshort *sh)
{
    int i,ret;
    ret = sscanf (s, "%d", &i);
    *sh = i;    
    return (ret == 1);
}


gboolean string2ushort (const gchar *s, gushort *sh)
{
    int i,ret;
    ret = sscanf (s, "%d", &i);
    *sh = i;    
    return (ret == 1);
}


gboolean string2char (const gchar *s, gchar *c)
{
    int i, ret;
    ret = sscanf (s, "%d", &i);
    *c = i;
    return (ret == 1);
}


gboolean string2uchar (const gchar *s, guchar *c)
{
    int i, ret;
    ret = sscanf (s, "%d", &i);
    *c = i;
    return (ret == 1);
}


gboolean string2float (const gchar *s, gfloat *f)
{
    int ret;
    ret = sscanf (s, "%f", f);
    return (ret == 1);
}


char *int2string (int i)
{
    return g_strdup_printf ("%d", i);
}



gchar *concat_dir_file (const gchar *dir, const gchar *file)
{
	if (!(dir && file))
		return NULL;

	if (strcmp (dir, "/") == 0)
		return g_strdup_printf ("/%s", file);

	if (dir[strlen (dir) - 1] == '/')
		return g_strdup_printf ("%s%s", dir, file);

	return g_strdup_printf ("%s/%s", dir, file);
}


gchar *str_uri_basename (const gchar *uri)
{
	int i, len, last_slash=0;

	if (!uri)
		return NULL;

	len = strlen (uri);

	if (len < 2)
		return NULL;

	for ( i=0 ; i<len ; i++ )
	{
		if (uri[i] == '/')
			last_slash = i;
	}

	return gnome_vfs_unescape_string (&uri[last_slash+1], NULL);
}          


void
type2string (GnomeVFSFileType type,
			 gchar *buf,
			 guint max)
{
	char *s;
	
	switch (type) {
	case GNOME_VFS_FILE_TYPE_UNKNOWN:
		s = "?";
		break;
	case GNOME_VFS_FILE_TYPE_REGULAR:
		s = " ";
		break;
	case GNOME_VFS_FILE_TYPE_DIRECTORY:
		s = "/";
		break;
	case GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK:
		s = "@";
		break;
	case GNOME_VFS_FILE_TYPE_FIFO:
		s = "F";
		break;
	case GNOME_VFS_FILE_TYPE_SOCKET:
		s = "S";
		break;
	case GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE:
		s = "C";
		break;
	case GNOME_VFS_FILE_TYPE_BLOCK_DEVICE:
		s = "B";
		break;
	default:
		 s = "?";
	}

	g_snprintf (buf, max, "%s", s);
}


GdkPixmap *type2pixmap (GnomeVFSFileType type, gboolean big)
{
	if (big)
	{
		switch (type)
		{
			case GNOME_VFS_FILE_TYPE_UNKNOWN:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_UNKNOWN_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_REGULAR:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_REGULAR_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_DIRECTORY:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_DIRECTORY_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_SYMLINK_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_FIFO:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_FIFO_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_SOCKET:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_SOCKET_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_CHAR_DEVICE_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_BLOCK_DEVICE:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_BLOCK_DEVICE_BIG);
				break;
		}
	}
	else
	{
		switch (type)
		{
			case GNOME_VFS_FILE_TYPE_UNKNOWN:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_UNKNOWN);
				break;
			case GNOME_VFS_FILE_TYPE_REGULAR:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_REGULAR);
				break;
			case GNOME_VFS_FILE_TYPE_DIRECTORY:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_DIRECTORY);
				break;
			case GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_SYMLINK);
				break;
			case GNOME_VFS_FILE_TYPE_FIFO:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_FIFO);
				break;
			case GNOME_VFS_FILE_TYPE_SOCKET:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_SOCKET);
				break;
			case GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_CHAR_DEVICE);
				break;
			case GNOME_VFS_FILE_TYPE_BLOCK_DEVICE:
				return IMAGE_get_pixmap (PIXMAP_FILE_TYPE_BLOCK_DEVICE);
				break;
		}
	}

	return NULL;
}


GdkBitmap *type2mask (GnomeVFSFileType type, gboolean big)
{
	if (big)
	{
		switch (type)
		{
			case GNOME_VFS_FILE_TYPE_UNKNOWN:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_UNKNOWN_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_REGULAR:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_REGULAR_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_DIRECTORY:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_DIRECTORY_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_SYMLINK_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_FIFO:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_FIFO_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_SOCKET:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_SOCKET_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_CHAR_DEVICE_BIG);
				break;
			case GNOME_VFS_FILE_TYPE_BLOCK_DEVICE:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_BLOCK_DEVICE_BIG);
				break;
		}
	}
	else
	{
		switch (type)
		{
			case GNOME_VFS_FILE_TYPE_UNKNOWN:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_UNKNOWN);
				break;
			case GNOME_VFS_FILE_TYPE_REGULAR:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_REGULAR);
				break;
			case GNOME_VFS_FILE_TYPE_DIRECTORY:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_DIRECTORY);
				break;
			case GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_SYMLINK);
				break;
			case GNOME_VFS_FILE_TYPE_FIFO:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_FIFO);
				break;
			case GNOME_VFS_FILE_TYPE_SOCKET:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_SOCKET);
				break;
			case GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_CHAR_DEVICE);
				break;
			case GNOME_VFS_FILE_TYPE_BLOCK_DEVICE:
				return IMAGE_get_mask (PIXMAP_FILE_TYPE_BLOCK_DEVICE);
				break;
		}
	}

	return NULL;
}


GdkPixmap *mime2pixmap (const gchar *mime_type, GnomeVFSFileType type, gboolean big)
{
	GdkPixmap *pm;
	GdkBitmap *bm;
	
	if (IMAGE_get_mime_pixmap_and_mask (mime_type, &pm, &bm, big))
		return pm;
	else
		return type2pixmap (type, big);
}


GdkBitmap *mime2mask (const gchar *mime_type, GnomeVFSFileType type, gboolean big)
{
	GdkPixmap *pm;
	GdkBitmap *bm;
	
	if (IMAGE_get_mime_pixmap_and_mask (mime_type, &pm, &bm, big))
		return bm;
	else
		return type2mask (type, big);
}


void name2string (gchar *filename, gchar *buf, guint max)
{
	g_snprintf (buf, max, "%s", filename);
}


void perm2string (GnomeVFSFilePermissions p, gchar *buf, guint max)
{
	g_snprintf (buf, max, "%s%s%s%s%s%s%s%s%s", 
				(p & GNOME_VFS_PERM_USER_READ) ? "r" : "-",
				(p & GNOME_VFS_PERM_USER_WRITE) ? "w" : "-",
				(p & GNOME_VFS_PERM_USER_EXEC) ? "x" : "-",
				(p & GNOME_VFS_PERM_GROUP_READ) ? "r" : "-",
				(p & GNOME_VFS_PERM_GROUP_WRITE) ? "w" : "-",
				(p & GNOME_VFS_PERM_GROUP_EXEC) ? "x" : "-",
				(p & GNOME_VFS_PERM_OTHER_READ) ? "r" : "-",
				(p & GNOME_VFS_PERM_OTHER_WRITE) ? "w" : "-",
				(p & GNOME_VFS_PERM_OTHER_EXEC) ? "x" : "-");

}


void size2string (GnomeVFSFileSize size, char *buf, int max,
				  GnomeCmdSizeDispMode size_disp_mode)
{
    if (size_disp_mode == GNOME_CMD_SIZE_DISP_MODE_POWERED)
    {
        char *prefixes[5] = {"b","K","M","G","T"};
        int i=0;
        float fsize = (float)size;
        
        for ( i=0 ; i<5 ; i++ )
        {
            if (fsize > 1024)
                fsize /= 1024;
            else
                break;
        }

        if (i)
            g_snprintf (buf, max, "%.1f %s",fsize,prefixes[i]);
		else
			g_snprintf (buf, max, "%d %s", (guint)size, prefixes[0]);
    }
    else if (size_disp_mode == GNOME_CMD_SIZE_DISP_MODE_GROUPED)
    {
        int i,j,len,outlen;
        char tmp[256];
        char *out;

        sprintf (tmp, "%lld", size);
        len = strlen (tmp);
        
        if (len < 4)
		{
            strncpy (buf, tmp, max);
			return;
		}
            
        outlen = len/3 + len;
		
        if ((len/3)*3 == len)
			outlen--;
           
        out = buf;
        memset (out, '\0', max);
         
        for ( i=len-1,j=outlen-1 ; i>=0 ; i--,j-- )
        {
            if (((outlen-j)/4)*4 == outlen-j)
            {
                out[j] = ',';
                i++;
            }
            else
                out[j] = tmp[i];
        }
    }
	else
	{
		g_snprintf (buf, max, "%d", (guint)size);
	}
}


void time2string (time_t t, gchar *buf, guint max, const gchar *date_format)
{
    struct tm *lt;

	lt = localtime (&t);
    strftime (buf, max, date_format, lt);
}


static void
no_mime_app_found_error (gchar *mime_type)
{
	gchar *msg;

	msg = g_strdup_printf (_("No default application found for the mime-type %s.\nOpen the \"File types and programs\" page in the Control Center to add one."),
						   mime_type);
	gnome_error_dialog (msg);
}


void mime_exec_single (GnomeCmdFile *finfo)
{
	gchar *cmd, *tmp;
	GnomeVFSMimeApplication *app;
	
	g_return_if_fail (finfo != NULL);
	g_return_if_fail (finfo->info != NULL);

	if (!finfo->info->mime_type)
		return;

	app = gnome_vfs_mime_get_default_application (finfo->info->mime_type);

	if (!app) {
		no_mime_app_found_error (finfo->info->mime_type);
		return;
	}

	if (!app->expects_uris && !gnome_vfs_uri_is_local (gnome_cmd_file_get_uri (finfo))) {
		gchar *msg = g_strdup_printf ("Can't open this file since the default application \nfor this mime-type (%s) can only handle local files.", app->name);
		gnome_error_dialog (msg);
		g_free (msg);
		gnome_vfs_mime_application_free (app);
		return;
	}
	
	if (app->expects_uris && !gnome_vfs_uri_is_local (gnome_cmd_file_get_uri (finfo)))
		cmd = g_strdup_printf ("%s \"%s\"", app->command, gnome_cmd_file_get_uri_str (finfo));
	else
		cmd = g_strdup_printf ("%s \"%s\"", app->command, gnome_cmd_file_get_path (finfo));

	if (app->requires_terminal) {
		tmp = cmd;
		cmd = g_strdup_printf (gnome_cmd_data_get_term (), tmp);
		g_free (tmp);
	}
	
	run_command (cmd);
	g_free (cmd);
	gnome_vfs_mime_application_free (app);
}


void mime_exec_multiple (GList *files, GnomeVFSMimeApplication *app)
{
	gchar *cmd, *tmp;
	
	g_return_if_fail (files != NULL);
	g_return_if_fail (app != NULL);

	cmd = g_strdup_printf ("%s ", app->command);

	while (files) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)files->data;
		tmp = cmd;
		if (app->expects_uris && !gnome_vfs_uri_is_local (gnome_cmd_file_get_uri (finfo)))
			cmd = g_strdup_printf ("%s \"%s\" ", tmp, gnome_cmd_file_get_uri_str (finfo));
		else
			cmd = g_strdup_printf ("%s \"%s\" ", tmp, gnome_cmd_file_get_path (finfo));
		g_free (tmp);
		files = files->next;
	}

	if (app->requires_terminal) {
		tmp = cmd;
		cmd = g_strdup_printf (gnome_cmd_data_get_term (), tmp);
		g_free (tmp);
	}

	run_command (cmd);
	g_free (cmd);
}


