/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <unistd.h>
#include "gnome-cmd-includes.h"


typedef struct
{
	// Source and target uri's. The first src_uri should be transfered to the
	// first dest_uri and so on...
	GList *src_uri_list;
	GList *dest_uri_list;

	// Used for showing the progress
	GnomeCmdXferProgressWin *win;
	gint prev_phase;
	gint prev_file;
	gfloat prev_fileprog;
	gfloat prev_totalprog;

} XferData;


static void
free_xfer_data (XferData *data)
{
	GList *tmp_list;
	GnomeVFSURI *uri;

	g_list_free (data->src_uri_list);

	/* free the list with target uris */
	tmp_list = data->dest_uri_list;
	while (tmp_list)
	{
		uri = (GnomeVFSURI*)tmp_list->data;
		gnome_vfs_uri_unref (uri);
		tmp_list = tmp_list->next;
	}
	g_list_free (data->dest_uri_list);

	g_free (data);
}


static gint
xfer_callback (GnomeVFSXferProgressInfo *info, gpointer user_data)
{
	return 1;
}


static gint
async_xfer_callback (GnomeVFSAsyncHandle *handle,
					 GnomeVFSXferProgressInfo *info,
					 gpointer user_data)
{
	gfloat file_diff=0, total_diff=0;
	XferData *data = (XferData*)user_data;

	
	if (data->win && data->win->cancel_pressed)
	{
		gnome_vfs_async_cancel (handle);
		gtk_widget_hide (GTK_WIDGET (data->win));
		
		return 0;
	}

	if (info->status == GNOME_VFS_XFER_PROGRESS_STATUS_OVERWRITE)
		return GNOME_VFS_XFER_OVERWRITE_ACTION_SKIP_ALL;

	if (info->phase == GNOME_VFS_XFER_PHASE_DELETESOURCE)
	{
		if (data->win == NULL)
		{
			data->win = GNOME_CMD_XFER_PROGRESS_WIN (gnome_cmd_xfer_progress_win_new ());
			gtk_widget_show (GTK_WIDGET (data->win));
		}

		if (data->prev_phase != GNOME_VFS_XFER_PHASE_DELETESOURCE)
		{
			gnome_cmd_xfer_progress_win_set_action (data->win, _("deleting source..."));
			data->prev_file = -1;
		}

		if (data->prev_file != (gint)info->file_index)
		{
			gchar *fn = str_uri_basename (info->source_name);
			gchar *msg = g_strdup_printf (
				_("file %d of %d \"%s\""),
				(guint)info->file_index,
				(guint)info->files_total,
				fn);

			gnome_cmd_xfer_progress_win_set_msg (data->win, msg);

			data->prev_file = (gint)info->file_index;
			
			if (msg) g_free (msg);
			if (fn) g_free (fn);
		}
	}
	
	if (info->phase == GNOME_VFS_XFER_PHASE_COPYING)
	{
		if (data->win == NULL)
		{
			data->win = GNOME_CMD_XFER_PROGRESS_WIN (gnome_cmd_xfer_progress_win_new ());
			gtk_widget_show (GTK_WIDGET (data->win));
		}
		
		if (data->prev_phase != GNOME_VFS_XFER_PHASE_COPYING)
		{
			gnome_cmd_xfer_progress_win_set_action (data->win, _("copying..."));
			data->prev_file = -1;
		}
		
		if (data->prev_file != (gint)info->file_index)
		{
			gchar *fn = str_uri_basename (info->source_name);
			gchar *msg = g_strdup_printf (
				_("file %d of %d \"%s\""),
				(guint)info->file_index,
				(guint)info->files_total,
				fn);
			
			gnome_cmd_xfer_progress_win_set_msg (data->win, msg);

			data->prev_fileprog = (gfloat)0.0;
			data->prev_file = (gint)info->file_index;
			
			if (msg) g_free (msg);
			if (fn) g_free (fn);
		}
		
		if ((gint)info->file_size > 0)
		{
			gfloat file_prog;
		
			file_prog = (gfloat)info->bytes_copied / (gfloat)info->file_size;
			file_diff = file_prog - data->prev_fileprog;

			if (file_diff > (gfloat)0.01 && file_prog >= 0.00 && file_prog <= 1.00)
			{
				gnome_cmd_xfer_progress_win_set_file_progress (
					data->win, info->bytes_copied, info->file_size);
				data->prev_fileprog = file_prog;
				while (gtk_events_pending ())
					gtk_main_iteration_do (FALSE);
			}
		}

		if ((gint)info->bytes_total > 0)
		{
			gfloat total_prog;
			
			total_prog = (gfloat)info->total_bytes_copied / (gfloat)info->bytes_total;
			total_diff = total_prog - data->prev_totalprog;

			if (total_diff > (float)0.01 && total_prog >= 0.0 && total_prog <= 1.0)
				gnome_cmd_xfer_progress_win_set_total_progress (
					data->win, info->total_bytes_copied, info->bytes_total);
		}
		
	}

	if (info->phase == GNOME_VFS_XFER_PHASE_COMPLETED)
	{
		if (data->win)
		{
			/* Sleep for 100 ms so that the window gets a chance to be showed before we hide it
			   when copying very small files. This is a test-workaround for the problem of
			   progress-windows not getting hidden when xfer is complete. */
			usleep (100000);
			gtk_widget_hide (GTK_WIDGET (data->win));

			data->win = NULL;
		}
		
		free_xfer_data (data);
	}
	
	data->prev_phase = info->phase;
	
	return 1;
}


static gboolean
uri_is_parent_to_dir (GnomeVFSURI *uri, GnomeCmdDir *dir)
{
	gchar *uri_str;
	const gchar *dir_str;

	uri_str = gnome_vfs_uri_to_string (uri, 0);
	dir_str = gnome_cmd_dir_get_uri_str (dir);

	return (strncmp (uri_str, dir_str, strlen (uri_str)) == 0);
}


static gboolean
file_is_already_in_dir (GnomeVFSURI *uri, GnomeCmdDir *dir) {
	gchar *uri_str;
	const gchar *dir_str;

	uri_str = gnome_vfs_uri_to_string (uri, 0);
	dir_str = gnome_cmd_dir_get_uri_str (dir);
	
	return (strncmp (uri_str, dir_str, strlen (dir_str)) == 0);
}


void
gnome_cmd_xfer_uris_start (GList *src_uri_list,
						   GnomeCmdDir *to_dir,
						   gchar *dest_fn,
						   GnomeVFSXferOptions xferOptions,
						   GnomeVFSXferOverwriteMode xferOverwriteMode)
{
	GnomeVFSURI *src_uri, *dest_uri;
	GnomeVFSResult result;
	gint num_files;
	GnomeVFSAsyncHandle *handle;
	XferData *data;
	GList *tmp;

	g_return_if_fail (src_uri_list != NULL);
	g_return_if_fail (to_dir != NULL);

	/*
	 * Sanity check
	 */
	tmp = src_uri_list;
	while (tmp) {
		src_uri = (GnomeVFSURI*)tmp->data;
		if (uri_is_parent_to_dir (src_uri, to_dir)) {
			gnome_error_dialog (_("Copying a directory into it self is a bad idea.\nThe whole operation was canceled."));
			return;
		}
		if (file_is_already_in_dir (src_uri, to_dir)) {
			g_printerr (_("Copying a file to the same directory as it's already in, is not permitted\n"));
			return;
		}
		
		tmp = tmp->next;
	}	
	
	data = g_new (XferData, 1);

	data->src_uri_list = src_uri_list;
	data->dest_uri_list = NULL;
	data->win = NULL;
	data->prev_phase = -1;
	data->prev_file = -1;
	data->prev_fileprog = (gfloat)0.00;
	data->prev_totalprog = (gfloat)0.00;

	num_files = g_list_length (src_uri_list);

	if (g_list_length (src_uri_list) == 1 && dest_fn != NULL) {
		dest_uri = gnome_cmd_dir_get_file_uri (to_dir, dest_fn);

		data->dest_uri_list = g_list_append (data->dest_uri_list, dest_uri);
	}
	else {
		while (src_uri_list) {
			gchar *basename;
			src_uri = (GnomeVFSURI*)src_uri_list->data;
			basename = gnome_vfs_unescape_string (
				gnome_vfs_uri_get_basename (src_uri), 0);

			dest_uri = gnome_cmd_dir_get_file_uri ( 
				to_dir, basename);
			g_free (basename);
		
			data->dest_uri_list = g_list_append (data->dest_uri_list, dest_uri);
				
			src_uri_list = src_uri_list->next;
		}
	}

	if (dest_fn != NULL)
		g_free (dest_fn);
	
	/* start the transfer */
	result = gnome_vfs_async_xfer (
		&handle,
		data->src_uri_list,
		data->dest_uri_list,
		xferOptions,
		GNOME_VFS_XFER_ERROR_MODE_ABORT,
		xferOverwriteMode,
		async_xfer_callback,
		data,
		xfer_callback,
		data);
}


void
gnome_cmd_xfer_start (GList *src_files,
					  GnomeCmdDir *to_dir,
					  gchar *dest_fn,
					  GnomeVFSXferOptions xferOptions,
					  GnomeVFSXferOverwriteMode xferOverwriteMode)
{
	GList *tmp = src_files;
	GList *src_uri_list = NULL;

	while (tmp) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		src_uri_list = g_list_append (
			src_uri_list,
			gnome_cmd_file_get_uri (finfo));
		
		tmp = tmp->next;
	}

	gnome_cmd_xfer_uris_start (src_uri_list,
							   to_dir,
							   dest_fn,
							   xferOptions,
							   xferOverwriteMode);
}
