/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <sys/types.h>
#include <regex.h>
#include "gnome-cmd-includes.h"

static GnomeDialogClass *parent_class = NULL;

typedef struct
{
	gchar *name_pattern;           // the pattern that filenames should match to end up in the file-list
	gchar *content_pattern;        // the pattern that the content of a file should match to end up int the file-list.
	gchar *dir;                    // the current dir of the search routine.
	gchar *text;                   

	regex_t *name_regex;            
	regex_t *content_regex;

	gboolean content_search;       // should we do content search?
	gint matches;                  // the number of matching files
	GList *match_list;             // a list with all matching files.
	gint context_id;               // the context id of the statusbar
	GnomeCmdSearchDialog *dialog;

	gboolean stopped;              // stopps the search routine if set to TRUE. This is done by the stop_button
	gboolean recurse;              // should we recurse or just search in the selected directory? 
	
} SearchData;

struct _GnomeCmdSearchDialogPrivate
{
	SearchData *data;              // holds data needed by the search routines
	gint selected_row;             // the selected row in the file-list

	GnomeVFSURI *baseuri;
	GtkWidget *popup_menu;
	GtkWidget *pattern_combo;
	GtkWidget *pattern_entry;
	GtkWidget *dir_browser;
	GtkWidget *dir_entry;
	GtkWidget *find_text_combo;
	GtkWidget *find_text_entry;
	GtkWidget *find_text_check;
	GtkWidget *result_clist;
	GtkWidget *statusbar;
	GtkWidget *view_button;
	GtkWidget *recurse_check;
};



static gint
file_popup_handler(GtkWidget *widget, GdkEvent *event)
{
	GtkMenu *menu;
	GdkEventButton *event_button;

	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_MENU (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	/* The "widget" is the menu that was supplied when 
	 * gtk_signal_connect_object was called.
	 */
	menu = GTK_MENU (widget);

	if (event->type == GDK_BUTTON_PRESS)
	{
		event_button = (GdkEventButton *) event;
		if (event_button->button == 3)
		{
			gtk_menu_popup (menu, NULL, NULL, NULL, NULL, 
							event_button->button, event_button->time);
			return TRUE;
		}
	}

	return FALSE;
}


/**
 * Clears the list with matching files.
 *
 */
static void
clear_match_list (GList *list)
{
	while (list)
	{
		GnomeCmdFile *finfo = (GnomeCmdFile*)list->data;
		gnome_cmd_file_unref (finfo);
		list = list->next;
	}

	g_list_free (list);
}


/**
 * Puts a string in the statusbar.
 *
 */
static void
set_statusmsg (SearchData *data, gchar *msg)
{
	gtk_statusbar_push (GTK_STATUSBAR (data->dialog->priv->statusbar),
						data->context_id, msg);
}


/**
 * Loads a file and returns the content.
 * FIXME: Do this with GnomeVFS calls instead.
 */
static gchar*
load_file (GnomeCmdFile *finfo)
{
	gchar *buf;
	const gchar *path;
	FILE* fd;
	gint ret;
	
	g_return_val_if_fail (finfo != NULL, NULL);	
	g_return_val_if_fail (finfo->info != NULL, NULL);
	
	buf = malloc (finfo->info->size+1);
	path = gnome_cmd_file_get_path (finfo);	
	fd = fopen (path, "r");

	if (fd == NULL)
	{
		warn_print (_("Failed to open file %s\n"), path);
		return NULL;
	}

	ret = fread ((void*)buf, 1, finfo->info->size, fd);
	if (ret != finfo->info->size)
	{
		warn_print (_("Failed to read file %s\n"), finfo->path);
		fclose (fd);
		return NULL;
	}

	buf[finfo->info->size] = '\0';
	
	fclose (fd);

	return buf;
}


/**
 * Determinates if the name of a file matches an regexp
 *
 */
static gboolean
name_matches (gchar *name, SearchData *data)
{
	static regmatch_t match;
	return (regexec (data->name_regex, name, 1, &match, 0) == 0);
}


/**
 * Determinates if the content of a file matches an regexp
 *
 */
static gboolean
content_matches (GnomeCmdFile *finfo, SearchData *data)
{
	gint ret;
	static regmatch_t match;
	gchar *buf = load_file (finfo);
	ret = regexec (data->content_regex, buf, 1, &match, 0);
	debug_print ("Matching \"%s\" against \"%s\" ret %d\n",
				data->content_pattern,
				buf,
				ret);
	g_free (buf);
	return (ret == 0);
}


/**
 * Searches a given directory for files that matches the criteria given by data.
 *
 */
static void
search_dir_r (GnomeCmdDir *dir, SearchData *data)
{
	GList *tmp, *list;
	gchar *msg;
	const gchar *path;

	
	/* update the statusbar with the currently searched directory */
	path = gnome_cmd_dir_get_path (dir);
	msg = g_strdup_printf ("searching in %s", path);
	set_statusmsg (data, msg);
	g_free (msg);
	

	/* do some eventhandling here so that the buttons in the dialog responds */
	while (gtk_events_pending ())
	{
		if (gtk_main_iteration ())
			exit (0);
	}

	
	list = tmp = gnome_cmd_dir_list_files (dir);
	
	/* Let's iterate through all files */
	while (tmp)
	{
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;

		/* If the stopbutton was pressed let's abort here */
		if (data->stopped)
			return;

		/* If the current file is a directory lets continue our recursion */
		if (finfo->info->type == GNOME_VFS_FILE_TYPE_DIRECTORY && data->recurse)
		{
			/* we dont want to go backwards or follow symlinks */
			if (strcmp (finfo->info->name, ".") != 0
				&& strcmp (finfo->info->name, "..") != 0
				&& finfo->info->flags != GNOME_VFS_FILE_FLAGS_SYMLINK)
			{
				const gchar *dir_uri_str = gnome_cmd_file_get_uri_str (finfo);
				GnomeCmdDir *new_dir = dir_pool_get (dir_uri_str);
				search_dir_r (new_dir, data);
			}
		}
		/* if the file is a regular one it might match the search criteria */
		else if (finfo->info->type == GNOME_VFS_FILE_TYPE_REGULAR)
		{
			gint row;
			gchar *text[4];
			gchar *size;

			/* if the name doesn't match lets go to the next file */
			if (!name_matches (finfo->info->name, data))
				goto next;

			gnome_cmd_file_ref (finfo);
				
			/* if the user wants to we should do some content matching here */
			if (data->content_search)
			{
				if (!content_matches (finfo, data))
				{
					gnome_cmd_file_unref (finfo);
					goto next;
				}
			}

			/* the file matches the search criteria, lets add it to the list */
			if (gnome_cmd_data_get_layout () == GNOME_CMD_LAYOUT_TEXT)
				text[0] = (gchar*)gnome_cmd_file_get_type_string (finfo);
			else
				text[0] = NULL;
		
			text[1] = finfo->info->name;
			text[2] = (gchar*)gnome_cmd_dir_get_path (dir); 
			size = g_strdup_printf ("%d", (gint)finfo->info->size);
			text[3] = size;				

			/* show the match in the list */
			row = gtk_clist_append (GTK_CLIST (data->dialog->priv->result_clist), text);
			gtk_clist_set_row_data (GTK_CLIST (data->dialog->priv->result_clist), row, finfo);
				
			if (gnome_cmd_data_get_layout () == GNOME_CMD_LAYOUT_ICONS)
			{
				gtk_clist_set_pixmap (GTK_CLIST (data->dialog->priv->result_clist),
									  row, 0,
									  gnome_cmd_file_get_type_pixmap_small (finfo),
									  gnome_cmd_file_get_type_mask_small (finfo));
			}
				
			g_free (size);

			/* Add the file to a list so that we can free it later */
			data->match_list = g_list_append (data->match_list, finfo);

			/* count the match */
			data->matches++;
		}
	  next:
		tmp = tmp->next;
	}
}


/**
 * The user has clicked on the search button 
 *
 */
static void
on_search (GtkButton *button, GnomeCmdSearchDialog *dialog)
{
	gchar *msg;
	SearchData *data;
	GnomeCmdDir *start_dir;
	GnomeVFSURI *dir_uri;

	/* if data is null this is the first search since the dialog was created */
	if (!dialog->priv->data)
	{
		data = g_new (SearchData, 1);
		dialog->priv->data = data;
	}
	else
	{
		/* otherwise we should free some mem from the previous search */
		data = dialog->priv->data;
		clear_match_list (data->match_list);
	}
	
	dialog->priv->selected_row = -1;
	data->dialog = dialog;
	data->name_pattern = gtk_entry_get_text (GTK_ENTRY (dialog->priv->pattern_entry));
	data->content_pattern = gtk_entry_get_text (GTK_ENTRY (dialog->priv->find_text_entry));
	data->dir = gtk_entry_get_text (GTK_ENTRY (dialog->priv->dir_entry));
	data->text = gtk_entry_get_text (GTK_ENTRY (dialog->priv->find_text_entry));
	data->context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (data->dialog->priv->statusbar), "info");
	data->content_search = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->find_text_check));
	data->matches = 0;
	data->match_list = NULL;
	data->stopped = FALSE;
	data->recurse = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->recurse_check));

	gtk_clist_clear (GTK_CLIST (dialog->priv->result_clist));

	/* create an re for filename matching */
	data->name_regex = g_new (regex_t, 1);
	regcomp (data->name_regex, data->name_pattern, 0);


	/* if we're going to search through files create an re for that too */
	if (data->content_search)
	{
		data->content_regex = g_new (regex_t, 1);
		regcomp (data->content_regex, data->content_pattern, 0);
	}

	
	/* start the search */
	dir_uri = gnome_vfs_uri_append_path (dialog->priv->baseuri, data->dir);
	data->dir = gnome_vfs_uri_to_string (dir_uri, 0);
	start_dir = dir_pool_get (data->dir);
	search_dir_r (start_dir, data);
	gnome_vfs_uri_unref (dir_uri);
	g_free (data->dir);

	
	/* print the number of files found to the statusbar when finished */
	if (data->stopped)
		msg = g_strdup_printf (_("Found %d matches before I was stopped"), data->matches);
	else
		msg = g_strdup_printf (_("Found %d matches"), data->matches);	
	set_statusmsg (data, msg);
	g_free (msg);

	
	/* free regexps */
	regfree (data->name_regex);
	g_free (data->name_regex);
	if (data->content_search)
	{
		regfree (data->content_regex);
		g_free (data->content_regex);
	}
}


/**
 * The user has clicked on the close button 
 *
 */
static void
on_close (GtkButton *button, GnomeCmdSearchDialog *dialog)
{
	gtk_widget_hide (GTK_WIDGET (dialog));
}


/**
 * The user has clicked on the stop button 
 *
 */
static void
on_stop (GtkButton *button, GnomeCmdSearchDialog *dialog)
{
	g_return_if_fail (dialog != NULL);
	g_return_if_fail (dialog->priv != NULL);
	g_return_if_fail (dialog->priv->data != NULL);
	
	dialog->priv->data->stopped = TRUE;
	debug_print ("Stopping search...\n");
}


/**
 * The user has clicked on the view button 
 *
 */
static void
on_view (GtkButton *button, GnomeCmdSearchDialog *dialog)
{
	GnomeCmdFile *finfo;

	if (dialog->priv->selected_row == -1) {
		g_printerr (_("No file selected\n"));
		return;
	}

	finfo = (GnomeCmdFile*)gtk_clist_get_row_data (
		GTK_CLIST (dialog->priv->result_clist), dialog->priv->selected_row);

	gnome_cmd_file_view (finfo);
}


/**
 * The user has clicked on the "search by content" checkbutton.
 *
 */
static void
find_text_toggled (GtkToggleButton *togglebutton, GnomeCmdSearchDialog *dialog)
{
	if (gtk_toggle_button_get_active (togglebutton))
		gtk_widget_set_sensitive (dialog->priv->find_text_combo, TRUE);
	else
		gtk_widget_set_sensitive (dialog->priv->find_text_combo, FALSE);
}


/**
 * The user has selected a file in the list.
 *
 */
static void
on_row_selected (GtkCList *clist, gint row, gint col,
				 GdkEventButton *event, GnomeCmdSearchDialog *dialog)
{
	if (!event)
		return;
	
	dialog->priv->selected_row = row;

	if (event->type == GDK_2BUTTON_PRESS)
	{
		GnomeCmdFile *finfo = (GnomeCmdFile*)gtk_clist_get_row_data (clist, row);
		if (finfo && finfo->dir)
		{
			GnomeCmdFileSelector *fs = gnome_cmd_main_win_get_active_fs (main_win);
			gnome_cmd_file_selector_set_directory (fs, finfo->dir);
			gnome_cmd_file_selector_focus_file (fs, finfo->info->name);
			gtk_widget_hide (GTK_WIDGET(dialog));
		}
			
	}	
}


/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdSearchDialog *dialog = GNOME_CMD_SEARCH_DIALOG (object);

	/* free memory from the previous search if necessary */
	if (dialog->priv->data)
	{
		clear_match_list (dialog->priv->data->match_list);
		g_free (dialog->priv->data);
	}
	
	g_free (dialog->priv);

	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdSearchDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_dialog_get_type ());

	object_class->destroy = destroy;

	widget_class->map = map;
}

static void
init (GnomeCmdSearchDialog *dialog)
{
	GtkWidget *window;
	GtkWidget *vbox1;
	GtkWidget *table1, *table2;
	GtkWidget *label5;
	GtkWidget *label6;
	GtkWidget *search_button;
	GtkWidget *close_button;
	GtkWidget *stop_button;
	GtkWidget *label1;
	GtkWidget *label2;
	GtkWidget *label3;
	GtkWidget *label4;
	GtkWidget *sw;
	GtkWidget *notebook;
	GList     *buttons;

	dialog->priv = g_new (GnomeCmdSearchDialogPrivate, 1);
	dialog->priv->data = NULL;

	
	window = GTK_WIDGET (dialog);
	gtk_object_set_data (GTK_OBJECT (window), "window", window);
	gtk_window_set_title (GTK_WINDOW (window), _("Search..."));
	gtk_window_set_policy (GTK_WINDOW (window), FALSE, TRUE, FALSE);
	gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);

	vbox1 = GNOME_DIALOG (window)->vbox;

	notebook = gtk_notebook_new ();
	gtk_widget_ref (notebook);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "notebook",
							  notebook, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (notebook);
	gtk_box_pack_start (GTK_BOX (vbox1), notebook,
						TRUE, TRUE, 0);


	/*
	 * The "Basic" tab
	 */
	table1 = gtk_table_new (3, 2, FALSE);
	gtk_widget_ref (table1);
	gtk_object_set_data_full (GTK_OBJECT (window), "table1", table1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (table1);
	gtk_container_set_border_width (GTK_CONTAINER (table1), 6);
	gtk_table_set_row_spacings (GTK_TABLE (table1), 13);
	gtk_table_set_col_spacings (GTK_TABLE (table1), 2);

	label5 = gtk_label_new (_("Search for: "));
	gtk_widget_ref (label5);
	gtk_object_set_data_full (GTK_OBJECT (window), "label5", label5,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label5);
	gtk_table_attach (GTK_TABLE (table1), label5, 0, 1, 0, 1,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label5), 0, 0.5);

	label6 = gtk_label_new (_("Search in: "));
	gtk_widget_ref (label6);
	gtk_object_set_data_full (GTK_OBJECT (window), "label6", label6,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label6);
	gtk_table_attach (GTK_TABLE (table1), label6, 0, 1, 1, 2,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label6), 0, 0.5);

	dialog->priv->pattern_combo = gtk_combo_new ();
	gtk_widget_ref (dialog->priv->pattern_combo);
	gtk_object_set_data_full (GTK_OBJECT (window), "pattern_combo", dialog->priv->pattern_combo,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->pattern_combo);
	gtk_table_attach (GTK_TABLE (table1), dialog->priv->pattern_combo, 1, 2, 0, 1,
					  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	dialog->priv->pattern_entry = GTK_COMBO (dialog->priv->pattern_combo)->entry;
	gtk_widget_ref (dialog->priv->pattern_entry);
	gtk_object_set_data_full (GTK_OBJECT (window), "pattern_entry", dialog->priv->pattern_entry,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->pattern_entry);
    gnome_dialog_editable_enters (GNOME_DIALOG (dialog), GTK_EDITABLE (dialog->priv->pattern_entry));

	dialog->priv->dir_browser = gnome_file_entry_new (NULL, NULL);
	gtk_widget_ref (dialog->priv->dir_browser);
	gtk_object_set_data_full (GTK_OBJECT (window), "dir_browser", dialog->priv->dir_browser,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->dir_browser);
	gtk_table_attach (GTK_TABLE (table1), dialog->priv->dir_browser, 1, 2, 1, 2,
					  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	dialog->priv->dir_entry =
		gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (dialog->priv->dir_browser));
	gtk_widget_ref (dialog->priv->dir_entry);
	gtk_object_set_data_full (GTK_OBJECT (window), "dir_entry", dialog->priv->dir_entry,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->dir_entry);
    gnome_dialog_editable_enters (GNOME_DIALOG (dialog), GTK_EDITABLE (dialog->priv->dir_entry));

	
	dialog->priv->recurse_check = gtk_check_button_new_with_label (_("Recurse through subdirectories"));
	gtk_widget_ref (dialog->priv->recurse_check);
	gtk_object_set_data_full (GTK_OBJECT (window),
							  "recurse_check", dialog->priv->recurse_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->recurse_check);
	gtk_table_attach (GTK_TABLE (table1), dialog->priv->recurse_check, 1, 2, 2, 3,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);


	/*
	 * The "Advanced" tab
	 */
	table2 = gtk_table_new (1, 2, FALSE);
	gtk_widget_ref (table2);
	gtk_object_set_data_full (GTK_OBJECT (window), "table2", table2,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (table2);
	gtk_container_set_border_width (GTK_CONTAINER (table2), 6);
	gtk_table_set_row_spacings (GTK_TABLE (table2), 13);
	gtk_table_set_col_spacings (GTK_TABLE (table2), 2);


	dialog->priv->find_text_check = gtk_check_button_new_with_label (_("Find text: "));
	gtk_widget_ref (dialog->priv->find_text_check);
	gtk_object_set_data_full (GTK_OBJECT (window),
							  "find_text_check", dialog->priv->find_text_check,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->find_text_check);
	gtk_table_attach (GTK_TABLE (table2), dialog->priv->find_text_check, 0, 1, 0, 1,
					  (GtkAttachOptions) (GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);

	dialog->priv->find_text_combo = gtk_combo_new ();
	gtk_widget_ref (dialog->priv->find_text_combo);
	gtk_object_set_data_full (GTK_OBJECT (window),
							  "find_text_combo", dialog->priv->find_text_combo,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->find_text_combo);
	gtk_table_attach (GTK_TABLE (table2), dialog->priv->find_text_combo, 1, 2, 0, 1,
					  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
					  (GtkAttachOptions) (0), 0, 0);
	gtk_widget_set_sensitive (dialog->priv->find_text_combo, FALSE);

	dialog->priv->find_text_entry = GTK_COMBO (dialog->priv->find_text_combo)->entry;
	gtk_widget_ref (dialog->priv->find_text_entry);
	gtk_object_set_data_full (GTK_OBJECT (window),
							  "find_text_entry", dialog->priv->find_text_entry,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->find_text_entry);
    gnome_dialog_editable_enters (GNOME_DIALOG (dialog), GTK_EDITABLE (dialog->priv->find_text_entry));

	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (window), "Search", GNOME_STOCK_PIXMAP_SEARCH);
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (window), "Stop", GNOME_STOCK_PIXMAP_STOP);
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (window), "View", GNOME_STOCK_PIXMAP_JUMP_TO);
	gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (window), "Close", GNOME_STOCK_PIXMAP_CLOSE);
	

	buttons = g_list_first (GNOME_DIALOG (window)->buttons);
	search_button = GTK_WIDGET (buttons->data);
	buttons = buttons->next;
	stop_button = GTK_WIDGET (buttons->data);
	buttons = buttons->next;
	dialog->priv->view_button = GTK_WIDGET (buttons->data);
	buttons = buttons->next;
	close_button = GTK_WIDGET (buttons->data);
	
	gnome_dialog_set_default (GNOME_DIALOG (dialog), 0);
	
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_ref (sw);
	gtk_object_set_data_full (GTK_OBJECT (window), "sw", sw,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (sw);
	gtk_box_pack_start (GTK_BOX (vbox1), sw, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
									GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	
	dialog->priv->result_clist = gtk_clist_new (4);
	gtk_widget_ref (dialog->priv->result_clist);
	gtk_object_set_data_full (GTK_OBJECT (window), "result_clist", dialog->priv->result_clist,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_set_usize (dialog->priv->result_clist, -1, 200);
	gtk_widget_show (dialog->priv->result_clist);
	gtk_container_add (GTK_CONTAINER (sw), dialog->priv->result_clist);
	gtk_container_set_border_width (GTK_CONTAINER (dialog->priv->result_clist), 4);
	gtk_clist_set_column_width (GTK_CLIST (dialog->priv->result_clist), 0, 21);
	gtk_clist_set_column_width (GTK_CLIST (dialog->priv->result_clist), 1, 100);
	gtk_clist_set_column_width (GTK_CLIST (dialog->priv->result_clist), 2, 206);
	gtk_clist_set_column_width (GTK_CLIST (dialog->priv->result_clist), 3, 80);
	gtk_clist_column_titles_show (GTK_CLIST (dialog->priv->result_clist));

	label1 = gtk_label_new ("");
	gtk_widget_ref (label1);
	gtk_object_set_data_full (GTK_OBJECT (window), "label1", label1,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label1);
	gtk_clist_set_column_widget (GTK_CLIST (dialog->priv->result_clist), 0, label1);

	label2 = gtk_label_new (_("name"));
	gtk_widget_ref (label2);
	gtk_object_set_data_full (GTK_OBJECT (window), "label2", label2,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label2);
	gtk_clist_set_column_widget (GTK_CLIST (dialog->priv->result_clist), 1, label2);

	label3 = gtk_label_new (_("dir"));
	gtk_widget_ref (label3);
	gtk_object_set_data_full (GTK_OBJECT (window), "label3", label3,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label3);
	gtk_clist_set_column_widget (GTK_CLIST (dialog->priv->result_clist), 2, label3);

	label4 = gtk_label_new (_("size"));
	gtk_widget_ref (label4);
	gtk_object_set_data_full (GTK_OBJECT (window), "label4", label4,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (label4);
	gtk_clist_set_column_widget (GTK_CLIST (dialog->priv->result_clist), 3, label4);

	dialog->priv->statusbar = gtk_statusbar_new ();
	gtk_widget_ref (dialog->priv->statusbar);
	gtk_object_set_data_full (GTK_OBJECT (window), "statusbar", dialog->priv->statusbar,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->priv->statusbar);
	gtk_box_pack_start (GTK_BOX (vbox1), dialog->priv->statusbar, FALSE, FALSE, 0);


	gtk_container_add (GTK_CONTAINER (notebook), table1);
	gtk_container_add (GTK_CONTAINER (notebook), table2);

	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 0),
		gtk_label_new (_("Basic")));

	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), 1),
		gtk_label_new (_("Advanced")));


	/* create the popup menu */

	gtk_signal_connect (GTK_OBJECT (search_button), "clicked",
						GTK_SIGNAL_FUNC (on_search), dialog);
	gtk_signal_connect (GTK_OBJECT (stop_button), "clicked",
						GTK_SIGNAL_FUNC (on_stop), dialog);
	gtk_signal_connect (GTK_OBJECT (dialog->priv->view_button), "clicked",
						GTK_SIGNAL_FUNC (on_view), dialog);
	gtk_signal_connect (GTK_OBJECT (close_button), "clicked",
						GTK_SIGNAL_FUNC (on_close), dialog);

	gtk_signal_connect (GTK_OBJECT (dialog->priv->find_text_check), "toggled",
						GTK_SIGNAL_FUNC (find_text_toggled), dialog);
	gtk_signal_connect (GTK_OBJECT (dialog->priv->result_clist), "select-row",
						GTK_SIGNAL_FUNC (on_row_selected), dialog);

	gtk_widget_grab_focus (dialog->priv->pattern_entry);
}






/***********************************
 * Public functions
 ***********************************/

GtkType
gnome_cmd_search_dialog_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdSearchDialog",
			sizeof (GnomeCmdSearchDialog),
			sizeof (GnomeCmdSearchDialogClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gnome_dialog_get_type (), &dlg_info);
	}
	return dlg_type;
}


GtkWidget*
gnome_cmd_search_dialog_new (GnomeVFSURI *baseuri, GnomeCmdDir *default_dir)
{
	const gchar *path;
	GnomeCmdSearchDialog *dialog;

	dialog = gtk_type_new (gnome_cmd_search_dialog_get_type ());

	path = gnome_cmd_dir_get_path (default_dir);
	gtk_entry_set_text (
		GTK_ENTRY (dialog->priv->dir_entry), path);

	dialog->priv->baseuri = baseuri;

	/* create the popup menu */
	dialog->priv->popup_menu = gnome_cmd_search_popmenu_new (
		GTK_CLIST (dialog->priv->result_clist));
	gtk_widget_ref (dialog->priv->popup_menu);
	gtk_object_set_data_full (GTK_OBJECT (dialog),
							  "file_popup_menu", dialog->priv->popup_menu,
							  (GtkDestroyNotify)gtk_widget_unref);


    /* connect our handler which will popup the menu */
	gtk_signal_connect_object(GTK_OBJECT(dialog->priv->result_clist), "button_press_event",
							  GTK_SIGNAL_FUNC (file_popup_handler),
							  GTK_OBJECT(dialog->priv->popup_menu));
	
	return GTK_WIDGET (dialog);
}
