/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"

//#define DEBUG_DIR_REF
//#define DEBUG_DIR_LIST

int dir_counter = 0;

typedef struct
{
	GnomeCmdDirFileCreatedFunc created_func;
	GnomeCmdDirFileDeletedFunc deleted_func;
	GnomeCmdDirFileChangedFunc changed_func;
	gpointer user_data;
} DirObserverData;

typedef enum {
	FILE_CREATED,
	FILE_DELETED,
	FILE_CHANGED
} FileUpdateFuncType;


static void update_observers (GnomeCmdDir *dir, GnomeCmdFile *finfo, FileUpdateFuncType type)
{
	GList *tmp;

	g_return_if_fail (dir != NULL);

	tmp = dir->observers;
	while (tmp)
	{
		DirObserverData *data = (DirObserverData*)tmp->data;
		if (data) {
			switch (type) {
				case FILE_CREATED:
					if (data->created_func)
						data->created_func (dir, finfo, data->user_data);
					break;
				case FILE_DELETED:
					if (data->deleted_func)
						data->deleted_func (dir, finfo, data->user_data);
					break;
				case FILE_CHANGED:
					if (data->changed_func)
						data->changed_func (dir, finfo, data->user_data);
					break;
			}
		}
		tmp = tmp->next;
	}
}


static void show_error_dialog (const gchar *msg)
{
	GtkWidget *dlg;
		
	dlg = gnome_error_dialog (msg);
	
	gtk_window_set_transient_for (GTK_WINDOW (dlg), GTK_WINDOW (main_win));
	gtk_window_set_position (GTK_WINDOW (dlg), GTK_WIN_POS_CENTER);
	gtk_window_set_modal (GTK_WINDOW (dlg), TRUE);
	gtk_widget_show (dlg);
}


GnomeCmdDir *gnome_cmd_dir_new (void)
{
	GnomeCmdDir *dir = (GnomeCmdDir*)g_new (GnomeCmdDir, 1);
	g_return_val_if_fail (dir != NULL, NULL);
	dir->ref_cnt = 0;
	dir->path = NULL;
	dir->uri = NULL;
	dir->uri_str = NULL;
	dir->files = NULL;
	dir->content_changed = FALSE;
	dir->prev_error = FALSE;
	dir->observers = NULL;
	dir->not_listed = TRUE;
	dir_counter++;
	dir_pool_add (dir);
	return dir;	
}


GnomeCmdDir *gnome_cmd_dir_new_with_values (GnomeVFSURI *baseuri, const gchar *path)
{
	GnomeCmdDir *dir;

	g_return_val_if_fail (baseuri != NULL, NULL);
	g_return_val_if_fail (path != NULL, NULL);
	
	dir = gnome_cmd_dir_new ();
	g_return_val_if_fail (dir != NULL, NULL);

	dir->uri = gnome_vfs_uri_append_path (baseuri, path);
	dir->uri_str = gnome_vfs_uri_to_string (dir->uri, 0);
	dir->path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (dir->uri), 0);

#ifdef HAVE_LIBFAM
	dir->request = NULL;
	fam_monitor_start (dir);
#endif
	
	return dir;
}


GnomeCmdDir *gnome_cmd_dir_new_from_uri_str (const gchar *dir_uri_str)
{
	GnomeCmdDir *dir;

	g_return_val_if_fail (dir_uri_str != NULL, NULL);
	
	dir = gnome_cmd_dir_new ();
	g_return_val_if_fail (dir != NULL, NULL);

	dir->uri = gnome_vfs_uri_new (dir_uri_str);
	dir->uri_str = g_strdup (dir_uri_str);
	dir->path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (dir->uri), 0);

#ifdef HAVE_LIBFAM
	dir->request = NULL;
	fam_monitor_start (dir);
#endif
	
	return dir;
}


static void gnome_cmd_dir_free (GnomeCmdDir *dir)
{
#ifdef DEBUG_DIR_REF
	g_printerr ("dir freeing 0x%x %s\n", (guint)dir, dir->path);
#endif

#ifdef HAVE_LIBFAM
	fam_monitor_cancel (dir);
#endif

	dir_pool_remove (dir);
	g_list_foreach (dir->files, (GFunc)gnome_cmd_file_unref, NULL);
	gnome_vfs_uri_unref (dir->uri);

	if (dir->uri_str)
		g_free (dir->uri_str);
	if (dir->path)
		g_free (dir->path);
	g_free (dir);
	dir_counter--;
}


void gnome_cmd_dir_ref (GnomeCmdDir *dir)
{
	g_return_if_fail (dir != NULL);

	dir->ref_cnt++;
#ifdef DEBUG_DIR_REF
	g_printerr ("dir refing: 0x%x %s to %d\n", (guint)dir, dir->path, dir->ref_cnt);
#endif
}


void gnome_cmd_dir_unref (GnomeCmdDir *dir)
{
	g_return_if_fail (dir != NULL);
	
	dir->ref_cnt--;

#ifdef DEBUG_DIR_REF
	g_printerr ("dir un-refing: 0x%x %s to %d\n", (guint)dir, dir->path, dir->ref_cnt);
#endif
	
	if (dir->ref_cnt < 1)
		gnome_cmd_dir_free (dir);
}


void gnome_cmd_dir_add_observer (GnomeCmdDir *dir,
								 GnomeCmdDirFileCreatedFunc created_func,
								 GnomeCmdDirFileDeletedFunc deleted_func,
								 GnomeCmdDirFileChangedFunc changed_func,
								 gpointer user_data)
{
	DirObserverData *data = g_new (DirObserverData, 1);
	
	g_return_if_fail (dir != NULL);

	data->created_func = created_func;
	data->deleted_func = deleted_func;
	data->changed_func = changed_func;
	data->user_data = user_data;

	dir->observers = g_list_append (dir->observers, data);
}


void gnome_cmd_dir_remove_observer (GnomeCmdDir *dir,
									gpointer user_data)
{
	GList *tmp;
	
	g_return_if_fail (dir != NULL);

	tmp = dir->observers;
	while (tmp)
	{
		DirObserverData *data = (DirObserverData*)tmp->data;
		if (data && data->user_data == user_data)
		{
			dir->observers = g_list_remove (dir->observers, data);
			g_free (data);
			return;
		}
		tmp = tmp->next;
	}
}


/******************************************************************************
*
*   Function: gnome_cmd_dir_list_files
*
*   Purpose: Return a list of all files in this directory. When calling this function
*            the first time the files will be fetched but repeated calls will just return
*            the same list. See the gnome_cmd_dir_relist function if you want to force an
*            update.
*   Params: 
*
*   Returns: A GList containing the files in this directory. The list is the same as the
*            insternal one in the directory so be sure to make a copy if you intend to save
*            it.
*
*   Statuses: 
*
******************************************************************************/

GList *gnome_cmd_dir_list_files (GnomeCmdDir *dir)
{
	g_return_val_if_fail (dir != NULL, NULL);

	if (!dir->files || dir->content_changed || dir->not_listed)
	{
#ifdef DEBUG_DIR_LIST
		g_printerr ("going to relist in list due to content change or no previous files where present\n");
#endif
		return gnome_cmd_dir_relist_files (dir);
	}
	else
	{
#ifdef DEBUG_DIR_LIST
		g_printerr ("list\n");
#endif
	}

	return dir->files;
}


/******************************************************************************
*
*   Function: gnome_cmd_dir_relist_files
*
*   Purpose: Returns a fresh list of all files in this directory every time its called
*
*   Params: 
*
*   Returns: A GList containing the files in this directory. The list is the same as the
*            insternal one in the directory so be sure to make a copy if you intend to save
*            it.
*
*   Statuses: 
*
******************************************************************************/

GList *gnome_cmd_dir_relist_files (GnomeCmdDir *dir)
{
	GnomeVFSResult result;
	GList *list = NULL;
	GList *tmp;
	gchar *dir_path;
	GnomeVFSFileInfoOptions infoOpts = GNOME_VFS_FILE_INFO_FOLLOW_LINKS;
	
	g_return_val_if_fail (dir != NULL, NULL);
	g_return_val_if_fail (dir->uri != NULL, NULL);

	if (dir->prev_error) {
#ifdef DEBUG_DIR_LIST
		g_printerr ("Skipping relist due to previous error\n");
#endif
		return NULL;
	}
	
#ifdef DEBUG_DIR_LIST
	g_printerr ("relist\n");
#endif

	if (dir->files != NULL)
	{
		gnome_cmd_file_list_free (dir->files);
		dir->files = NULL;
	}

	if (gnome_cmd_data_get_use_mime ())
		infoOpts |= GNOME_VFS_FILE_INFO_GET_MIME_TYPE;

	/* list all files and place them in list */
	dir_path = gnome_vfs_uri_to_string (dir->uri, 0);
	result = gnome_vfs_directory_list_load (
		&list,
		dir_path,
		infoOpts,
		NULL);
	g_free (dir_path);


	if (result != GNOME_VFS_OK)
	{
		dir->prev_error = TRUE;
		show_error_dialog (gnome_vfs_result_to_string (result));
		return NULL;
	}
	
	if (list)
	{
		/* create a new list with GnomeCmdFile objects */
		tmp = list;
		while (tmp)
		{
			GnomeVFSFileInfo *info = (GnomeVFSFileInfo*)tmp->data;
			GnomeCmdFile *finfo = gnome_cmd_file_new_with_values (info, dir);
			gnome_cmd_file_ref (finfo);
			dir->files = g_list_append (dir->files, finfo);
			tmp = tmp->next;
		}

		gnome_vfs_file_info_list_free (list);
	}

	dir->not_listed = FALSE;

	return dir->files;
}


const gchar *gnome_cmd_dir_get_path (GnomeCmdDir *dir)
{
	g_return_val_if_fail (dir != NULL, NULL);
	g_return_val_if_fail (dir->uri != NULL, NULL);
	
	return dir->path;
}


GnomeVFSURI *gnome_cmd_dir_get_uri (GnomeCmdDir *dir)
{
	g_return_val_if_fail (dir != NULL, NULL);
	g_return_val_if_fail (dir->uri != NULL, NULL);
	
	return dir->uri;
}


const gchar *gnome_cmd_dir_get_uri_str (GnomeCmdDir *dir)
{
	g_return_val_if_fail (dir != NULL, NULL);

	return dir->uri_str;
}


GnomeVFSURI *gnome_cmd_dir_get_file_uri (GnomeCmdDir *dir, const gchar *filename)
{
	g_return_val_if_fail (dir != NULL, NULL);
	g_return_val_if_fail (dir->uri != NULL, NULL);

	return gnome_vfs_uri_append_file_name (dir->uri, filename);
}


/* A file has been created. Create a new GnomeCmdFile object for that file and 
 * notify the observers.
 */
void gnome_cmd_dir_file_created (GnomeCmdDir *dir, const gchar *filename)
{
	GnomeVFSURI *uri;
	GnomeVFSResult res;
	GnomeVFSFileInfo *info;
	GnomeCmdFile *finfo;
	GnomeVFSFileInfoOptions infoOpts = GNOME_VFS_FILE_INFO_FOLLOW_LINKS;

	g_return_if_fail (dir != NULL);
	g_return_if_fail (filename != NULL);
	
	if (gnome_cmd_data_get_use_mime ())
		infoOpts |= GNOME_VFS_FILE_INFO_GET_MIME_TYPE;
	
	uri = gnome_cmd_dir_get_file_uri (dir, filename);	
	info = gnome_vfs_file_info_new ();	
	res = gnome_vfs_get_file_info_uri (
		uri, info, infoOpts);
	gnome_vfs_uri_unref (uri);
	
	finfo = gnome_cmd_file_new_with_values (info, dir);
	dir->files = g_list_append (dir->files, finfo);
	gnome_cmd_file_ref (finfo);

	update_observers (dir, finfo, FILE_CREATED);
}


/* A file has been deleted. Find the corresponding GnomeCmdFile and
 * notify the observers.
 */
void gnome_cmd_dir_file_deleted (GnomeCmdDir *dir, const gchar *filename)
{
	GList *tmp = dir->files;

	while (tmp) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		if (strcmp (finfo->info->name, filename) == 0) {
			dir->files = g_list_remove (dir->files, finfo);
			update_observers (dir, finfo, FILE_DELETED);
			gnome_cmd_file_unref (finfo);
			return;
		}
		tmp = tmp->next;
	}
}


/* A file has been changed. Find the corresponding GnomeCmdFile, update its
 * GnomeVFSFileInfo and notify the observers.
 */
void gnome_cmd_dir_file_changed (GnomeCmdDir *dir, const gchar *filename)
{
	GList *tmp;

	g_return_if_fail (dir != NULL);
	g_return_if_fail (filename != NULL);

	tmp = dir->files;
	
	while (tmp) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)tmp->data;
		if (strcmp (finfo->info->name, filename) == 0) {
			GnomeVFSResult res;
			GnomeVFSURI *uri;
			GnomeVFSFileInfo *info;
			GnomeVFSFileInfoOptions infoOpts = 0;
	
			if (gnome_cmd_data_get_use_mime ())
				infoOpts |= GNOME_VFS_FILE_INFO_GET_MIME_TYPE;

			uri = gnome_cmd_dir_get_file_uri (dir, filename);
			info = gnome_vfs_file_info_new ();
			res = gnome_vfs_get_file_info_uri (
				uri, info, infoOpts);
			gnome_vfs_uri_unref (uri);
			
			gnome_cmd_file_update_info (finfo, info);
			update_observers (dir, finfo, FILE_CHANGED);
			return;
		}
		tmp = tmp->next;
	}
}

