/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifndef GNOME_APT_PKGTREE_H
#define GNOME_APT_PKGTREE_H

#include "gnome-apt.h"
#include "cache.h"
#include "drawtree.h"
#include "filter.h"

#include <vector>

class GAptPkgTree : public GAptCacheFile::CacheView,
                    public Filter::View {
public:
  class Item;
  class Category;
  class Pkg;
  class CategoryList;

  // This is basically a "View" monitoring the tree
  
  class Status {
  public:
    virtual void set_status   (const char* s) = 0;
    virtual void set_selection(pkgCache::Package* pkg) = 0;
  };


  GAptPkgTree();
  virtual ~GAptPkgTree();

  GAptCache* cache() { return cache_; }

  void set_cache(GAptCache* cache);

  DrawTree* tree() { return tree_; }

  void set_status(Status* s) { status_ = s; update_status(); }

  void filter_changed();
  void set_filter(Filter* f);
  Filter* filter() { return filter_; }

  // iff this package exists, we move focus to it and select.
  void move_selection(pkgCache::Package* p);
  
  // We have toplevel categories, and then packages are sorted within
  // those.

  typedef enum {
    SortAlpha,
    SortSection,
    SortStatus,
    SortSource, 
    SortPriority,
    SortNone,

    SortTypeEnd
  } SortType;

  typedef enum {
    CategoryAlpha,
    CategorySection,
    CategoryStatus,
    CategorySource,
    CategoryPriority,
    CategoryNone,

    CategoryTypeEnd
  } CategoryType;
    
  // Eventually we could allow setting the sort
  //  for individual categories, rather than the 
  //  whole list.
  void set_sort    (SortType st);
  void set_category(CategoryType ct);

  CategoryType get_category() const { return category_; }
  SortType     get_sort() const { return sort_; }

  typedef enum {
    ColumnDelete,
    ColumnKeep,
    ColumnInstall,
    
    ColumnName,
    ColumnCurrent,
    ColumnAvailable,

    ColumnStatus,

    ColumnSection,
    ColumnPriority,

    ColumnDescription,

    ColumnTypeEnd
  } ColumnType;

  // These translate from ColumnType to cols in the 
  //  draw tree
  void set_visible(ColumnType ct, bool visibility);
  bool is_visible (ColumnType ct) const;
  void set_width  (ColumnType ct, double width);

  ColumnType column_type(gushort col) const { 
    g_return_val_if_fail(col < columns_.size(), ColumnTypeEnd);
    return columns_[col]; 
  }

  gushort    type_column(ColumnType col) const;

  void set_column_order(const vector<ColumnType> & cts);

  static const char* column_name(ColumnType ct);
  static const char* long_column_name(ColumnType ct);

  // For use by Items when drawing
  typedef enum {
    ConflictsPixmap,
    DependsPixmap,
    RecommendsPixmap,
    SuggestsPixmap,
    PackagePixmap,
    FolderPixmap,
    ReplacesPixmap,
    PixmapTypeEnd
  } PixmapType;

  static const char** pixmap_data(PixmapType p);
  
  void pixmap(PixmapType p, GdkPixmap** pixmap, GdkPixmap ** mask);

  GdkGC* broken_gc();

  void set_broken_color(GdkColor* c);

  SortType sort() const { return sort_; }

  // Queues a redisplay - this is the master point of synchronization
  void package_change_notify();

  // for use by the nodes
  void selection_change_notify(pkgCache::Package* p);

private:
  GAptCache* cache_;

  SortType sort_;
  CategoryType category_;

  DrawTree* tree_;

  CategoryList* category_list_;

  GdkPixmap* pixmaps_[PixmapTypeEnd];
  GdkPixmap* masks_[PixmapTypeEnd];
  GdkGC* broken_gc_;

  // Mapping from column type to DrawTree column
  vector<ColumnType> columns_;
  map<ColumnType,gint> column_widths_;
  map<ColumnType,bool> column_vis_;

  Status* status_;

  Filter* filter_;

  friend gint package_changes_idle(gpointer data);
  void do_package_changes();
  bool change_notify_queued_;

  void create_category(CategoryType ct);
  void reorder_columns();
  void update_status();

  GdkColor broken_color_;

  void update_broken_gc();

  GAptPkgTree(const GAptPkgTree &);
};

struct GnomeCanvasRow;

class GAptPkgTree::Item : public DrawTree::Node {
public:

  typedef enum {
    CategoryItem,
    PackageItem,
    DependencyItem,
    PreDependencyItem,    
    RecommendedItem,
    SuggestedItem,
    ConflictingItem,
    ReplacedItem,
    InvalidItem
  } ItemRelationshipType;

  Item(ItemRelationshipType rel, GAptPkgTree* t);
  virtual ~Item();

  virtual DrawTree::Node* parent() { return 0; }

  virtual void expand() = 0;
  virtual void collapse() = 0;
  virtual bool expanded() = 0;
  virtual bool expandable() = 0;

  virtual bool get_bool(gushort column) = 0;

  virtual void set_bool(gushort column, bool state) = 0;

  virtual void draw_column(gushort column, gushort remaining,
                           GtkStateType state,
                           GdkDrawable* drawable, 
                           GdkRectangle* rect) = 0;

  ItemRelationshipType relation() { return relation_; }
  GAptPkgTree* tree() { return tree_; }
  
  void sort(GAptPkgTree::SortType st);
  virtual bool filter(Filter* filter) = 0;

  // Stuff for sorting
  virtual const char* name() = 0;
  virtual const char* section() = 0;
  virtual const char* priority() = 0;
  virtual Util::StatusType status() = 0;
protected:  
  GAptPkgTree* tree_;

  ItemRelationshipType relation_;
  
private:
  Item();
  Item(const Item &);
};

class GAptPkgTree::Category : public GAptPkgTree::Item {
private:
  const string name_;
  bool expanded_;
public:
  Category(const char* name, GAptPkgTree* t);
  virtual ~Category();

  void add   (Item* child);
  void remove(Item* child);

  virtual void expand();
  virtual void collapse();
  virtual bool expanded() { return expanded_; }
  virtual bool expandable() { return !children_.empty(); }
  virtual void refresh_expansion() {} // do nothing, hasn't changed

  virtual bool get_bool(gushort column);

  virtual void set_bool(gushort column, bool state);

  virtual bool display_bool(gushort column);

  virtual bool filter(Filter* f);

  virtual void draw_column(gushort column, gushort remaining,
                           GtkStateType state,
                           GdkDrawable* drawable,
                           GdkRectangle* rect); 

  virtual void select();
  virtual void unselect();

  // for the sort routines
  const char* name() { return name_.c_str(); }
  const char* section() { return name_.c_str(); } // has no section
  const char* priority() { return name_.c_str(); } // no priority
  Util::StatusType status() { return Util::StatusTypeEnd; }
};

class GAptPkgTree::Pkg : public GAptPkgTree::Item {
private:
  pkgCache::Package* pkg_;
  Item* parent_;

public:
  Pkg(ItemRelationshipType rel,
      pkgCache::Package* pkg, 
      GAptPkgTree* t, Item* p);
  virtual ~Pkg();

  pkgCache::PkgIterator 
  package(pkgCache &cache) {
    return pkgCache::PkgIterator(cache,pkg_); // pkg_ == 0 is OK here
  };

  virtual DrawTree::Node* parent() { return parent_; }

  virtual bool filter(Filter* f);

  virtual void expand();
  virtual void collapse();
  virtual bool expanded() { return !children_.empty(); } // only valid if expandable
  virtual bool expandable();
  virtual void refresh_expansion();

  virtual bool get_bool(gushort column);

  virtual void set_bool(gushort column, bool state);

  virtual bool display_bool(gushort column);

  virtual void draw_column(gushort column, gushort remaining,
                           GtkStateType state,
                           GdkDrawable* drawable, 
                           GdkRectangle* rect);

  virtual void select();
  virtual void unselect();
  
  // for the sort routines
  const char* name() { return package(*(tree_->cache())).Name(); }
  const char* section() { return package(*(tree_->cache())).Section(); }
  const char* priority();
  Util::StatusType status();

  // For finding a node by Package*
  bool match(pkgCache::Package* p) const {
    return p == pkg_;
  }

protected:

};

class GAptPkgTree::CategoryList : public DrawTree::NodeList
{
public:

  CategoryList(GAptPkgTree* tree);
  virtual ~CategoryList();
    
  virtual iterator begin() { 
    return reinterpret_cast<iterator>(nodes_.begin());
  };
  virtual iterator end() { 
    return reinterpret_cast<iterator>(nodes_.end());
  };

  // add a toplevel node.
  void add_node(Item* node);
  void add_nodes(vector<Item*> & nodes);

  // deletes it!
  void remove_node(Item* node);

  // deletes them!
  void clear_nodes();

private:
  vector<DrawTree::Node*> nodes_;
  GAptPkgTree* tree_;
};

#endif
