/* Ada Ravenscar thread support.
   Free Software Foundation, Inc.
   Copyright 2004 Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "defs.h"
#include "gdbcore.h"
#include "gdbthread.h"
#include "ada-lang.h"
#include "target.h"
#include "inferior.h"
#include "command.h"
#include "ravenscar-thread.h"
#include "observer.h"
#include "gdb_string.h"
#include "top.h"

/* This module's target-specific operations. */
static struct target_ops ravenscar_ops;

/* Copy of the target over which the ravenscar target is pushed.  This is
   more convenient than a pointer to procfs_ops or core_ops, because
   they lack current_target's default callbacks. */
static struct target_ops base_ops;

/* Some base target uses a special value for the null PID (exempli gratia
   remote).  */
static ptid_t base_magic_null_ptid;

static const char *running_thread_name = "running_thread";

static const char *known_tasks_name = "system__tasking__debug__known_tasks";


static struct observer *update_target_observer;

/* Saved pointer to previous owner of target_new_objfile_hook.  */

static void (*target_new_objfile_chain)(struct objfile *);

/* Architecture-specific hooks.  */
static struct ravenscar_arch_ops* current_arch_ops;

static void new_objfile (struct objfile *objfile);

static CORE_ADDR read_thread_id (const char* symbol_name);

static ptid_t ravenscar_wait (ptid_t ptid, struct target_waitstatus *status);
static void ravenscar_find_new_threads (void);
static ptid_t ravenscar_running_thread (void);
static char * ravenscar_extra_thread_info (struct thread_info *tp);
static int ravenscar_thread_alive (ptid_t ptid);
static char * ravenscar_pid_to_str (ptid_t ptid);
static void ravenscar_fetch_registers (int regnum);
static void ravenscar_store_registers (int regnum);
static void ravenscar_prepare_to_store (void);
static void ravenscar_close (int quitting);
static void ravenscar_open  (char *name, int from_tty);

/* target_new_objfile_hook callback.

   If OBJFILE is non-null, check whether a Ravenscar application is
   being debugged, and if so, push the Ravenscar target.

   If OBJFILE is null, stop debugging Ravenscar tasks.  */

static void
new_objfile (struct objfile *objfile)
{
  struct minimal_symbol *msym;
  msym = lookup_minimal_symbol (known_tasks_name, NULL, NULL);

  if (objfile && msym)
    ravenscar_open (NULL, 0);
  else
    unpush_target (&ravenscar_ops);

  if (target_new_objfile_chain)
    target_new_objfile_chain (objfile);
}

/* Read the thread ID whose symbol name is SYMBOL_NAME.  */

static CORE_ADDR
read_thread_id (const char* symbol_name)
{
  const struct minimal_symbol *object_msym =
    lookup_minimal_symbol (symbol_name, NULL, NULL);
  int object_size;
  int buf_size;
  char* buf;
  CORE_ADDR object_addr;

  if (!object_msym)
    return 0;

  object_addr = SYMBOL_VALUE_ADDRESS (object_msym);
  object_size = TYPE_LENGTH (builtin_type_void_data_ptr);
  buf_size = object_size;
  buf = alloca (buf_size);
  read_memory (object_addr, buf, buf_size);
  return extract_typed_address (buf, builtin_type_void_data_ptr);
}

static ptid_t
ravenscar_wait (ptid_t ptid, struct target_waitstatus *status)
{
  base_ops.to_wait (ptid, status);
  ravenscar_find_new_threads ();
  inferior_ptid = ravenscar_running_thread ();
  return inferior_ptid;
}

void
ravenscar_find_new_threads (void)
{
  struct task_entry *task;
  ada_build_task_list ();
  init_thread_list ();
  for (task = task_list; task; task = task->next_task)
    {
      add_thread (ptid_build (task->atcb.thread, 0, task->atcb.thread));
    }
}

ptid_t
ravenscar_running_thread (void)
{
  CORE_ADDR tid = read_thread_id (running_thread_name);
  return ptid_build (tid, 0, tid);
}

char *
ravenscar_extra_thread_info (struct thread_info *tp)
{
  return "Ravenscar task";
}

int
ravenscar_thread_alive (ptid_t ptid)
{
  /* Ravenscar tasks are non-terminating.  */
  return 1;
}

static char *
ravenscar_pid_to_str (ptid_t ptid)
{
  static char buf[30];

  sprintf (buf, "Thread %#x", (int) ptid_get_tid (ptid));
  return buf;
}

void
ravenscar_fetch_registers (int regnum)
{
  if (ptid_equal (inferior_ptid, base_magic_null_ptid)
      || ptid_equal (inferior_ptid, ravenscar_running_thread ()))
    base_ops.to_fetch_registers (regnum);
  else
    current_arch_ops->to_fetch_registers (regnum);
}

void
ravenscar_store_registers (int regnum)
{
  if (ptid_equal (inferior_ptid, base_magic_null_ptid)
      || ptid_equal (inferior_ptid, ravenscar_running_thread ()))
    base_ops.to_store_registers (regnum);
  else
    current_arch_ops->to_store_registers (regnum);
}

void
ravenscar_prepare_to_store (void)
{
  if (ptid_equal (inferior_ptid, base_magic_null_ptid)
      || ptid_equal (inferior_ptid, ravenscar_running_thread ()))
    base_ops.to_prepare_to_store ();
  else
    current_arch_ops->to_prepare_to_store ();
}

/* Update_current_target hook.  Rebuilt base_ops from the new target
   stack.  */
static void
ravenscar_update_target_hook (void* unused)
{
  target_partial_init (&base_ops, process_stratum);
}

void
ravenscar_register_arch_ops (struct ravenscar_arch_ops *ops)
{
  /* FIXME: To be clean, we would need to handle a list of
     architectures, just like in remote-wtx-hw.c. However, for now the
     only Ravenscar run-time for bare board that is implemented in
     GNAT is for only one architecture: erc32-elf. So no need to care about
     that for now...*/
  current_arch_ops = ops;
}

static void
ravenscar_open  (char *name, int from_tty)
{
  char target_command [7] = "target ";
  char *target_name;
  if (name)
    {
      target_name = (char *) alloca ((strlen (name) + 7) * sizeof (char));
      sprintf (target_name, "%s%s", target_command, name);
      execute_command (target_name, from_tty);
    }
  update_target_observer =
    observer_attach_update_current_target (ravenscar_update_target_hook);
  base_magic_null_ptid = inferior_ptid;
  inferior_ptid = ravenscar_running_thread ();
  push_target (&ravenscar_ops);
}

static void
ravenscar_close (int quitting)
{
  if (update_target_observer)
    observer_detach_update_current_target (update_target_observer);
  update_target_observer = NULL;
}

static void
init_ravenscar_thread_ops (void)
{
  ravenscar_ops.to_shortname         = "ravenscar";
  ravenscar_ops.to_longname          = "Ravenscar tasks.";
  ravenscar_ops.to_doc               = "Ravenscar tasks support.";
  ravenscar_ops.to_open              = ravenscar_open;
  ravenscar_ops.to_close             = ravenscar_close;
  ravenscar_ops.to_wait              = ravenscar_wait;
  ravenscar_ops.to_fetch_registers   = ravenscar_fetch_registers;
  ravenscar_ops.to_store_registers   = ravenscar_store_registers;
  ravenscar_ops.to_prepare_to_store  = ravenscar_prepare_to_store;
  ravenscar_ops.to_thread_alive      = ravenscar_thread_alive;
  ravenscar_ops.to_find_new_threads  = ravenscar_find_new_threads;
  ravenscar_ops.to_pid_to_str        = ravenscar_pid_to_str;
  ravenscar_ops.to_extra_thread_info = ravenscar_extra_thread_info;
  ravenscar_ops.to_stratum           = thread_stratum;
  ravenscar_ops.to_magic             = OPS_MAGIC;
}

/* Module startup initialization function, automagically called by
   init.c. */

void
_initialize_ravenscar (void)
{
  extern struct cmd_list_element *cmdlist;

  init_ravenscar_thread_ops ();

  /* Notice when object files get loaded and unloaded.  */
  target_new_objfile_chain = deprecated_target_new_objfile_hook;
  deprecated_target_new_objfile_hook = new_objfile;

  add_target (&ravenscar_ops);
}
