/* non blocking implementations of various socket functions */

#include <sys/wait.h>

#include "gnapster.h"

#include "nb.h"

#define NB_INC 64

GList *nb_rd = NULL;

void nb_gethostbyname(void *cb, char *name) {
   int pd[2], cpid;
   Connection *c;
   
   pipe(pd);
   
   cpid = fork();
   
   if (cpid == 0) {
      /* child */
      close(pd[0]);
      hostresolve(pd[1], name);
      close(pd[1]);
      _exit(0);
   }
   
   close(pd[1]);

   c = d_new(CONNECTION);
   
   c->data = cb;
   
   c->tag = INPUT_ADD(pd[0], GDK_INPUT_READ, nb_gethostcb, c);
   
   wait(NULL);
}

void nb_gethostcb(gpointer data, int source, GdkInputCondition cond) {
   Connection *c;
   unsigned int ip;
   int n;
   
   d_assert(data != NULL);
   
   c = data;
   
   n = read(source, &ip, sizeof(ip));
   if (n <= 0 || ip == 0)
     ((GetHostCB)c->data) (0);
   
   INPUT_REMOVE(c->tag);
   
   ((GetHostCB)c->data) (ip);
   
   j_free(CONNECTION, c);
}

void hostresolve(int fd, char *name) {
   struct hostent *host;
   unsigned int ip = 0;
   
   host = gethostbyname(name);
   
   if (!host) {
      write(fd, &ip, sizeof(ip));
      return;
   }
   
   memcpy(&ip, host->h_addr_list[0], host->h_length);
   
   write(fd, &ip, sizeof(ip));
}

NBRead *nb_active(int type) {
   GList *ptr;
   NBRead *nb;

   for(ptr=nb_rd; ptr; ptr=ptr->next) {
      nb = ptr->data;
      if (!nb)
	continue;
      
      if (nb->nb_type == type)
	return nb;
   }
   
   /* no active read found, let's create one */
   nb = d_malloc(sizeof(NBRead));
   memset(nb, 0, sizeof(NBRead));
   
   nb->nb_type = type;
   
   nb_rd = g_list_prepend(nb_rd, nb);
   
   return nb;
}

void nb_terminate(NBRead *nb) {
   if (!nb)
     return;
   
   nb->data = d_realloc(nb->data, nb->len + 1);
   
   nb->data[nb->len] = 0;

   nb->term = 1;
}

void nb_add(NBRead *nb, char c) {
   /* if the last packet was terminated, we need to reset the data to
    * be filled again */
   if (nb->term) {
      d_free(nb->data);
      
      nb->len = nb->datalen = nb->term = 0;
      
      nb->data = NULL;
   }
   
   /* check to make sure we need more mem */
   if (nb->len >= nb->datalen) {
      nb->datalen += NB_INC;
      nb->data = d_realloc(nb->data, nb->datalen);
   }

   nb->data[nb->len++] = c;
}

int nb_read(NBRead *nb, int source, size_t count, char term) {
   char buf[RW_BUFFER], *ptr;
   int n, len, tlen;
   unsigned int flags;
   
   if (nb->term)
     nb->len = nb->datalen = 0;
   
   /* if we specify a count, force that amount of bytes, else, we'll
    * read until we meet term */
   len = count ? (count - nb->len) : (sizeof(buf) - 1);
   flags = count ? 0 : MSG_PEEK;
   
   if (len > (sizeof(buf) - 1))
     len = sizeof(buf) - 1;
   
   tlen = 0;
   
   n = recv(source, buf, len, flags);
   if (n <= 0)
     return 0;
   
   for(ptr=buf; n>0; ptr++, n--) {
      if (!count && *ptr == term) {
	 nb_terminate(nb);
	 
	 /* we have found the character, push tlen bytes off the queue */
	 return recv(source, buf, tlen + 1, 0);
      }

      nb_add(nb, *ptr);
      tlen++;
   }
   
   /* we didn't find it in this packet, push it all off the queue
    * and return */
   if (!count)
     recv(source, buf, len, 0);
   
   return 1;
}

void nb_free(NBRead *nb) {
   if (!nb)
     return;
   
   d_free(nb->data);
   d_free(nb);
}

void nb_finish(NBRead *nb) {
   nb_rd = g_list_remove(nb_rd, nb);
   
   nb_free(nb);
}

void nb_destroy(int type) {
   GList *ptr, *prev;
   NBRead *nb;
   
   prev = NULL;
   
   if (!nb_rd)
     return;
   
   for(ptr=nb_rd; ptr; ptr=ptr->next) {
      nb = ptr->data;
      if (!nb)
	continue;

      if (nb->type == type) {
	 nb_free(nb);
	 nb_rd = g_list_remove_link(nb_rd, ptr);
	 g_list_free_1(ptr);
	 ptr = prev;
      }
      
      prev = ptr;
      
      if (!ptr)
	break;
   }
}

void nb_destroy_all() {
   GList *ptr;
   NBRead *nb;
   
   if (!nb_rd)
     return;
   
   for(ptr=nb_rd; ptr; ptr=ptr->next) {
      nb = ptr->data;
      if (!nb)
	continue;
      
      nb_free(nb);
   }
   
   g_list_free(nb_rd);
   
   nb_rd = NULL;
}

