------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2006 Dmitriy Anisimkov.                                   --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: ocimem.adb,v 1.1 2006/06/16 09:03:14 vagul Exp $

--  Test for simultaneous multitasking connect.

with Ada.Exceptions;
with Ada.Text_IO;
with Ada.Unchecked_Deallocation;

with OCI.Thick.Connections;
with System;

procedure OCIMem is
   use Ada.Text_IO;
   use type System.Address;

   type Element is mod 2**8;

   type Array_Type is array (1 .. 4096) of Element;

   type List_Record;

   type List_Access is access all List_Record;

   type List_Record is record
      Data : Array_Type;
      Prev : List_Access;
      Next : List_Access;
   end record;

   List_Length : constant := 100_000;

   Sample : Array_Type;

   First : List_Access;
   Last  : List_Access;
   C     : List_Access;
   Char  : Character;

   procedure Test_Link;

   procedure Free is new Ada.Unchecked_Deallocation (List_Record, List_Access);

   task type DB_Connection;

   protected Start_Line is
      entry Wait (You_Number : out Positive);
      procedure Start;
   private
      Go    : Boolean := False;
      Count : Natural := 0;
   end Start_Line;

   ----------------
   -- Start_Line --
   ----------------

   protected body Start_Line is

      entry Wait (You_Number : out Positive) when Go is
      begin
         Count := Count + 1;
         You_Number := Count;
      end Wait;

      procedure Start is
      begin
         Go := True;
      end Start;

   end Start_Line;

   ---------------
   -- Test_Link --
   ---------------

   procedure Test_Link is
   begin
      C := First;

      while C /= null loop
         if C.Data /= Sample then
            Put_Line ("Data broken.");
            raise Program_Error;
         end if;

         if C.Next /= null and then C.Data'Address > C.Next.Data'Address then
            Put_Line ("Less address");
         end if;

         C := C.Next;
      end loop;
   end Test_Link;

   -------------------
   -- DB_Connection --
   -------------------

   task body DB_Connection is
      package OTC renames OCI.Thick.Connections;

      My_Number : Positive;
      OC : OTC.Connection;
   begin
      Start_Line.Wait (My_Number);

      OC := OTC.Logon ("scott/tiger");

      if My_Number = 3 then
         Put_Line (OTC.Server_Version (OC));
      end if;
   exception
      when E : others =>
         Put_Line (Ada.Exceptions.Exception_Information (E));
   end DB_Connection;

   DBC : array (1 .. 8) of DB_Connection;

begin
   for J in Sample'Range loop
      Sample (J) := Element (J mod Element'Modulus);
   end loop;

   for J in 1 .. List_Length loop
      C := Last;

      Last := new List_Record'(Data => Sample, Prev => Last, Next => null);

      if C = null then
         First := Last;
      else
         C.Next := Last;
      end if;

      if J rem 1025 = 0 then
         Put_Line (Integer'Image (J mod Element'Modulus));
      end if;
   end loop;

   Test_Link;

   Put_Line ("Tested 1.");

   for J in 1 .. 10 loop
      C := First;
      First := First.Next;
      Free (C);
   end loop;

   First.Prev := null;

   C := new List_Record'(Sample, Prev => null, Next => First);

   First.Prev := C;
   First := C;

   Start_Line.Start;

   Put_Line ("DB started.");

   delay 10.0;
   Get_Immediate (Char);

   Test_Link;

   Put_Line ("Tested 2.");

   Get_Immediate (Char);
end OCIMem;
