/**
 * GMyth Library
 * 
 * @file gmyth/gmyth_recprofile.c
 * 
 * @brief <p> This file contains the recprofile class.
 *
 * Copyright (C) 2007 INdT - Instituto Nokia de Tecnologia.
 * @author Artur Duque de Souza <artur.souza@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "gmyth_recprofile.h"
#include "gmyth_util.h"
#include "gmyth_debug.h"
#include "gmyth_http.h"

static void     gmyth_recprofile_class_init(GMythRecProfileClass * klass);
static void     gmyth_recprofile_init(GMythRecProfile * object);

static void     gmyth_recprofile_dispose(GObject * object);
static void     gmyth_recprofile_finalize(GObject * object);

G_DEFINE_TYPE(GMythRecProfile, gmyth_recprofile, G_TYPE_OBJECT)
    static void     gmyth_recprofile_class_init(GMythRecProfileClass *
                                                klass)
{
    GObjectClass   *gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->dispose = gmyth_recprofile_dispose;
    gobject_class->finalize = gmyth_recprofile_finalize;
}

static void
gmyth_recprofile_init(GMythRecProfile * recprofile)
{
}

static void
gmyth_recprofile_dispose(GObject * object)
{
    GMythRecProfile *recprofile = GMYTH_RECPROFILE(object);

    if (recprofile->name)
        g_free(recprofile->name);

    if (recprofile->group)
        g_free(recprofile->group);

    if (recprofile->vcodec)
        g_free(recprofile->vcodec);

    if (recprofile->acodec)
        g_free(recprofile->acodec);

    if (recprofile->options)
        g_free(recprofile->options);

    G_OBJECT_CLASS(gmyth_recprofile_parent_class)->dispose(object);
}

static void
gmyth_recprofile_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);
    G_OBJECT_CLASS(gmyth_recprofile_parent_class)->finalize(object);
}

/**
 * Creates a new instance of GMythRecProfile.
 * 
 * @return a new instance of GMythRecProfile.
 **/
GMythRecProfile *
gmyth_recprofile_new(void)
{
    GMythRecProfile *recprofile = GMYTH_RECPROFILE
        (g_object_new(GMYTH_RECPROFILE_TYPE, NULL));

    recprofile->id = 0;
    recprofile->name = NULL;
    recprofile->group = NULL;
    recprofile->vcodec = NULL;
    recprofile->acodec = NULL;
    recprofile->options = NULL;

    return recprofile;
}


/**
 *
 * gmyth_recprofile_get_profile_list
 * @brief get profile list from the backend
 * @param backend_info GMythBackendInfo*
 * @return GSList
 *
 **/
GSList         *
gmyth_recprofile_get_profile_list(GMythBackendInfo * backend_info)
{
    return gmyth_http_retrieve_rec_profiles(backend_info, "Transcoders");
}

/**
 *
 * gmyth_recprofile_create_profile
 * @brief get profile list from the backend
 * @param backend_info GMythBackendInfo*
 * @param profile GMythRecProfile*
 * @return gint representing the result
 *
 **/
gint
gmyth_recprofile_create_profile(GMythBackendInfo * backend_info,
                                GMythRecProfile * profile)
{
    return gmyth_http_create_rec_profile(backend_info, profile);
}

/**
 *
 * gmyth_recprofile_del_profile
 * @brief del profile from the backend
 * @param backend_info GMythBackendInfo*
 * @param id profile's id
 * @return gint representing the result
 *
 **/
gint
gmyth_recprofile_del_profile_list(GMythBackendInfo * backend_info, gint id)
{
    return gmyth_http_del_rec_profile(backend_info, id);
}

/**
 *
 * gmyth_recprofile_set_id
 * @brief set recprofile's id
 * @param rec GMythRecProfile*
 * @param id profile's id
 * @return gint representing the result
 *
 **/
gint
gmyth_recprofile_set_id(GMythRecProfile * rec, gint id)
{
    rec->id = id;
    return 0;
}

/**
 *
 * gmyth_recprofile_set
 * @brief set recprofile's property
 * @param rec GMythRecProfile*
 * @param member the member you want to modify
 * @param value the value
 * @return gint representing the result
 *
 **/
gint
gmyth_recprofile_set(GMythRecProfile * rec, gchar * member, gchar * value)
{
    int             ret = 0;

    if (value != NULL) {
        if (g_ascii_strcasecmp(member, "name") == 0)
            rec->name = g_strndup(value, strlen(value));
        else if (g_ascii_strcasecmp(member, "group") == 0)
            rec->group = g_strndup(value, strlen(value));
        else if (g_ascii_strcasecmp(member, "vcodec") == 0)
            rec->vcodec = g_strndup(value, strlen(value));
        else if (g_ascii_strcasecmp(member, "acodec") == 0)
            rec->acodec = g_strndup(value, strlen(value));
        else
            ret = -1;
    } else
        ret = -1;

    return ret;
}

/**
 *
 * gmyth_recprofile_set_name
 * @brief set recprofile's name
 * @param rec GMythRecProfile*
 * @param name profile's name
 * @return gint representing the result
 *
 **/
gint
gmyth_recprofile_set_name(GMythRecProfile * rec, gchar * name)
{
    return gmyth_recprofile_set(rec, "name", name);
}

/**
 *
 * gmyth_recprofile_set_group
 * @brief set recprofile's group
 * @param rec GMythRecProfile*
 * @param group profile's group
 * @return gint representing the result
 *
 **/
gint
gmyth_recprofile_set_group(GMythRecProfile * rec, gchar * group)
{
    return gmyth_recprofile_set(rec, "group", group);
}

/**
 *
 * gmyth_recprofile_set_vcodec
 * @brief set recprofile's vcodec
 * @param rec GMythRecProfile*
 * @param vcodec profile's vcodec
 * @return gint representing the result
 *
 **/
gint
gmyth_recprofile_set_vcodec(GMythRecProfile * rec, gchar * vcodec)
{
    return gmyth_recprofile_set(rec, "vcodec", vcodec);
}

/**
 *
 * gmyth_recprofile_set_acodec
 * @brief set recprofile's acodec
 * @param rec GMythRecProfile*
 * @param acodec profile's acodec
 * @return gint representing the result
 *
 **/
gint
gmyth_recprofile_set_acodec(GMythRecProfile * rec, gchar * acodec)
{
    return gmyth_recprofile_set(rec, "acodec", acodec);
}
