/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include "mcp.h"
#include "packages.h"

mcp_version client_min = {2, 1};
mcp_version client_max = {2, 1};

void mcp_multiline_free(mcp_multiline *m);
void mcp_package_free(mcp_package *p);
mcp_multiline *mcp_multiline_find(mcp_session *s, const char *key);
void mcp_handle_multiline_line(mcp_session *s, const char *line, int len);
void mcp_handle_multiline_end(mcp_session *s, const char *line, int len);

mcp_version *str_to_version(const char *version);
mcp_version *version_match(const mcp_version *client_min,
			   const mcp_version *client_max,
			   const mcp_version *server_min,
			   const mcp_version *server_max);

char **mcp_parse_line(const char *line, int len, int *count);
char *generate_key();

char *my_strndup(const char *s, int len);

void create_new_session_for(world *w, const mcp_version *version) {
    mcp_session *s = g_malloc(sizeof(mcp_session));
    char *str;

    s->belongs_to = w;
    w->mcp = s;
    s->key = generate_key();
    s->version.major = version->major;
    s->version.minor = version->minor;
    s->packages = NULL;
    s->multilines = NULL;

    str = g_malloc(strlen(s->key) + 100);
    sprintf(str, "#$#mcp authentication-key: %s version: %d.%d to: %d.%d\n",
            s->key, client_min.major, client_min.minor,
            client_max.major, client_max.minor);
    gm_world_writeln(w, str);
    g_free(str);

    mcp_negotiate_startup(s);
}

void try_open_for_world(world *w, const char *from, const char *to) {
    mcp_version *server_min = NULL;
    mcp_version *server_max = NULL;
    mcp_version *match = NULL;

    if((server_min = str_to_version(from)) &&
       (server_max = str_to_version(to))) {
        if((match = version_match(&client_min, &client_max,
                                  server_min, server_max))) {
            create_new_session_for(w, match);
        }
    }
    g_free(match);
    g_free(server_min);
    g_free(server_max);
}

void gm_mcp_open_for_world(world *w, const char *line, int len) {
    char **fields;
    int count;
    char *from, *to;
    int i;
    fields = mcp_parse_line(line, len, &count);
    if(!fields) return;
    if(count != 5) {
        if(debug) printf("\tIncorrent number of args (%d)\n", count);
        goto done;
    }

    from = to = NULL;
    for(i = 1; i < 5; i += 2) {
        if(strcasecmp(fields[i], "version:") == 0) from = fields[i + 1];
        else if(strcasecmp(fields[i], "to:") == 0) to = fields[i + 1];
    }
    if(!from || !to) {
        if(debug) printf("\tMissing argument :(\n");
        goto done;
    }
    try_open_for_world(w, from, to);
  done:
    for(i = 0; i < count; i++) g_free(fields[i]);
    g_free(fields);
}

mcp_package *find_package(mcp_session *s, const char *name) {
    GList *l;
    mcp_package *p;
    for(l = s->packages; l; l = g_list_next(l)) {
        p = l->data;
        if(gm_package_name_matches(p, name))
            return p;
    }
    return NULL;
}

void mcp_try_command(mcp_session *s, const char *key, const char *command,
                     int argc, char **argv) {
    char *package, *func, *tmp;
    mcp_package *p;

    if(strcmp(key, s->key) != 0) {
        if(debug) {
            printf("MCP package with a wrong key !\n");
            printf("\tkey is >>%s<< package had >>%s<<\n", key, s->key);
        }
        return;
    }

    if((p = find_package(s, command))) {
        if(p->handle)(p->handle)(p, "", argc, argv);
    } else {
        package = g_strdup(command);
        tmp = NULL;
        while((func = strrchr(package, '-'))) {
            func[0] = '\0';
            if((p = find_package(s, package))) {
                func++;
                break;
            }
            if(tmp) tmp[0] = '-';
            tmp = func;
        }
        if(tmp) tmp[0] = '\0';
        if(p && p->handle)(p->handle)(p, func, argc, argv);
        g_free(package);
    }
}

void gm_mcp_handle_normal(mcp_session *s, const char *line, int len) {
    char **fields, *key, *command, **argv;
    int count, argc, i;
    if((fields = mcp_parse_line(line, len, &count))) {
        if(count % 2 == 0 && count >= 2) {
            command = fields[0];
            key = fields[1];
            argc = count - 2;
            argv = argc ? fields + 2 : NULL;
            for(i = 0; i < argc; i += 2) {
                if(!(len = strlen(argv[i])) || argv[i][len - 1] != ':') {
                    goto mcp_line_malformed;
                }
                argv[i][len - 1] = '\0';
            }
            mcp_try_command(s, key, command, argc, argv);
        }
      mcp_line_malformed:
        for(i = 0; i < count; i++) {
            g_free(fields[i]);
        }
        g_free(fields);
    }
}

void gm_mcp_handle(mcp_session *s, const char *line, int len) {
    if(strncmp(line, "* ", 2) == 0) {
        mcp_handle_multiline_line(s, line + 2, len - 2);
    } else if(strncmp(line, ": ", 2) == 0) {
        mcp_handle_multiline_end(s, line + 2, len - 2);
    } else {
        gm_mcp_handle_normal(s, line, len);
    }
}

void gm_mcp_close(mcp_session *s) {
    if(!s) return;

    g_list_foreach(s->multilines, (GFunc) mcp_multiline_free, NULL);
    g_list_free(s->multilines);

    g_list_foreach(s->packages, (GFunc) mcp_package_free, NULL);
    g_list_free(s->packages);

    s->belongs_to->mcp = NULL;

    g_free(s->key);
    g_free(s);
}

void gm_mcp_add_package(mcp_session *s, mcp_package *p) {
    s->packages = g_list_append(s->packages, p);
}

void gm_mcp_send(mcp_session *s,
		 const char *name, const char *func, const char *args) {
    char *line = g_strconcat("#$#", name, "-", func, " ", s->key,
			     " ", args, "\n", NULL);
    gm_world_write(s->belongs_to, line);
    g_free(line);
}


/* Multiline stuff **************************************************************/
void gm_mcp_multiline_send(mcp_session *s, const char *key, const char *name,
                           const char *line, int has_newline) {
    char *str = g_strdup_printf("#$#* %s %s: %s%s",
                                key, name, line, has_newline ? "" : "\n");
    gm_world_write(s->belongs_to, str);
    g_free(str);
}

void gm_mcp_multiline_end(mcp_session *s, const char *key) {
    char *str = g_strconcat("#$#: ", key, "\n", NULL);
    gm_world_write(s->belongs_to, str);
    g_free(str);
}

void mcp_parse_multiline(const char *line, int len,
                        char **key, char **name, char **value) {
    const char *s;
    
    while(*line && isspace(*line)) line++;
    if(!*line || !key) return;
    
    s = line;
    while(*s && !isspace(*s)) s++;
    *key = g_strdup(line);
    (*key)[s-line] = '\0';
    while(*s && isspace(*s)) s++;
    if(!*s || !name) return;
    
    line = s;
    while(*s && *s != ':') s++;
    *name = g_strdup(line);
    (*name)[s-line] = '\0';
    if(*s) s++;
    if(*s) s++;
/* This is a bug?:   if(!*s || !line) return; */
    
    *value = g_strdup(s);
}

void mcp_handle_multiline_line(mcp_session *s, const char *line, int len) {
    char *key = NULL, *name = NULL, *value = NULL;
    mcp_multiline *m;

    mcp_parse_multiline(line, len, &key, &name, &value);
    if(key && name && value) {
        if((m = mcp_multiline_find(s, key))) {
            if(m->line)(m->line)(m, name, value);
        } else if(debug) {
            printf("Error in ML: cant find ML with key %s\n", key);
        }
    } else if(debug) {
        printf("Error in ML: >>%s<<\n", line);
    }
    g_free(key);
    g_free(name);
    g_free(value);    
}

void mcp_handle_multiline_end(mcp_session *s, const char *line, int len) {
    char *key = NULL;
    mcp_multiline *m;

    mcp_parse_multiline(line, len, &key, NULL, NULL);
    
    if(key && (m = mcp_multiline_find(s, key))) {
        if(m->end)(m->end)(m);
        s->multilines = g_list_remove(s->multilines, m);
        mcp_multiline_free(m);
    }
    g_free(key);
}

void gm_mcp_multiline_new(mcp_package *p,
                          const char *key,
                          void *data,
                          multiline_line_f handle_line,
                          multiline_end_f  handle_end) {
    mcp_multiline *m = g_malloc(sizeof(mcp_multiline));
    if(debug) printf("\x1b[34mAdding ML with key: %s\n\x1b[0m", key);
    m->key = g_strdup(key);
    m->belongs_to = p;
    m->data = data;
    m->line = handle_line;
    m->end = handle_end;
    p->session->multilines = g_list_append(p->session->multilines, m);
}

mcp_multiline *mcp_multiline_find(mcp_session *s, const char *key) {
    mcp_multiline *m;
    GList *l;
    for(l = s->multilines; l; l = g_list_next(l)) {
        m = l->data;
        if(strcmp(m->key, key) == 0)
            return m;
    }
    return NULL;
}

void mcp_multiline_free(mcp_multiline *m) {
    g_free(m->key);
    g_free(m);
}

/* Package stuff ****************************************************************/
void mcp_package_free(mcp_package *p) {
    if(p->free)(p->free)(p);
    g_free(p->name);
    g_free(p);
}


/* Support stuff ****************************************************************/
char *my_strndup(const char *s, int len) {
    char *ret = g_malloc(len + 1);
    strncpy(ret, s, len);
    ret[len] = '\0';
    return ret;
}

char *my_strndup_quoted(const char *s, int len) {
    int i,j;
    char *ret = g_malloc(len + 1);
    j = 0;
    for(i = 1; i < len - 1; i++) {
	if(s[i] == '\\') i++;
	ret[j] = s[i];
	j++;
    }
    ret[j] = '\0';
    return ret;
}

char **mcp_parse_line(const char *line, int len, int *count) {
    int b, e;
    char **ret = g_malloc(sizeof(char *) * len);
    b = e = 0;
    *count = 0;
    
    while(e < len) {
        for(b = e; line[b] == ' ' && b < len; b++);
        if(b >= len) break;
        e = b;
        if(line[b] == '\"') { /* quoted string */
            e = b + 1;
            while(line[e] != '\"' && e < len) {
                if(line[e] == '\\') {
                    if(line[e + 1] == '\\' || line[e + 1] == '\"') {
                        e += 2;
                    } else {
                        goto cleanup;
                    }
                } else {
                    e++;
                }
            }
            e++;
            ret[(*count)++] = my_strndup_quoted(line + b, e - b);
        } else { /* unquoted string */
            while(line[e] != ' ' && e < len) e++;
            ret[(*count)++] = my_strndup(line + b, e - b);
        }
    }
    return ret;
    
  cleanup: /* Some kind of error occured :( */
    if(debug) {
        printf("\tError in MCP string string, got:\n");
        for(b = 0; b < *count; b++)
            printf("\t\t>>%s<<\n", ret[b]);
    }
    
    for(b = 0; b < *count; b++) g_free(ret[b]);
    g_free(ret);
    return NULL;
}

/* Version stuff ***********/

int version_compare(const mcp_version *v1, const mcp_version *v2) {
    if(v1->major > v2->major) 
	return TRUE;
    else if(v1->major == v2->major && v1->minor >= v2->minor)
	return TRUE;
    else 
	return FALSE;
}

mcp_version *version_min(const mcp_version *v1, const mcp_version *v2) {
    mcp_version *ver = g_malloc(sizeof(mcp_version));
    if(v1->major > v2->major) {
	ver->major = v2->major;
	ver->minor = v2->minor;
    } else if(v1->major == v2->major) {
	ver->major = v2->major;
	ver->minor = MIN(v1->minor, v2->minor);
    } else { /* v1->major < v2->major */
	ver->major = v1->major;
	ver->minor = v1->minor;
    }
    return ver;
}

mcp_version *version_match(const mcp_version *client_min,
			   const mcp_version *client_max,
			   const mcp_version *server_min,
			   const mcp_version *server_max) {
    if(version_compare(client_max, server_min) &&
       version_compare(server_max, client_min))
	return version_min(server_max, client_max);
    else 	
	return NULL;
}

mcp_version *str_to_version(const char *version) {
    mcp_version *ver;
    int major, minor;
    if(sscanf(version, "%d.%d", &major, &minor) != 2) {
	return NULL;
    }
    ver = g_malloc(sizeof(mcp_version));
    ver->major = major;
    ver->minor = minor;
    return ver;
}


#define AUTHKEY_LENGTH 6
char *generate_key() {
    char *s = g_malloc(AUTHKEY_LENGTH + 1);
    int i;

    srand((int) time(NULL));

    for(i = 0; i < AUTHKEY_LENGTH; i++) {
	s[i] = (rand() % 94) + '!';
	if(strchr("\"\\!:$",s[i]))
	    i--;
    }
    s[i] = 0;
    return s;
}
