/*
 * Gmail. A Gnome email client.
 * Copyright (C) 1999-2000 Wayne Schuller
 *
 * matching.c - matching is the gmail feature which keeps an indexed record of which vfolders a msg matches too.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "main.h"

void rebuild_mi_values (gchar *query);
void recreate_matched_index (GtkWidget *widget, void *data);
gboolean mark_as_matched(gint id, GString *vfolders, gboolean overwrite); 
gboolean check_vfolder_matches(gint id, DTimer *timer);
static void notify_popup_match(gchar *name, DTimer *timer);

extern gboolean	DestroyRightPane(void);
extern gboolean check_db(MYSQL *mysql);
extern DTimer * msg_match_popup (void);
extern gboolean vfolder_msg_unread(gint id);
extern void mark_vfolder_unread_status_by_name(gchar *vfolder, gboolean unread_status);
extern void mark_vfolder_unread_status_by_node(GtkCTreeNode *node, gboolean unread_status);
extern void rebuild_all_mi_values (GtkWidget *widget, void *data);

extern _GmailApp_	*GmailApp;


#define set_status_label(label, text) gtk_label_set_text((GtkLabel*)label, text); while (g_main_iteration(FALSE));
#define set_status(a) set_status_label(GmailApp->pop3_status_label, a)

/* recreate_matched_index
 *
 * Update the matched 'set' design in the database according to the names
 * of the vfolders that the user has.
 *
 */
void
recreate_matched_index (GtkWidget *widget, void *data) 
{
	GtkWidget *dialog;
	GtkWidget *label;
	gint 	i;
	GString *query;
	gchar *msg;
	Mailbox	*mailbox;
	GtkCTreeNode *node;

	dialog = gnome_dialog_new(_("Gmail Warning:"),
                        GNOME_STOCK_BUTTON_OK,
                        GNOME_STOCK_BUTTON_CANCEL,
                        NULL);
 	if (GmailApp->app)
		gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(GmailApp->app));

	label = gtk_label_new(_("Warning: Recreating the matched index involves rebuilding one of the gmail tables.\nIt is required for fast gmail caching, but may take a few minutes on large databases.\nClick OK to continue."));
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),label, FALSE,FALSE,0);
	gtk_widget_show_all(dialog);
	/*set up an infinite loop*/
	for(;;) {
		i = gnome_dialog_run(GNOME_DIALOG(dialog));
		if	(i == 0) {
			gnome_dialog_close(GNOME_DIALOG(dialog));
			break;
		} else if(i == 1) {

			gnome_dialog_close(GNOME_DIALOG(dialog));
			return;
			break;
		} else if(i < 0) {

			/*the user closed the dialog from the window
       manager*/
			return;
			break;
		}
 	}


	/* We are going to modify the matched field according to vfolder design */
	query = g_string_new("ALTER TABLE display MODIFY matched SET(' '");


	/* Iterate over the vfolder ctree */
	node = gtk_ctree_node_nth(GTK_CTREE(GmailApp->mblist), 0);
	for (; node != NULL; node = GTK_CTREE_NODE_NEXT(node)) {
		mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (node));

		if (mailbox->name != NULL)
			g_string_sprintfa(query, ",'%s'", mailbox->name);
	}

	g_string_append(query, ") NOT NULL");

	/* Open Database. */
	/* if (!check_db(&GmailApp->mysql)) { return; }; */

	/* FIXME: use the appbar and be nicer about failure here. */

	if (mysql_query(&GmailApp->mysql, query->str) != 0) {
		msg = g_strdup_printf ("Query failed:\n%s\nSee the standard output to see the full query.\n", mysql_error(&GmailApp->mysql));
		dialog = gnome_message_box_new(msg, GNOME_MESSAGE_BOX_ERROR, "doh!", 0);
		gtk_window_set_modal(GTK_WINDOW (dialog), TRUE);
 		gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
 		gtk_widget_show(dialog);
		g_free(msg);	
		g_print("Failed query was: |%s|\n", query->str);
		g_string_free(query, TRUE);
		return;
		} 

	g_string_free(query, TRUE);

	/* Report to the user. */
	dialog = gnome_dialog_new (
                 "Gmail Popup",
                 GNOME_STOCK_PIXMAP_EXEC,
                 GNOME_STOCK_BUTTON_OK,
                 NULL);
	if (GmailApp->app)
             gnome_dialog_set_parent (GNOME_DIALOG (dialog), GTK_WINDOW (GmailApp->app));
	/* gtk_widget_set_usize(GTK_WIDGET(dialog), 200, 125);	 */
	label = gtk_label_new (_("Matched Index Updated\nYou should now rescan the database using \"Settings->Advanced->Rebuild MI Values\"\nTo do this immediately, click 'Exec', otherwise click 'OK' to go back to gmail."));
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), label, TRUE, TRUE, 0);


	gtk_widget_show_all(GNOME_DIALOG(dialog)->vbox);

	switch(gnome_dialog_run_and_close(GNOME_DIALOG (dialog))) {
		case 0: rebuild_all_mi_values(NULL, NULL);
			break;

		case 1: break;

		default: break;
	}
}

/* 
 * Rebuild the MI value for every msg returned by the query.
 */
void
rebuild_mi_values (gchar *query) {
	MYSQL_RES *res;
	MYSQL_ROW row;
	gint affected_rows;
	gchar text[256];
	GtkWidget *child;
	DTimer *timer = NULL;

	/* if (!check_db(&GmailApp->mysql)) { return; }; */

	/* Execute query.. */
	if (mysql_query(&GmailApp->mysql, query) != 0) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_free(query);
		return;
	} 

	/* Check the result. */	
	res = mysql_store_result(&GmailApp->mysql);

	if (res == NULL) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
		g_warning("vfolder match query returned NULL. :(");
		mysql_free_result(res); 
		return;
		}

	affected_rows = mysql_num_rows(res);

	if (affected_rows == 0) {
		gnome_appbar_set_status (GNOME_APPBAR (GmailApp->appbar), "Query returned empty result.");
		return;
		}

	timer = msg_match_popup();
	timer->value = 0;
	timer->total = affected_rows;


	/* Process every row, stop if the pop3_download_popup dissappears. */
	while ((row = mysql_fetch_row(res)) && GTK_IS_WIDGET(timer->dialog)) {
			gboolean result;

			/* Check matches for this msg. */
			result = check_vfolder_matches(atoi(row[0]), timer); 

			if (!result) {
				sprintf(text, ("DB ERROR\n"));
				set_status_label(timer->label, text);
				break;
				}

			timer->value++;	/* Increment count */ 

			sprintf(text, _("Matching msg %i of %i\n"), timer->value, timer->total);
			set_status_label(timer->label, text);
			}

	/* Close the database, free result. */
	mysql_free_result(res); 
	
	/* Update the button text if the dialog is still open. */
	if (timer && GTK_IS_WIDGET(timer->dialog)) {
		child = GTK_BIN (timer->button)->child;
		if (GTK_IS_LABEL(child)) gtk_label_set(GTK_LABEL(child), "Close");
		}
}

/* mark_as_matched - add a vfolder name to the msg's matched field.
 * Put the vfolder name in the matched field of msg 'id' in database.
 * When check_vfolder_matches finds that an id is matched in a vfolder
 * query, it calls this function to put the name of that vfolder in it's
 * matched field.
 * Overwrite whatever is in the matched field currently if they want to.
 * If overwrite is false, we simply append vfolders to the current matched
 * field (but we check that it isn't there first.)
 */
gboolean
mark_as_matched(gint id, GString *vfolders, gboolean overwrite) {
	MYSQL_RES *res;
	MYSQL_ROW row;
	gchar *query;
	gboolean finished = FALSE;

	/* Open the database. (use second connection)*/
	if (!check_db(&GmailApp->mysql2)) { return(FALSE); };

	if (!overwrite) {
		/* We need to only add what is in the string. We assume there is
		 * only one vfolder being added.
		 * The first thing is to check that the name isn't already there
		 */

		if (vfolders->len <= 1) {
			return(TRUE); /* nothing to do */
			}


		/* FIXME: I think this query is flawed because it will get a
		 * match if the vfolder name is a superset of the current one.
		 */
		query = g_strdup_printf("SELECT (matched) FROM display where id = %i", id);

		/* Execute query.. */
		if (mysql_query(&GmailApp->mysql2, query) != 0) {
			g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql2));
			g_free(query);
			return(FALSE);
		} 

		res = mysql_use_result(&GmailApp->mysql2);

		if (res == NULL) {
			g_warning("Result failed: %s\n", mysql_error(&GmailApp->mysql2));
			g_free(query);
			return(FALSE);
			}

		row = mysql_fetch_row(res);


		/* Check if vfolder name is in this row. */
		if (strstr(row[0], vfolders->str) == NULL) {
				g_string_append(vfolders, ",");
				g_string_append(vfolders, row[0]);
				/* g_print("msg id %d matched value will be:  %s\n", id, vfolders->str);*/
				finished = FALSE;
			} else finished = TRUE; /* Our vfolder already is in the field. */

		/* Close the database connection. */
		g_free(query);
		mysql_free_result(res); 
		if (finished) {
			return(TRUE);
		}
	}

	if (vfolders->len == 0) g_string_assign(vfolders, "NULL");

	/* g_print("inserting: %s\n", vfolders->str); */

	query = g_strdup_printf("UPDATE display SET matched=\"%s\" WHERE id = %i", vfolders->str, id);

	/* Execute query.. */
	if (mysql_query(&GmailApp->mysql2, query) != 0) {
		g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql2));
		g_free(query);
		return(FALSE);
	} 

	/* g_print("Successfully marked %d as: %s\n", id, vfolders->str);  */

	g_free(query);
	return(TRUE);
}

/* check_vfolder_matches - check which vfolders this msg belongs to.
 *
 * Go through every vfolder and see if this message belongs to that query.
 * If so add the name of that vfolder to the msg's 'matched' field.
 * This function clears the current contents of the 'matched' field.
 *
 * This code is called once for every message on the server, and within it
 * there is a loop through every vfolder. Thus there are lots of iterations.
 * Any mem leaks or sub-optimal things will really drag out gmail.
 *
 * FIXME: Check for mem leaks here.
 */
gboolean 
check_vfolder_matches(gint id, DTimer *timer) 
{
	MYSQL_RES *res;
	gchar *newquery;
	GString *matches; /* Names of vfolders that match. */
	gint affected_rows;
	gboolean found_match = FALSE;
	gboolean msg_unread = FALSE; 
	gboolean result = TRUE;
	Mailbox	*mailbox;
	GtkCTreeNode *node;

	matches = g_string_new(NULL);

	msg_unread = vfolder_msg_unread(id);

	/* if (!check_db(&GmailApp->mysql)) { return; }; */

	/* Iterate over the vfolder ctree */
	node = gtk_ctree_node_nth(GTK_CTREE(GmailApp->mblist), 0);
	for (; node != NULL; node = GTK_CTREE_NODE_NEXT(node)) {
		mailbox = gtk_ctree_node_get_row_data(GTK_CTREE (GmailApp->mblist), GTK_CTREE_NODE (node));

		/* Don't match Inbox or Unread.
		 * Anything that doesn't get matched in this loop is matched 
		 * against the Inbox. 
		 *
		 * We handle Unread manually because if we matched it in this
		 * loop it would get excluded from Inbox. 
		 * We only want to exclude from the Inbox anything which 
		 * matches the users custom vfolders.
		 * 
		 * By the same logic we shouldn't be matching against Outbox 
		 * in here either, but we do because Inbox excludes outgoing 
		 * messages by definition, so it doesn't matter if this loop 			 * does it for us.
		 *
		 * If you don't get this read the online docs for gmail
		 * and make sure you understand how the matched index works.
		 */	
		if ((strcmp(mailbox->name, "Inbox") != 0) && (strcmp(mailbox->name, "Unread") != 0) && (mailbox->query != NULL))	{

			/* g_print("is %d in %s? (mailbox->query: %s)\n", id, mailbox->name, mailbox->query); */

			newquery = g_strdup_printf("SELECT display.id FROM display,details WHERE display.id=details.id AND display.id = \"%d\" AND (%s)", id, mailbox->query); 

			/* g_print("oldquery: %s\n", mailbox->query);
			g_print("newquery: %s\n", newquery);  */


			/* Execute query.. */
			if (mysql_query(&GmailApp->mysql, newquery) != 0) {
				g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
				g_print("match query was: %s\n", newquery);

 				/* Abort this vfolder but keep going on the matching. */
				g_free(newquery);
				continue; 
			} 
	
			/* Check the result. */	
			res = mysql_store_result(&GmailApp->mysql);
		
			if (res == NULL) {
				g_warning("Query failed: %s\n", mysql_error(&GmailApp->mysql));
				g_warning("vfolder match query returned NULL. :(");
 				/* Abort this vfolder but keep going on the matching. */
				g_free(newquery);
				continue;
				}

			affected_rows = mysql_num_rows(res);

			/* We're finished with the results now. */
			mysql_free_result(res); 
			g_free(newquery);

			switch (affected_rows) {
				case 0:
				/* 0 matches - just ignore and keep going.*/
				/* g_print("id %d didn't match %s\n", id, mailbox->name);  */
				break;

				case 1:
				/* We have a match! */
				g_string_append(matches, mailbox->name);
				g_string_append_c(matches, ','); 

				/* If the message is unread, then mark the vfolder as
				 * having an unread message at this point.
				 */
				if (msg_unread) {
					mark_vfolder_unread_status_by_node(node, TRUE);
					}

				/* Send output to user.
				 */
				if (timer != NULL)
					notify_popup_match(mailbox->name, timer);

				found_match = TRUE; /* remember this */
				break;

				case -1:
				/* error */
				g_print("Affected rows error: %s\n", mysql_error(&GmailApp->mysql));
				break;

				default:
				/* If it matched more than one it's probably 
				 * due to the AND clause we add confuses some 
				 * other logical operator.
         			 * We need people to use brackets around 
				 * logical clauses.
				 */
				g_print("vfolder match (%d on %s) returned weird result: %d rows. You probably need to put round brackets around your vfolder queries and try again.\n", id, mailbox->name, affected_rows);
				break;
				}

			}
	}

	/* If we haven't found a match we consider it part of the Inbox. 
	 */
	if (!found_match) {

		/* Leftovers match the Inbox vfolder. (assumed to always exist) */
		g_string_append(matches, "Inbox");
		g_string_append_c(matches, ','); 

		/* Add it to the download report. */
		if (timer != NULL)
			notify_popup_match("Inbox", timer);

		/* Bold the vfolder name if the message is unread. */
		if (msg_unread) 
			mark_vfolder_unread_status_by_name("Inbox", TRUE);
		}

	if (msg_unread) {

		/* It matches the unread vfolder. (assumed to always exist)*/
		g_string_append(matches, "Unread");
		g_string_append_c(matches, ',');

		/* We don't add it to the download report because if it is a 
		 * download it is obvious to the user that the message would be
		 * unread.
		 * FIXME: But it might be useful to see the number of unread
		 * messages if they are rebuilding their matched fields.
		 */

		/* Bold the vfolder name. */
		mark_vfolder_unread_status_by_name("Unread", TRUE);
		}

	/* Update the matched index for this message.
	 * If there is a problem in mark_as_matched (eg: db lost) pass it
	 * back to the calling function.
	 */
	result = mark_as_matched(id, matches, TRUE); 

	g_string_free(matches, TRUE);

	return(result);
}

/* notify_popup_match - Add another matched msg to the popup menu. */
static void 
notify_popup_match(gchar *name, DTimer *timer) 
{
	gint row;

	g_return_if_fail(timer != NULL);
	g_return_if_fail(GTK_IS_WIDGET(timer->dialog));
	g_return_if_fail(GTK_IS_WIDGET(timer->clist));

	row = gtk_clist_find_row_from_data(GTK_CLIST(timer->clist), name);

	/* gtk_clist_freeze(GTK_CLIST(timer->clist)); */

	if (row == -1) {
		gchar *newtext[2];

		/* Create a new row. */
		newtext[0] = g_strdup(name);
		newtext[1] = g_strdup("1");
		gtk_clist_prepend (GTK_CLIST (timer->clist), newtext);
		gtk_clist_set_row_data(GTK_CLIST (timer->clist), 0, name);

		g_free(newtext[0]);
		g_free(newtext[1]);

	} else {
		gint val;
		gchar text[255], *orig;

		/* Increment row by one. This part of the function gets
		 * called lots.
		 */

		gtk_clist_get_text(GTK_CLIST(timer->clist), row, 1, &orig);
		val = atoi(orig); /* Convert it to an int. */
		sprintf(text, "%i", val + 1); /* Increment the value. */	
		gtk_clist_set_text(GTK_CLIST(timer->clist), row, 1, text);
	}

	/* gtk_clist_thaw(GTK_CLIST(timer->clist)); */
}
