/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/***********************
 * Configuration file. *
 ***********************/

#include "gliv.h"

#ifndef HAVE_GETLINE
#include "../lib/getline.h"
#endif

#include <unistd.h>             /* R_OK */
#include <stdio.h>              /* FILE, f*(), perror(), getline() */
#include <string.h>             /* strlen(), memcpy() */

static GHashTable *table = NULL;

/* The link between options and the rcfile. */
static options_struct opts = {
/* Default options */
    .bg_col = {0, 0, 0},
    .alpha1 = {0x6600, 0x6600, 0x6600},
    .alpha2 = {0x9900, 0x9900, 0x9900},
    .fullscreen = FALSE,
    .maximize = FALSE,
    .scaledown = FALSE,
    .menu_bar = TRUE,
    .status_bar = TRUE,
    .zoom_pointer = FALSE,
    .alpha_checks = TRUE,
    .dither = FALSE,
    .force = FALSE,
    .build_menus = TRUE,
    .delay = 0,
    .history_size = 1000
};

typedef struct {
    const gchar *name;
    gint *option;
    const gchar *comment;
    const gboolean is_bool;
} option_struct;

/* *INDENT-OFF* */
static option_struct option_names[] = {
/* To fill the hash table and the configuration file. */
{ "full-screen",  &opts.fullscreen,   N_("Start in full screen mode"),      1 },
{ "maximize",     &opts.maximize,     N_("Maximize small images"),          1 },
{ "scale-down",   &opts.scaledown,    N_("Scale down larges images"),       1 },
{ "menu",         &opts.menu_bar,     N_("Display the menu bar"),           1 },
{ "info",         &opts.status_bar,   N_("Show infos about current image"), 1 },
{ "zoom-pointer", &opts.zoom_pointer, N_("Zoom centered on pointer"),       1 },
{ "alpha-checks", &opts.alpha_checks, N_("Alpha checks in the background"), 1 },
{ "dither",       &opts.dither,       N_("Dithering"),                      1 },
{ "force-load",   &opts.force,        N_("Try to load every file"),         1 },
{ "build-menus",  &opts.build_menus,  N_("Build images menus at startup"),  1 },
{ "delay",        &opts.delay,        N_("Delay before hiding the cursor"), 0 },
{ "history",      &opts.history_size, N_("Length of history"),              0 },
{ "bg_col_red",   opts.bg_col,        N_("background : red channel"),       0 },
{ "bg_col_green", opts.bg_col + 1,    N_("background : green channel"),     0 },
{ "bg_col_blue",  opts.bg_col + 2,    N_("background : blue channel"),      0 },
{ "alpha1_red",   opts.alpha1,        N_("alpha1 tile : red channel"),      0 },
{ "alpha1_green", opts.alpha1 + 1,    N_("alpha1 tile : green channel"),    0 },
{ "alpha1_blue",  opts.alpha1 + 2,    N_("alpha1 tile : blue channel"),     0 },
{ "alpha2_red",   opts.alpha2,        N_("alpha2 tile : red channel"),      0 },
{ "alpha2_green", opts.alpha2 + 1,    N_("alpha2 tile : green channel"),    0 },
{ "alpha2_blue",  opts.alpha2 + 2,    N_("alpha2 tile : blue channel"),     0 },
{ NULL,           NULL,               NULL,                                 0 }
};
/* *INDENT-ON* */

/*
 * Maximum length of the option names, currently
 * it is strlen("zoom-pointer") == 12.
 * Used to indent the option file.
 */
#define MAX_OPT_LEN 12

static gchar *find_rcfile(gboolean test)
{
    gchar *path;

    path = g_build_filename(g_get_home_dir(), ".glivrc", NULL);

    if (test == FALSE || g_file_test(path, G_FILE_TEST_EXISTS) == TRUE)
        /* When loading the file, we really try to access it. */
        /* When saving, we just need the filename. */
        return path;

    g_free(path);
    return NULL;
}

/*** Loading options. ***/

static void init_hash_table(void)
{
    guint i;

    table = g_hash_table_new(g_str_hash, g_str_equal);

    /* (i + 1), because 0 is when no option is found, not the first option. */
    for (i = 0; option_names[i].name != NULL; i++)
        g_hash_table_insert(table, (gchar *) option_names[i].name,
                            GINT_TO_POINTER(i + 1));
}

 /* Processes spaces and '#'. */
static gchar *clean_str(gchar * str)
{
    gchar *new_str;
    gchar *ptr;

    new_str = g_new(gchar, strlen(str) + 1);

    for (ptr = new_str; *str != '\n'; str++) {
        if (*str == ' ')
            continue;

        if (*str == '#') {
            g_free(new_str);
            return NULL;
        }

        *ptr = *str;
        ptr++;
    }
    *ptr = '\0';

    new_str = g_renew(gchar, new_str, ptr - new_str + 1);
    return new_str;
}

static void process_line(gchar * line)
{
    gchar **res;
    guint hash_index;
    gchar *clean;

    if (*line == '\n')
        /* Skip this line. */
        return;

    clean = clean_str(line);
    if (clean == NULL)
        /* This line is a '#' comment. */
        return;

    /* res[0] : option name ; res[1] : value */
    res = g_strsplit(clean, "=", 2);

    g_free(clean);

    if (res[0] != NULL && res[1] != NULL && res[2] == NULL) {
        /* No error during split. */
        hash_index = GPOINTER_TO_INT(g_hash_table_lookup(table, res[0]));

        if (hash_index != 0) {
            /* Option found. */
            hash_index--;

            if (option_names[hash_index].is_bool == TRUE) {

                if (g_strcasecmp(res[1], "true") == 0)
                    *((gboolean *) option_names[hash_index].option) = TRUE;

                else if (g_strcasecmp(res[1], "false") == 0)
                    *((gboolean *) option_names[hash_index].option) = FALSE;

            } else
                /* option_names[hash_index].is_bool == FALSE */
                *(option_names[hash_index].option) =
                    (gint) g_strtod(res[1], NULL);
        }
    }

    g_strfreev(res);
}

options_struct *load_rc(gboolean from_file)
{
    gchar *filename;
    FILE *file;
    gchar *line = NULL;
    size_t nb = 0;

    if (from_file == FALSE)
        return &opts;

    init_hash_table();

    filename = find_rcfile(TRUE);
    if (filename == NULL)
        return &opts;

    file = fopen(filename, "r");
    if (file == NULL) {
        /*
         * Not very frequent because find_rcfile(TRUE)
         * should have made us return.
         */
        perror(filename);
        return &opts;
    }

    for (;;) {
        getline(&line, &nb, file);

        if (feof(file) == 0)
            process_line(line);
        else
            /* End Of File. */
            break;
    }

    g_free(line);
    fclose(file);
    g_free(filename);
    g_hash_table_destroy(table);

    return &opts;
}

/*** Saving options. ***/

static void write_option_line(FILE * f, guint index)
{
    guint i;
    guint value;

    fputs(option_names[index].name, f);

    for (i = strlen(option_names[index].name); i < MAX_OPT_LEN; i++)
        fputc(' ', f);

    value = *(option_names[index].option);

    if (option_names[index].is_bool == TRUE)
        fprintf(f, " = %s\n\n", (value == TRUE ? "True" : "False"));
    else
        fprintf(f, " = %d\n\n", value);
}

static void write_rc(gchar * filename)
{
    FILE *file;
    guint i;

    file = fopen(filename, "w");
    if (file == NULL) {
        perror(filename);
        return;
    }

    fprintf(file, _("# Configuration file for GLiv %s\n\n"), VERSION);
    fputs(_("# Option names are case sensitive.\n"), file);
    fputs(_("# Option values are case insensitive.\n\n"), file);
    fputs(_("# Note : 'maximize = True' or 'scale-down = True'\n"), file);
    fputs(_("# implies 'full-screen = True'.\n\n"), file);

    for (i = 0; option_names[i].name != NULL; i++) {
        /* The comment line. */
        fprintf(file, "# %s\n", _(option_names[i].comment));

        /* The option line. */
        write_option_line(file, i);
    }

    fclose(file);
    g_free(filename);
}

void save_rc(options_struct * opt)
{
    memcpy(&opts, opt, sizeof(options_struct));
    write_rc(find_rcfile(FALSE));
}
