/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/*********************
 * History handling. *
 *********************/

#include "gliv.h"

extern options_struct *options;

static GList *list = NULL;
static GList *node = NULL;      /* Currrent position in the list. */

/* 
 * Number of elements in the list, it would take 50 days with 1 event
 * each millisecond to overflow, so we feel safe.
 */
static guint32 count;

static gboolean used;           /* Wether the history has been initialized. */

void init_history(void)
{
    if (options->history_size != 0) {
        node = list = g_new0(GList, 1);
        count = 0;
        used = TRUE;
    } else
        used = FALSE;
}

/* Returns item->next. */
static GList *free_item(GList * item)
{
    GList *next;

    if (item->prev != NULL)
        item->prev->next = NULL;

    if (item->next != NULL)
        item->next->prev = NULL;

    g_free(item->data);
    next = item->next;
    g_list_free_1(item);
    count--;

    return next;
}

static void free_list(GList * item)
{
    while (item != NULL)
        item = free_item(item);
}

static void clean_list(void)
{
    while (count > (guint32) options->history_size) {
        if (node == list)
            node = node->next;
        list = free_item(list);
    }

    list->prev = NULL;
}

void append_history(void)
{
    GList *item;

    if (used == FALSE || options->history_size == 0)
        return;

    item = g_list_alloc();
    item->data = g_new(gdouble, 8);

    matrix_cpy(item->data, NULL);

    item->prev = node;
    item->next = NULL;

    if (list == NULL)
        list = item;
    else {
        if (node->next != NULL)
            free_list(node->next);

        node->next = item;
    }

    node = item;
    count++;

    if (options->history_size > 0)
        clean_list();
}

static void set_state(GList * item)
{
    if (item != NULL && item->data != NULL) {
        node = item;
        matrix_cpy(NULL, item->data);
        refresh(REFRESH_IMAGE | REFRESH_STATUS);
    }
}

void undo(void)
{
    set_state(g_list_previous(node));
}

void redo(void)
{
    set_state(g_list_next(node));
}

void clear_history(void)
{
    if (list != NULL) {
        free_list(list);
        node = list = NULL;
        count = 0;
        init_history();
        append_history();
    }
}
