/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/************************************************
 * Creation and callbacks of the OpenGL widget. *
 ************************************************/

#include "gliv.h"

#include <stdlib.h>             /* exit() */
#include <GL/gl.h>
#include <gtkgl/gtkglarea.h>

extern GtkWidget *gl_widget;
extern rt_struct *rt;
extern options_struct *options;

static gint old_width, old_height;

void update_bg_color(gboolean do_gl)
{
    GdkColormap *cmap;
    GdkColor col;

    cmap = gtk_widget_get_colormap(gl_widget);

    col.red = (guint16) options->bg_col[0];
    col.green = (guint16) options->bg_col[1];
    col.blue = (guint16) options->bg_col[2];

    gdk_colormap_alloc_color(cmap, &col, FALSE, TRUE);

    /*
     * Make the window background the same as OpenGL one. This
     * is used to avoid flickers when the widget size changes.
     * This is fixed in GTK2 but doesn't hurt.
     */
    gdk_window_set_background(gl_widget->window, &col);

    if (do_gl == TRUE)
        glClearColor(options->bg_col[0] / 65535.0, options->bg_col[1] / 65535.0,
                     options->bg_col[2] / 65535.0, 0.0);
}

gboolean toggle_cursor(gboolean visible)
{
    static GdkCursor *hidden_cursor = NULL;

    GdkPixmap *pmap;
    GdkColor *col;
    gchar *data;
    GdkCursor *new_cursor;

    if (hidden_cursor == NULL) {
        /* First time. */
        data = g_new0(gchar, 1);
        col = g_new0(GdkColor, 1);
        pmap = gdk_bitmap_create_from_data(NULL, data, 1, 1);
        hidden_cursor = gdk_cursor_new_from_pixmap(pmap, pmap, col, col, 0, 0);
        g_free(data);
        g_free(col);
        gdk_bitmap_unref(pmap);
    }

    if (visible == TRUE) {
        /* Show the cursor. */
        new_cursor = NULL;
        schedule_hide_cursor();
    } else {
        /* Hide the cursor. */
        new_cursor = hidden_cursor;
        gtk_widget_add_events(gl_widget, GDK_POINTER_MOTION_MASK);
    }

    rt->cursor_hidden = !visible;
    gdk_window_set_cursor(gl_widget->window, new_cursor);

    return FALSE;
}

void schedule_hide_cursor(void)
{
    static guint id = 0;

    if (id != 0)
        /* Remove previous schedule. */
        g_source_remove(id);

    if (options->delay == 0)
        id = 0;
    else
        id = g_timeout_add(options->delay, (GSourceFunc) toggle_cursor,
                           GINT_TO_POINTER(FALSE));
}

static gboolean expose_event(GtkWidget * unused, GdkEventExpose * event)
{
    if (event->count == 0)
        refresh(REFRESH_IMAGE);

    return TRUE;
}

/*
 * Probe the max_texture_size with a GL_PROXY_TEXTURE_2D.
 * Used only in the gl_widget_map calback.
 */
static gboolean check_texture_size(void)
{
    guint width;

    glTexImage2D(GL_PROXY_TEXTURE_2D, 0, GL_RGB,
                 rt->max_texture_size, rt->max_texture_size, 0, GL_RGBA,
                 GL_UNSIGNED_BYTE, NULL);

    glGetTexLevelParameteriv(GL_PROXY_TEXTURE_2D, 0, GL_TEXTURE_WIDTH, &width);

    return (width == rt->max_texture_size);
}

static void resize_widget(gint width, gint height)
{
    gdk_gl_wait_gdk();

    glViewport(0, 0, width, height);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glOrtho(-width / 2.0, width / 2.0, height / 2.0, -height / 2.0, -1.0, 1.0);

    glMatrixMode(GL_MODELVIEW);
}

static gboolean configure_event(GtkWidget * unused, GdkEventConfigure * event)
{
    guint what = REFRESH_IMAGE;

    resize_widget(event->width, event->height);

    if ((options->maximize == TRUE || options->scaledown == TRUE) &&
        matrix_set_max_zoom(old_width, old_height, FALSE) == FALSE) {
        /*
         * When options->maximize or options->scaledown is set we zoom only
         * if a matrix_set_max_zoom() with previous dimensions would have been
         * ignored. It happens if the image properties didn't change after the
         * previous matrix_set_max_zoom().
         * This is achieved by the test of matrix_set_max_zoom().
         */
        matrix_set_max_zoom(event->width, event->height, TRUE);
        what |= REFRESH_STATUS;
    }

    refresh(what);

    old_width = event->width;
    old_height = event->height;

    return TRUE;
}

/*
 * This callback is called only once, after the first
 * expose event to be sure the image is displayed and then,
 * load the next image.
 */
static gboolean first_expose(gpointer object)
{
    refresh(REFRESH_IMAGE);
    g_signal_handlers_disconnect_by_func(object, G_CALLBACK(first_expose),
                                         NULL);
    while (gtk_events_pending() != 0)
        gtk_main_iteration_do(FALSE);

    load_second_image();
    return TRUE;
}

static gboolean gl_widget_map(void)
{
    gtk_gl_area_make_current(GTK_GL_AREA(gl_widget));

    glEnable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);
    glShadeModel(GL_FLAT);
    glDisable(GL_POLYGON_SMOOTH);
    glDisable(GL_LINE_SMOOTH);
    glDisable(GL_POINT_SMOOTH);
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);

    update_bg_color(TRUE);

    glGetIntegerv(GL_MAX_TEXTURE_SIZE, &rt->max_texture_size);
    while (rt->max_texture_size > 0 && check_texture_size() == FALSE)
        rt->max_texture_size >>= 1;

#ifdef TEST_TILER
    rt->max_texture_size = 64;
#endif

    rt->wid_size = &gl_widget->allocation;

    g_signal_connect(gl_widget, "configure_event",
                     G_CALLBACK(configure_event), NULL);

    g_signal_connect(gl_widget, "expose_event", G_CALLBACK(first_expose), NULL);

    g_signal_connect(gl_widget, "expose_event", G_CALLBACK(expose_event), NULL);

    load_first_image();

    schedule_hide_cursor();

    old_width = rt->wid_size->width;
    old_height = rt->wid_size->height;

    if (options->fullscreen == TRUE) {
        toggle_fullscreen(TRUE);

        resize_widget(rt->wid_size->width, rt->wid_size->height);
    }

    return TRUE;
}

void create_gl_widget(void)
{
    gint attrlist[] = {
        GDK_GL_RGBA,
        GDK_GL_RED_SIZE, 1,
        GDK_GL_GREEN_SIZE, 1,
        GDK_GL_BLUE_SIZE, 1,
        GDK_GL_DOUBLEBUFFER,
        GDK_GL_NONE
    };

    if (gdk_gl_query() == FALSE) {
        g_printerr(_("OpenGL not supported\n"));
        exit(1);
    }

    gl_widget = gtk_gl_area_new(attrlist);
    if (gl_widget == NULL) {
        g_printerr(_("Cannot create an OpenGL widget.\n"));
        exit(1);
    }

    gtk_widget_set_events(gl_widget,
                          GDK_BUTTON_PRESS_MASK | GDK_BUTTON1_MOTION_MASK |
                          GDK_BUTTON3_MOTION_MASK | GDK_BUTTON_RELEASE_MASK);

    if (options->delay != 0)
        gtk_widget_add_events(gl_widget, GDK_POINTER_MOTION_MASK);

    g_signal_connect_after(gl_widget, "map", G_CALLBACK(gl_widget_map), NULL);
}
