/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * See the COPYING file for license information.
 *
 * Guillaume Chazarain <booh@altern.org>
 */

/**************************************************************
 * getline() is a GNU extension so here is an implementation. *
 **************************************************************/

#include <stdlib.h>             /* malloc(), realloc() */
#include <string.h>             /* strchr() */
#include "getline.h"

/* Number of chars read at a time. */
#define SIZE 64

static char *buf;

static ssize_t line_length(FILE * f)
{
    char *tmp;
    long offset;
    ssize_t len = 0;

    /* Remember the current position. */
    offset = ftell(f);

    /* Walk until the end of the file and stop on newline. */
    while (feof(f) == 0) {
        fread(buf, sizeof(char), SIZE, f);
        tmp = strchr(buf, '\n');
        if (tmp == NULL)
            len += SIZE;
        else {
            len += tmp - buf;
            break;
        }
    }

    /* Then go back. */
    fseek(f, offset, SEEK_SET);

    /* But remember the distance. */
    return len;
}

ssize_t getline(char **LINEPTR, size_t * N, FILE * STREAM)
{
    ssize_t nb;

    /* Initialize the buffer. */
    buf = malloc((SIZE + 1) * sizeof(char));
    buf[SIZE] = '\0';

    /* + 1 for the newline. */
    nb = line_length(STREAM) + 1;

    if (*N == 0)
        *LINEPTR = malloc((nb + 1) * sizeof(char));
    else if (*N < nb + 1)
        *LINEPTR = realloc(*LINEPTR, (nb + 1) * sizeof(char));

    fread(*LINEPTR, sizeof(char), nb, STREAM);
    (*LINEPTR)[nb] = '\0';

    *N = nb + 1;

    free(buf);
    return nb;
}
