/* GKrellMSS - GKrellM Sound Scope
|  Copyright (C) 2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellmss.h"

#include "../pixmaps/krell_vu.xpm"
#include "../pixmaps/krell_vu_peak.xpm"

#include "../pixmaps/button_sweep_dn.xpm"
#include "../pixmaps/button_sweep_up.xpm"

#include "../pixmaps/krell_sensitivity.xpm"

/* ======================================================================== */
#define CONFIG_NAME		"Sound Scope"
#define CONFIG_KEYWORD	"sound"
#define STYLE_NAME		"sound"

#define	LOW_SWEEP	100
#define	HIGH_SWEEP	50000

#define DEFAULT_CHART_HEIGHT 30

SoundMonitor		*sound;
Oscope				*oscope;

static Monitor		*mon_scope;
Chart				*chart;
static ChartConfig	*chart_config;

static gint			style_id;
static gboolean		extra_info = TRUE;

static GdkPixmap	*button_sweep_up_pixmap,
					*button_sweep_dn_pixmap;
static GdkBitmap	*button_sweep_up_mask,
					*button_sweep_dn_mask;

static Decal		*decal_sweep_up,
					*decal_sweep_dn;

static Decal		*decal_usec;

static Krell		*krell_in_motion;

static gint			vu_meter_width;

static void
oscope_horizontal_scaling(void)
	{
	gchar	buf[32];

	oscope->t_sample = 1 / (gfloat) SAMPLE_RATE;
	oscope->dx_per_point = 0;
	do
		{
		++oscope->dx_per_point;
		oscope->t_trace = (gfloat) oscope->usec_per_div * 1e-6;
		oscope->t_trace *= HORIZONTAL_DIVS;
		oscope->samples_per_point = oscope->t_trace / oscope->t_sample
					/ (gfloat) chart->w * (gfloat) oscope->dx_per_point;
		}
	while (oscope->samples_per_point < 1);

	if (oscope->usec_per_div >= 1000)
		sprintf(buf, "%d msec", oscope->usec_per_div / 1000);
	else
		sprintf(buf, "%d usec", oscope->usec_per_div);
	gkrellm_draw_decal_text(NULL, decal_usec, buf, -1);
	}

static void
trigger_delay(gint channel)
	{
	SoundSample	*ss = sound->buffer;
	gfloat		flimit;
	gint		y0, i, n, limit, trigger, delay;
	
	trigger = (gint) (oscope->vert_trigger * oscope->vert_max);
	delay = -2;
	flimit = oscope->samples_per_point;
	limit = flimit;
	for (i = 0; i < sound->buf_count - limit; )
		{
		for (y0 = 0, n = 0; n < limit; ++n)
			{
			if (channel == CHANNEL_L)
				y0 += ss[i].left;
			else if (channel == CHANNEL_R)
				y0 += ss[i].right;
			else if (channel == CHANNEL_LR)
				y0 += (ss[i].left + ss[i].right) / 2;
			}
		y0 = y0 / limit;
		if (y0 < trigger)
			delay = -1;
		if (y0 >= trigger && delay == -1)
			{
			delay = i;
			break;
			}
		flimit += oscope->samples_per_point;
		i = flimit;
		}
	sound->buf_index = (delay < 0) ? 0 : delay;
	}


void
draw_oscope_line_trace(gint channel)
	{
	SoundSample	*ss = sound->buffer;
	gfloat	flimit;
	gint	x, x0, y0, y1, i, limit, n;

	oscope->gc = gkrellm_draw_GC(1);
	gdk_gc_set_foreground(oscope->gc, gkrellm_in_color());

	y1 = oscope->y_append;
	x0 = oscope->x_append;
	oscope->x_append = 0;
	oscope->y_append = 0;
	flimit = sound->buf_index + oscope->samples_per_point;
	for (i = sound->buf_index, x = x0; x < chart->w; x += oscope->dx_per_point)
		{
		limit = flimit;
		if (limit >= sound->buf_count - 1)
			{
			oscope->y_append = y1;
			oscope->x_append = x;
			break;
			}
		for (n = 0, y0 = 0; i < limit; ++i, ++n)
			{
			if (channel == CHANNEL_L)
				y0 += ss[i].left;
			else if (channel == CHANNEL_R)
				y0 += ss[i].right;
			else if (channel == CHANNEL_LR)
				y0 += (ss[i].left + ss[i].right) / 2;
			}
		y0 /= n;
		y0 = (chart->h / 2) * -y0 / oscope->vert_max;
		y0 += chart->h / 2;
		if (x > 0)
			{
			gdk_draw_line(chart->pixmap, oscope->gc,
					x - oscope->dx_per_point, y1, x, y0);
			}
		y1 = y0;
		flimit += oscope->samples_per_point;
		}
	sound->buf_index = 0;
	sound->buf_count = 0;
	}

void
draw_oscope_bar_trace(gint channel)
	{
	SoundSample	*ss = sound->buffer;
	gfloat	flimit;
	gint	x, x0, y, y0, y1, i, limit, n;
	gint	y0_prev, y1_prev;

	oscope->gc = gkrellm_draw_GC(1);
	gdk_gc_set_foreground(oscope->gc, gkrellm_in_color());

	y = y0 = y1 = 0;
	y0_prev = y1_prev = oscope->y_append;
	x0 = oscope->x_append;
	oscope->x_append = 0;
	oscope->y_append = 0;
	flimit = sound->buf_index + oscope->samples_per_point;
	for (i = sound->buf_index, x = x0; x < chart->w; x += oscope->dx_per_point)
		{
		limit = flimit;
		if (limit >= sound->buf_count - 1)
			{
			oscope->y_append = (y1_prev + y0_prev) / 2;
			oscope->x_append = x;
			break;
			}
		for (n = 0; i < limit; ++i, ++n)
			{
			if (channel == CHANNEL_L)
				y = ss[i].left;
			else if (channel == CHANNEL_R)
				y = ss[i].right;
			else if (channel == CHANNEL_LR)
				y = (ss[i].left + ss[i].right) / 2;
			else
				break;
			if (n == 0)
				y0 = y1 = y;
			else
				{
				if (y < y0)
					y0 = y;
				if (y > y1)
					y1 = y;
				}
			if (x > 0)
				{
				if (y0 > y1_prev)
					y0 = y1_prev;
				if (y1 < y0_prev)
					y1 = y0_prev;
				}
			}
		y0_prev = y0;
		y1_prev = y1;

		y0 = (chart->h / 2) * -y0 / oscope->vert_max;
		y1 = (chart->h / 2) * -y1 / oscope->vert_max;
		y0 += chart->h / 2;
		y1 += chart->h / 2;
		gdk_draw_line(chart->pixmap, oscope->gc, x, y1, x, y0);
		flimit += oscope->samples_per_point;
		}
	sound->buf_index = 0;
	sound->buf_count = 0;
	}

void
oscope_trace(gint channel)
	{
	if (oscope->dx_per_point > 1)
		draw_oscope_line_trace(channel);
	else
		draw_oscope_bar_trace(channel);
	}

static void
draw_oscope_grid(void)
	{
	GdkImage	*grid_image;
	GdkGC		*gc;
	GdkColor	color;
	gint		h, x, dx;

	/* Draw grid lines on bg_src_pixmap so gkrellm_clear_chart_pixmap()
	|  will include grid lines.
	*/
	gkrellm_clean_bg_src_pixmap(chart);
	gkrellm_draw_chart_grid_line(chart, chart->bg_src_pixmap, chart->h / 4);
	gkrellm_draw_chart_grid_line(chart, chart->bg_src_pixmap, chart->h / 2);
	gkrellm_draw_chart_grid_line(chart, chart->bg_src_pixmap, 3 * chart->h/4);

	/* There is no vertical grid image, so I make a vertical grid line using
	|  pixel values from the horizontal grid image at desired x positions.
	*/
	h = ((GdkWindowPrivate *) chart->bg_grid_pixmap)->height;
	grid_image = gdk_image_get(chart->bg_grid_pixmap, 0, 0,
					gkrellm_chart_width(), h);
	gc = gkrellm_draw_GC(3);
	dx = chart->w / HORIZONTAL_DIVS;
	for (x = dx; x < HORIZONTAL_DIVS * dx; x += dx)
		{
		color.pixel = gdk_image_get_pixel(grid_image, x, 0);
		gdk_gc_set_foreground(gc, &color);
		gdk_draw_line(chart->bg_src_pixmap, gc, x - 1, 0, x - 1, chart->h - 1);
		if (h > 1)
			{
			color.pixel = gdk_image_get_pixel(grid_image, x, 1);
			gdk_gc_set_foreground(gc, &color);
			gdk_draw_line(chart->bg_src_pixmap, gc, x, 0, x, chart->h - 1);
			}
		}
	gdk_image_destroy(grid_image);
	}

static void
draw_chart_to_screen(void)
	{
	if (sound->fd < 0)
		gkrellm_draw_chart_text(chart, DEFAULT_STYLE_ID,
			"\\cNo esd?\\n\\f\\cClick here to\\n\\f\\ctry to open");
	else if (extra_info)
		gkrellm_draw_decal_on_chart(chart, decal_usec, 2,
					chart->h - decal_usec->h);
	gkrellm_draw_chart_to_screen(chart);
	}

static void
reset_oscope(gboolean force)
	{
	gint	y;

	if (oscope->dirty || force || (GK.two_second_tick && sound->fd < 0))
		{
		y = chart->h / 2;
		gkrellm_clear_chart_pixmap(chart);
		oscope->gc = gkrellm_draw_GC(1);
		gdk_gc_set_foreground(oscope->gc, gkrellm_in_color());
		gdk_draw_line(chart->pixmap, oscope->gc, 0, y, chart->w - 1, y);
		draw_chart_to_screen();
		}
	oscope->dirty = FALSE;
	sound->buf_count = 0;
	sound->buf_index = 0;
	oscope->x_append = oscope->y_append = 0;
	}

static void
draw_oscope(void)
	{
	gkrellm_clear_chart_pixmap(chart);
	trigger_delay(CHANNEL_LR);
	oscope_trace(CHANNEL_LR);
	draw_chart_to_screen();
	oscope->dirty = TRUE;
	}

static gint
cb_extra(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	if (ev->button == 1 && ev->type == GDK_BUTTON_PRESS)
		{
		if (sound->fd < 0)
			{
			sound_open_stream(sound->host);
			oscope->dirty = TRUE;
			}
		else
			{
			extra_info = !extra_info;
			gkrellm_config_modified();
			reset_oscope(TRUE);
			}
		}
	else if (   ev->button == 3
			 || (ev->button == 1 && ev->type == GDK_2BUTTON_PRESS)
			)
		gkrellm_chartconfig_window_create(chart);
	return TRUE;
	}

static void
update_scope(void)
	{
	gint		d, left, right, left_peak, right_peak;

	if ((left = sound->left_value) > oscope->vert_max)
		left = oscope->vert_max;
	if ((right = sound->right_value) > oscope->vert_max)
		right = oscope->vert_max;
	left_peak = sound->left_peak_value - oscope->vert_max / 30;
	if ((d = sound->left_peak_value - left) > 0)
		left_peak -= d / 30;
	right_peak = sound->right_peak_value - oscope->vert_max / 30;
	if ((d = sound->right_peak_value - right) > 0)
		right_peak -= d / 30;

	if (left_peak < left)
		left_peak = left;
	if (right_peak < right)
		right_peak = right;

	gkrellm_update_krell(chart->panel, sound->krell_left, left);
	gkrellm_update_krell(chart->panel, sound->krell_left_peak, left_peak);
	gkrellm_update_krell(chart->panel, sound->krell_right, right);
	gkrellm_update_krell(chart->panel, sound->krell_right_peak, right_peak);

	sound->left_peak_value = left_peak;
	sound->right_peak_value = right_peak;
	sound->left_value = 0;
	sound->right_value = 0;

	if (sound->krell_sensitivity_y_target > sound->krell_sensitivity_y)
		{
		sound->krell_sensitivity_y += 1;
		gkrellm_move_krell_yoff(chart->panel, sound->krell_sensitivity,
					sound->krell_sensitivity_y);
		}
	else if (sound->krell_sensitivity_y_target < sound->krell_sensitivity_y)
		{
		sound->krell_sensitivity_y -= 1;
		gkrellm_move_krell_yoff(chart->panel, sound->krell_sensitivity,
					sound->krell_sensitivity_y);
		}

	gkrellm_draw_panel_layers(chart->panel);

	if (!sound->streaming)
		reset_oscope(FALSE);
	else if (!oscope->x_append && sound->buf_count)
		draw_oscope();
	sound->streaming = FALSE;
	}


static void
sound_vertical_scaling(void)
	{
	oscope->vert_max = (gint) (oscope->vert_sensitivity * 32767.0);

	gkrellm_set_krell_full_scale(sound->krell_left, oscope->vert_max, 1);
	gkrellm_set_krell_full_scale(sound->krell_right, oscope->vert_max, 1);
	gkrellm_set_krell_full_scale(sound->krell_left_peak, oscope->vert_max, 1);
	gkrellm_set_krell_full_scale(sound->krell_right_peak, oscope->vert_max, 1);
	}

static void
update_slider_position(Krell *k, gint x_ev)
	{
	gint	x,
			w = gkrellm_chart_width();

	if (x_ev < sound->vu_x0)
		x_ev = sound->vu_x0;
	if (x_ev >= w)
		x_ev = w - 1;
	sound->x_sensitivity_raw = x_ev;
	x = x_ev - sound->vu_x0;
	x = k->full_scale * x / (vu_meter_width - 1);
	if (x < 0)
		x = 0;

	oscope->vert_sensitivity = (100.0 - (gfloat) x) / 100.0;
	if (oscope->vert_sensitivity < 0.05)
		oscope->vert_sensitivity = 0.05;
	if (oscope->vert_sensitivity > 1.0)
		oscope->vert_sensitivity = 1.0;
	sound_vertical_scaling();
	gkrellm_config_modified();
	gkrellm_update_krell(chart->panel, k, (gulong) x);
	gkrellm_draw_panel_layers(chart->panel);
	}


static gint
expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GdkPixmap	*pixmap	= NULL;

	if (widget == chart->panel->drawing_area)
		pixmap = chart->panel->pixmap;
	else if (widget == chart->drawing_area)
		pixmap = chart->pixmap;
	if (pixmap)
		gdk_draw_pixmap(widget->window, gkrellm_draw_GC(1), pixmap,
				ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				ev->area.width, ev->area.height);
	return FALSE;
	}


static gint
cb_panel_enter(GtkWidget *widget, GdkEventButton *ev)
	{
	sound->krell_sensitivity_y_target = sound->krell_sensitivity_y_dn;
	return TRUE;
	}

static gint
cb_panel_leave(GtkWidget *widget, GdkEventButton *ev)
	{
	sound->krell_sensitivity_y_target = sound->krell_sensitivity_y_up;
	return TRUE;
	}

static gint
cb_panel_release(GtkWidget *widget, GdkEventButton *ev)
	{
	gint	dx = 0;
	gint	delta;

	delta = vu_meter_width / 30;
	if (delta == 0)
		delta = 1;
	if (ev->button == 1)
		krell_in_motion = NULL;
	else if (ev->button == 4)
		dx = delta;
	else if (ev->button == 5)
		dx = -delta;
	if (dx)
		update_slider_position(sound->krell_sensitivity,
					sound->x_sensitivity_raw + dx);
	return TRUE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
    {
	Krell	*k = sound->krell_sensitivity;

	if (ev->button == 3)
		{
		gkrellm_open_config_window(mon_scope);
		return TRUE;
		}
	if (   ev->button == 1 && ev->x > sound->vu_x0
		&& ev->y >= k->y0 && ev->y <= k->y0 + k->h_frame)
		{
		krell_in_motion = k;
		update_slider_position(krell_in_motion, ev->x);
		}
	return TRUE;
	}

static gint
cb_panel_motion(GtkWidget *widget, GdkEventButton *ev)
	{
	GdkModifierType	state;
	Krell			*k;

	if (krell_in_motion)
		{
		/* Check if button is still pressed, in case missed button_release
		*/
		state = ev->state;
		if (!(state & GDK_BUTTON1_MASK))
			{
			krell_in_motion = NULL;
			return TRUE;
			}
		k = krell_in_motion;
		update_slider_position(krell_in_motion, ev->x);
		}
	return TRUE;
	}


static void
height_changed(gpointer data)
	{
	draw_oscope_grid();
	reset_oscope(TRUE);
	}

static void
create_chart(GtkWidget *vbox, gint first_create)
	{
	Chart		*cp = chart;

	/* Since decal_usec will be drawn on the chart, remove it from the panel.
	*/
	gkrellm_destroy_decal(decal_usec);
	decal_usec = gkrellm_create_decal_text(chart->panel, "888 msec",
			gkrellm_chart_alt_textstyle(DEFAULT_STYLE_ID), NULL, 2, 0, 0);
	gkrellm_remove_decal(chart->panel, decal_usec);

	gkrellm_set_chart_height_default(cp, DEFAULT_CHART_HEIGHT);
	gkrellm_chart_create(vbox, mon_scope, cp, &chart_config);

	gkrellm_set_chartconfig_flags(chart_config, NO_CONFIG_FIXED_GRIDS);

	/* The only config is height and I don't call gkrellm draw chart routines,
	|  so set the chart draw function to just reset the chart.  It will only
	|  be called when the chart height is changed.
	*/
	gkrellm_set_draw_chart_function(cp, height_changed, NULL);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "expose_event",
				(GtkSignalFunc) expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_extra, cp);
		gtk_widget_show(vbox);
		}
	draw_oscope_grid();
	oscope_horizontal_scaling();
	reset_oscope(TRUE);
	}

static void
cb_sweep_pressed(DecalButton *button)
	{
	gint	dir;

	dir = GPOINTER_TO_INT(button->data);
	dir = (dir == 0) ? 1 : -1;
	oscope->usec_per_div = gkrellm_125_sequence(oscope->usec_per_div + dir,
				TRUE, LOW_SWEEP, HIGH_SWEEP, FALSE, FALSE);
	oscope_horizontal_scaling();
	reset_oscope(TRUE);
	}

static void
create_panel_buttons(Panel *p)
	{
	GdkImlibImage	*im	= NULL;
	Style			*style;
	gint			x, y;

	style = gkrellm_meter_style(style_id);

	y = 3;
	gkrellm_get_gkrellmrc_integer("sound_button_sweep_yoff", &y);

	gkrellm_load_image("buttom_sweep_dn", button_sweep_dn_xpm,
				&im, STYLE_NAME);
	gkrellm_render_to_pixmap(im, &button_sweep_dn_pixmap,
				&button_sweep_dn_mask, 0, 0);
	decal_sweep_dn = gkrellm_create_decal_pixmap(p, button_sweep_dn_pixmap,
				button_sweep_dn_mask, 2, style, -1, y);

	x = decal_sweep_dn->x;
	y = decal_sweep_dn->y;

	gkrellm_load_image("buttom_sweep_up", button_sweep_up_xpm,
				&im, STYLE_NAME);
	gkrellm_render_to_pixmap(im, &button_sweep_up_pixmap,
				&button_sweep_up_mask, 0, 0);
	decal_sweep_up = gkrellm_create_decal_pixmap(p, button_sweep_up_pixmap,
				button_sweep_up_mask, 2, style,
				x + decal_sweep_dn->w, y);


	gkrellm_decal_on_top_layer(decal_sweep_dn, TRUE);
	gkrellm_decal_on_top_layer(decal_sweep_up, TRUE);

	sound->vu_x0 = decal_sweep_up->x + decal_sweep_up->w;
	vu_meter_width = gkrellm_chart_width() - sound->vu_x0;
	}

static void
create_panel(GtkWidget *vbox, gint first_create)
	{
	GdkImlibImage	*im = NULL;
	Panel			*p;
	Style			*s;
	gint			x, yoff;

	p = chart->panel;
	create_panel_buttons(p);
	x = sound->vu_x0;

	s = gkrellm_copy_style(gkrellm_meter_style(style_id));

	s->krell_yoff = 3;
	s->krell_depth = 1;
	s->krell_x_hot = 59;
	s->krell_expand = KRELL_EXPAND_LEFT;
	gkrellm_get_gkrellmrc_integer("sound_krell_vu_depth", &s->krell_depth);
	gkrellm_get_gkrellmrc_integer("sound_krell_vu_x_hot", &s->krell_x_hot);
	gkrellm_get_gkrellmrc_integer("sound_krell_vu_expand", &s->krell_expand);
	gkrellm_get_gkrellmrc_integer("sound_krell_vu_left_yoff", &s->krell_yoff);
	gkrellm_load_image("krell_vu", krell_vu_xpm, &im, STYLE_NAME);
	sound->krell_left = gkrellm_create_krell(p, im, s);
	gkrellm_monotonic_krell_values(sound->krell_left, FALSE);
	gkrellm_set_krell_margins(p, sound->krell_left, x, 0);

	s->krell_yoff = 9;
	gkrellm_get_gkrellmrc_integer("sound_krell_vu_right_yoff", &s->krell_yoff);
	sound->krell_right = gkrellm_create_krell(p, im, s);
	gkrellm_monotonic_krell_values(sound->krell_right, FALSE);
	gkrellm_set_krell_margins(p, sound->krell_right, x, 0);

	s->krell_yoff = 2;
	s->krell_depth = 5;
	s->krell_x_hot = -1;
	s->krell_expand = KRELL_EXPAND_NONE;
	gkrellm_get_gkrellmrc_integer("sound_krell_vu_peak_depth",
				&s->krell_depth);
	gkrellm_get_gkrellmrc_integer("sound_krell_vu_peak_x_hot",
				&s->krell_x_hot);
	gkrellm_get_gkrellmrc_integer("sound_krell_vu_peak_left_yoff",
				&s->krell_yoff);
	gkrellm_load_image("krell_vu_peak", krell_vu_peak_xpm, &im, STYLE_NAME);
	sound->krell_left_peak = gkrellm_create_krell(p, im, s);
	gkrellm_monotonic_krell_values(sound->krell_left_peak, FALSE);
	gkrellm_set_krell_margins(p, sound->krell_left_peak, x, 0);

	s->krell_yoff = 8;
	gkrellm_get_gkrellmrc_integer("sound_krell_vu_peak_right_yoff",
				&s->krell_yoff);
	sound->krell_right_peak = gkrellm_create_krell(p, im, s);
	gkrellm_monotonic_krell_values(sound->krell_right_peak, FALSE);
	gkrellm_set_krell_margins(p, sound->krell_right_peak, x, 0);

	sound_vertical_scaling();

	sound->krell_sensitivity_y_up = -10;
	sound->krell_sensitivity_y_dn = 0;
	s->krell_yoff = 4;
	s->krell_depth = 1;
	s->krell_x_hot = -1;
	gkrellm_get_gkrellmrc_integer("sound_krell_sensitivity_depth",
				&s->krell_depth);
	gkrellm_get_gkrellmrc_integer("sound_krell_sensitivity_x_hot",
				&s->krell_x_hot);
	gkrellm_get_gkrellmrc_integer("sound_krell_sensitivity_y_up",
				&sound->krell_sensitivity_y_up);
	gkrellm_get_gkrellmrc_integer("sound_krell_sensitivity_y_dn",
				&sound->krell_sensitivity_y_dn);
	gkrellm_load_image("krell_sensitivity", krell_sensitivity_xpm,
            	&im, STYLE_NAME);
	/* A negative y_up won't work in the create, so save its value and
	|  do the real negative positioning later.
	*/
	yoff = s->krell_yoff = sound->krell_sensitivity_y_up;
	sound->krell_sensitivity_y_target = s->krell_yoff;
	sound->krell_sensitivity = gkrellm_create_krell(p, im, s);
	gkrellm_monotonic_krell_values(sound->krell_sensitivity, FALSE);
	gkrellm_set_krell_margins(p, sound->krell_sensitivity, x, 0);
	gkrellm_set_krell_full_scale(sound->krell_sensitivity, 100, 1);

	/* Unlike the style pointer passed to gkrellm_panel_configure(), the krells
	|  don't need the style to persist.
	*/
	g_free(s);
	if (im)
		gdk_imlib_kill_image(im);

	gkrellm_panel_configure(p, NULL, gkrellm_meter_style(style_id));
	gkrellm_panel_create(vbox, mon_scope, p);

	gkrellm_make_decal_button(p, decal_sweep_dn, cb_sweep_pressed,
				GINT_TO_POINTER(0), 1, 0);
	gkrellm_make_decal_button(p, decal_sweep_up, cb_sweep_pressed,
				GINT_TO_POINTER(1), 1, 0);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT (p->drawing_area),
				"expose_event", (GtkSignalFunc) expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_panel_press, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"button_release_event", (GtkSignalFunc) cb_panel_release,NULL);
		gtk_signal_connect(GTK_OBJECT (p->drawing_area),
				"motion_notify_event", (GtkSignalFunc) cb_panel_motion, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"enter_notify_event", (GtkSignalFunc) cb_panel_enter, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"leave_notify_event", (GtkSignalFunc) cb_panel_leave, NULL);
		}
	gkrellm_update_krell(chart->panel, sound->krell_sensitivity,
				(gulong) (100.0 * (1.0 - oscope->vert_sensitivity)));
	sound->x_sensitivity_raw = (gint) ((1.0 - oscope->vert_sensitivity)
				* (gfloat) vu_meter_width) + sound->vu_x0;

	/* To get the negative y_up, must move the krell since the create
	|  interprets negative offsets as special placement codes.  Also,
	|  this does the initial draw for me.
	*/
	gkrellm_move_krell_yoff(p, sound->krell_sensitivity, yoff);
	}

static void
create_scope(GtkWidget *vbox, gint first_create)
	{
	if (first_create)
		{
		chart = gkrellm_chart_new0();
		chart->panel = gkrellm_panel_new0();
		sound_open_stream(sound->host);
		}
	create_chart(vbox, first_create);
	create_panel(vbox, first_create);
	}


static void
save_config(FILE *f)
	{
	fprintf(f, "%s sensitivity %f\n", CONFIG_KEYWORD,oscope->vert_sensitivity);
	fprintf(f, "%s usec_per_div %d\n", CONFIG_KEYWORD, oscope->usec_per_div);
	fprintf(f, "%s extra_info %d\n", CONFIG_KEYWORD, extra_info);
	gkrellm_save_chartconfig(f, chart_config, CONFIG_KEYWORD, NULL);
	}

static void
load_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n != 2)
		return;
	if (!strcmp(config, "sensitivity"))
		{
		sscanf(item, "%f", &oscope->vert_sensitivity);
		if (oscope->vert_sensitivity < 0.05)
			oscope->vert_sensitivity = 0.05;
		if (oscope->vert_sensitivity > 1.0)
			oscope->vert_sensitivity = 1.0;
		}
	else if (!strcmp(config, "usec_per_div"))
		sscanf(item, "%d", &oscope->usec_per_div);
	else if (!strcmp(config, "extra_info"))
		sscanf(item, "%d", &extra_info);
	else if (!strcmp(config, GKRELLM_CHARTCONFIG_KEYWORD))
		gkrellm_load_chartconfig(&chart_config, item, 0);
	}

/* --------------------------------------------------------------------- */
static void
apply_config(void)
	{
	}


static gchar	*info_text[] =
{
N_("<b>GKrellMSS - GKrellM Sound Scope\n"),
"\n",
N_("<b>VU Meter\n"),
N_("Two fast response krells show left and right audio signal levels\n"
	"averaged over about 1/20 sec.  There are also two slow decay peak\n"
	"response krells.\n"),
"\n",

N_("<b>Oscope Chart\n"),
N_("Shows combined left and right audio signals as an oscilloscope trace.\n"),
"\n",

N_("<b>Oscope Sweep Speed Buttons\n"),
N_("The two buttons to the left of the VU Meter select an oscope horizontal\n"
	"sweep speed ranging from 100 microseconds (usec) per division to 50\n"
	"miliseconds (msec) per division.  There are 5 horizontal divisions,\n"
	"so a trace sweep time can range from 500 usec (1/2000 sec) to\n"
	"250 msec (1/4 sec).\n"),
"\n",
N_("<b>\nMouse Button Actions:\n"),
N_("<b>\tLeft "),
N_("click and drag in the VU Meter krell region to adjust VU Meter.\n"
	"\t\tand chart sensitivity.\n"),
N_("<b>\tWheel "),
N_("anywhere in panel also adjusts sensitivity.\n"),
N_("<b>\tRight "),
N_("on panel opens this configuration window.\n"),
N_("\t\tOn chart toggles display of chart labels.\n")

};

static void
create_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*tabs;
	GtkWidget	*vbox;
	GtkWidget	*text, *label;
	gchar		*buf;
	gint		i;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

#if 0
/* --Options Tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Options"));
#endif


/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(info_text[i]));

/* --About tab */
	vbox = gkrellm_create_framed_tab(tabs, _("About"));
	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

	buf = g_strdup_printf(_("GKrellMSS %d.%d\nGKrellM Sound Scope\n\n"
				"Copyright (c) 2001 by Bill Wilson\n"
				"bill@gkrellm.net\n"
				"http://gkrellm.net\n\n"
				"Released under the GNU Public License"),
				GKRELLMSS_VERSION_MAJOR, GKRELLMSS_VERSION_MINOR);
	label = gtk_label_new(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	g_free(buf);

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
	}


static Monitor	monitor_scope =
	{
	N_(CONFIG_NAME),	/* Name, for config tab.	*/
	0,					/* Id,  0 if a plugin		*/
	create_scope,		/* The create function		*/
	update_scope,		/* The update function		*/
	create_tab,			/* The config tab create function	*/
	apply_config,		/* Apply the config function		*/

	save_config,		/* Save user conifg			*/
	load_config,		/* Load user config			*/
	CONFIG_KEYWORD,		/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	MON_APM,			/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_plugin(void)
	{
	monitor_scope.name = _(monitor_scope.name);
	mon_scope = &monitor_scope;

	oscope = g_new0(Oscope, 1);
	oscope->usec_per_div = 2000;
	oscope->vert_sensitivity = 0.5;
	oscope->vert_max = (gint) (oscope->vert_sensitivity * 32767.0);

	sound = g_new0(SoundMonitor, 1);
	sound->x_sensitivity_raw = gkrellm_chart_width() / 2;

	style_id = gkrellm_add_meter_style(mon_scope, STYLE_NAME);

	return &monitor_scope;
	}
