/* GKrellM Volume plugin Version 0.8
|  Copyright (C) 1999-2000 Sjoerd Simons
|
|  Author:  Sjoerd Simons  sjoerd@luon.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
|
|  To get a copy of the GNU General Puplic License,  write to the
|  Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef __FreeBSD__
#include <machine/soundcard.h>
#else
#include <sys/soundcard.h>
#endif

#include "volume.h"
#include <signal.h>
#include <string.h>

#if ((VERSION_MAJOR>0)||(VERSION_MINOR>9))
  #define VOLUME_STYLE style_id
	static gint style_id;
#else
	#define VOLUME_STYLE DEFAULT_STYLE
#endif

static Sliders *sliderz;

static int
open_mixer(void) {
  if (sliderz->fd == -1) sliderz->fd = open("/dev/mixer",O_RDWR);
  if (sliderz->fd == -1) gkrellm_message_window("Gkrellm error",
        "Gkrellm Volume plugin couldn't open /dev/mixer",NULL);
return sliderz->fd == -1 ? 0 : 1;
}

static void
close_mixer(void) {
  if (sliderz->fd != -1)   close(sliderz->fd) ;
  sliderz->fd = -1;
}

static void
openclose_mixer(void) {
  if (sliderz->fd == -1) open_mixer();
  else close_mixer(); 
}

static void
set_volume(int x,int device) {
  gkrellm_config_modified();		// make sure the volumes are saved
  x += (x<<8);
  ioctl(sliderz->fd,MIXER_WRITE(device),&x);
}

static void 
mute (void) {
  Slider *slide;
  if(!(sliderz->toggles & IS_MUTED)) {
    for (slide = sliderz->slider ; slide ; slide = slide->next) 
       set_volume(0,slide->dev);
    sliderz->toggles |= IS_MUTED;
  } else { 
    for (slide = sliderz->slider ; slide ; slide = slide->next)
      set_volume(slide->setting,slide->dev);
    
    sliderz->toggles &= ~IS_MUTED;
  }

}

static void
do_toggle(void) {
 if (sliderz->toggles & MDCLOSE) {
   if (!(sliderz->toggles & MUTE)) openclose_mixer();
   else if(!(sliderz->toggles & IS_MUTED)) {
       mute();  close_mixer();
   }
   else { 
     if(!open_mixer()) return; /* make sure mixer is open */
     mute();
   }
 }
 else if (sliderz->toggles & MUTE) mute();
}

static void 
run_ext_mixer(void) {
	char *blerf;

	if (!sliderz->ext_mixer) return;
	blerf = malloc((strlen(sliderz->ext_mixer) + 4) * sizeof(char));
	strcpy(blerf,sliderz->ext_mixer);
	system(strcat(blerf," &"));
	free(blerf);
}

static gint
plug_expose_event(GtkWidget *widget, GdkEventExpose *event) {
    Slider *slide;

 for (slide = sliderz->slider ; slide ; slide = slide->next)
   if (widget == slide->panel->drawing_area) {
      gdk_draw_pixmap(widget->window,
        widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
        slide->panel->pixmap,
         event->area.x, event->area.y,
         event->area.x, event->area.y,
         event->area.width, event->area.height);
     }
  return FALSE;
}

static void
slider_motion(GtkWidget *widget, GdkEventMotion *ev, gpointer data)
	{
	Krell			*k;
	gint			x;
	Slider			*slide;
	GdkModifierType	state;
        if(sliderz->fd == -1 || sliderz->toggles & IS_MUTED) return ; 
	        /* don't do anything if mixer is closed or is muting is on*/
	if (sliderz->slider_in_motion)
		{
		/* Check if button is still pressed, 
		 | in case missed button_release.
		 */
		state = ev->state;
		if (!(state & GDK_BUTTON1_MASK))
			{
			sliderz->slider_in_motion = NULL;
			return;
			}
		k = sliderz->slider_in_motion;
                for (slide=sliderz->slider ; slide && slide->krell != k; slide = slide->next); 
                if (!slide) return;
                
		x = ev->x * k->full_scale / (gkrellm_chart_width() - 1);

		if (x < 0) x = 0;
		k->previous = 0;
		gkrellm_update_krell(slide->panel, k, (gulong) x);
		gkrellm_draw_layers(slide->panel);
                slide->setting = x;
                set_volume(x,slide->dev);
		}
	}

static void
panel_button_release(GtkWidget *widget, GdkEventButton *ev, Slider *slide) {
  gint x;
  if (ev->state & GDK_BUTTON1_MASK) sliderz->slider_in_motion = NULL;
  else if (ev->state & GDK_BUTTON2_MASK) do_toggle();
	else if (ev->state & GDK_BUTTON3_MASK) run_ext_mixer();
  else if (sliderz->fd == -1 || sliderz->toggles & IS_MUTED) return;
  /* no need to continue */
  else if (ev->state & GDK_BUTTON4_MASK || ev->state & GDK_BUTTON5_MASK)
     {
         x = slide->setting;
         x = ev->state & GDK_BUTTON4_MASK ? x + sliderz->wheeladjust 
              : x - sliderz->wheeladjust;
         /* Make sure we stay inside the limits... */
         if (x < 0) x = 0;
         if (x > 100) x = 100; 
         set_volume(x,slide->dev);
         slide->setting = x;
         slide->krell->previous = 0;
         gkrellm_update_krell(slide->panel,slide->krell, x);
         gkrellm_draw_layers(slide->panel);
     }
}

static void
panel_button_press(GtkWidget *widget, GdkEventButton *ev,Slider *slide) {
  gint x;

  if (ev->button != 1 || sliderz->toggles & IS_MUTED) return ;
  sliderz->slider_in_motion = slide->krell;

  x = ev->x * slide->krell->full_scale / (gkrellm_chart_width() - 1);
  
  if (x < 0) x = 0;
  slide->krell->previous = 0;
  gkrellm_update_krell(slide->panel, slide->krell, (gulong) x);
  gkrellm_draw_layers(slide->panel);
  slide->setting = x;
  set_volume(x,slide->dev);
}

static void
update_slider(void) {
  int x;
  Slider *slide;
  for (slide=sliderz->slider ; slide ; slide = slide->next) {
    if (sliderz->fd == -1 || sliderz->toggles & IS_MUTED) x = 0;
    else {
      ioctl(sliderz->fd,MIXER_READ(slide->dev),&x);
      x>>=8;
      slide->setting = x;
    }
    slide->krell->previous = 0;
    gkrellm_update_krell(slide->panel,slide->krell, x);
    gkrellm_draw_layers(slide->panel);
  }

}

static void
create_slider(int first_create,int device) {
  GdkImlibImage *bg_image,*krell_image;
  Style  	*panel_style,*slider_style;
  gint   	x,w;
  Slider 	*slide,*nslide;
  char 		*label, *text[] = SOUND_DEVICE_LABELS;

  nslide = NULL;  
  panel_style = gkrellm_meter_style(VOLUME_STYLE);
  x = panel_style->margin;
  w = gkrellm_chart_width() - panel_style->margin;
  bg_image = gkrellm_bg_meter_image(VOLUME_STYLE);

  for (slide = sliderz->slider ; slide && slide->dev != device; slide = slide->next) ;
    if (!slide) { 
      slide = g_new0(Slider,1); 
      slide->dev = device;
      if (!sliderz->slider) sliderz->slider = slide;
      else { 
        for (nslide = sliderz->slider; nslide->next ; nslide = nslide->next);
        nslide->next = slide;
      }
    }
  
  if (!first_create)
    gkrellm_destroy_krell_list(slide->panel);
  else 
    slide->panel = gkrellm_panel_new0();

  label = strdup( text[device] );
  slide->panel->textstyle = gkrellm_meter_textstyle(VOLUME_STYLE);
  gkrellm_configure_panel(slide->panel,label,panel_style);
  gkrellm_create_panel(sliderz->panelbox,slide->panel,bg_image);
  
  gkrellm_monitor_height_adjust(slide->panel->h);
   
  slider_style = gkrellm_krell_slider_style();
  krell_image = gkrellm_krell_slider_image();
  slide->krell = gkrellm_create_krell(slide->panel,krell_image,slider_style);
	slide->krell->y0 = (slide->panel->h - slide->krell->h_frame)/2;
  slide->krell->full_scale = 100;
    
  if (first_create) {
     gtk_signal_connect(GTK_OBJECT(slide->panel->drawing_area), "expose_event",
                (GtkSignalFunc) plug_expose_event,NULL);
     gtk_signal_connect(GTK_OBJECT(slide->panel->drawing_area), "button_press_event",
                (GtkSignalFunc) panel_button_press,slide);
     gtk_signal_connect(GTK_OBJECT(slide->panel->drawing_area), "button_release_event",
                (GtkSignalFunc) panel_button_release,slide);
     gtk_signal_connect(GTK_OBJECT(slide->panel->drawing_area), "motion_notify_event",
                (GtkSignalFunc) slider_motion,slide);
    
    }
  gkrellm_update_krell(slide->panel,slide->krell,device);
  gkrellm_draw_layers(slide->panel);
}


static void
create_sliderz(int first_create) {
   int i; 
   
   for (i = 0; i < SOUND_MIXER_NRDEVICES ; i ++) 
     if (sliderz->show_devices & 1 << i) {
        create_slider(first_create, i); 
     }
}


static void
create_plugin(GtkWidget *vbox, gint first_create)
	{
      if (sliderz->panelbox == NULL) {
         sliderz->panelbox = gtk_vbox_new(FALSE,0);
         gtk_container_add(GTK_CONTAINER(vbox),sliderz->panelbox);
         gtk_widget_show(sliderz->panelbox);
      }
      create_sliderz(first_create);
}


static void
toggled (GtkWidget *button,int device) {
   if (sliderz->config_devices & 1 << device) /* it was on */
     sliderz->config_devices = sliderz->config_devices & ~(1 << device);      
   else 
     sliderz->config_devices = sliderz->config_devices | 1 << device; 
}

static void 
toggle_toggled(GtkWidget *button, gint data) {
  if (sliderz->toggles & data) sliderz->toggles &= ~data;
  else sliderz->toggles |= data; 
}

static void 
signal_changed(GtkWidget *button,gint signal) {
  sliderz->config_signal = signal; 
}
static void
mixer_entry_changed(GtkEntry *entry,gpointer *data){
				sliderz->conf_ext_mixer = strdup(gtk_entry_get_text(entry));
}

static void
spin_button_changed (GtkSpinButton *spin_button,gpointer *data) {
  sliderz->config_wheeladjust = gtk_spin_button_get_value_as_int(spin_button);
}

static void 
create_help_text(GtkWidget *text) {
  gchar *info_text[] = 
    {
  	"<b>This plugin allows you to control the volume of the mixer devices \n",
    "<b>supported by your soundcard.\n",
    "<b>\nUser interface:\n",
    "*Left mouse button on a panel will change the volume",
    " of the panel's device\n",    
    "*Middle mouse button on a panel will do some configurable stuff\n",
		"*Right mouse button on a panel will run the external mixer command\n",
    "*Using your mousewheel on a panel will change the volume of the\n", 
    " panel's device with a configurable amount\n",
    "<b>\nConfigure stuff: \n",
    "<b>\n\tMixer error tab:\n",
    "\tThis tab only appears when the mixer device isn't opened by the plugin\n",
    "\tPress on that awfully big button, the mixer device will be opened and\n",
    "\t a Mixer devices tab will appear, if the mixer was succesfully opened\n",
    "<b>\n\tMixer devices tab:\n",
    "\tChoose which devices you want to control (all supported devices are \n"
    "\tshown)\n",
    "<b>\n\tOptions tab:\n",
    "\tsee the help text in the tab itself ;)\n",
    "\tA toggle signal can be usefull in scripts.\n",
		"<b>\n\tInfo tab:\n",
		"\tThe tab your currently looking at.\n",
		"<b>\n\tAbout tab:\n",
		"\tSome useless stuff that some people want to know.\n"};
    gkrellm_add_info_text(text,info_text,sizeof(info_text)/sizeof(gchar *));
}


static void 
create_mixer_devices_tab(GtkWidget *tabs) {
  GtkWidget *label;
  GtkWidget *frame; 
  GtkWidget  *vbox;
  GtkWidget *hbox;
  GtkWidget *button;
  int       i,n,devices;
  char      *labels[] = SOUND_DEVICE_LABELS;    
  
  sliderz->config_devices =  sliderz->show_devices;
  
  ioctl(sliderz->fd,SOUND_MIXER_READ_DEVMASK,&devices);
  
  frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame),3);
  gtk_widget_show(frame);
  
  label = gtk_label_new("Mixer devices");
  gtk_notebook_prepend_page(GTK_NOTEBOOK(tabs),frame,label);
  gtk_widget_show(label);
  
  vbox = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame),vbox);
  gtk_widget_show(vbox);
      
  hbox = NULL;
  n = 3; /* act like the last hbox is full, so a new one is created  */
  for (i = 0; i < SOUND_MIXER_NRDEVICES ; i++) 
     if (devices & 1 << i) { 
       if (n > 2) {  /* a hbox is full when it has 3 buttons in it...*/
         hbox = gtk_hbox_new(FALSE,0); 
         gtk_container_add(GTK_CONTAINER(vbox),hbox);
         gtk_widget_show(hbox);
         n = 0;
       }
       button = gtk_check_button_new_with_label(labels[i]);
       gtk_container_add(GTK_CONTAINER(hbox),button);
       gtk_widget_show(button);
       gtk_toggle_button_set_state((GtkToggleButton *) button,sliderz->show_devices & 1 << i);
       gtk_signal_connect(GTK_OBJECT(button),"toggled",
                             (GtkSignalFunc) toggled,GINT_TO_POINTER(i) ); 
       n++;
     }
}

static void
mixerbutton (GtkWidget *widget,GtkWidget *data) {
  if (!open_mixer()) return;
  gtk_notebook_remove_page(GTK_NOTEBOOK(data),0);
  create_mixer_devices_tab(data);
  gtk_notebook_set_page(GTK_NOTEBOOK(data),0);
}

static void 
create_plug_config(GtkWidget *tab) {
  GtkWidget *tabs;
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *button;
  GtkWidget *scrolled;
  GtkWidget *text;
  GtkWidget *radio_button;
  GtkWidget *spin_button;
  GtkWidget *seperator ;
	GtkWidget *ext_mixer_entry;
  GtkAdjustment *spin_adjust;
  GSList    *radio = NULL;
  gchar     *plugin_about_text = NULL;

  tabs = gtk_notebook_new();
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs),GTK_POS_TOP);
  gtk_box_pack_start(GTK_BOX(tab),tabs,TRUE,TRUE,0);  

  if (sliderz->toggles & MUTE) sliderz->toggles |= MUTE_CONFIG;
  if (sliderz->toggles & MDCLOSE) sliderz->toggles |=MDCLOSE_CONFIG;

  sliderz->config_wheeladjust = sliderz->wheeladjust;
  
/*-------------------devices tab-----------------*/
  if (sliderz->fd != -1) create_mixer_devices_tab(tabs);
  else {
    frame = gtk_frame_new(NULL);
    gtk_container_border_width(GTK_CONTAINER(frame),3);
    label = gtk_label_new("Mixer error");
    gtk_notebook_append_page(GTK_NOTEBOOK(tabs),frame,label);  
  
    vbox = gtk_vbox_new(FALSE,0);
    gtk_container_add(GTK_CONTAINER(frame),vbox);     
    button = gtk_button_new_with_label(
				"Mixer device is closed, press here to open");
    gtk_container_add(GTK_CONTAINER(vbox),button);
    gtk_signal_connect(GTK_OBJECT(button),"clicked",
    			(GtkSignalFunc) mixerbutton,tabs);

  }  
     
/* ----------- Options Menu -------------*/
  frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame),3);
  label = gtk_label_new("Options");
  gtk_notebook_append_page(GTK_NOTEBOOK(tabs),frame,label);  
  
  vbox = gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame),vbox);

/* >>>>>>>>>>>>> what to do on toggle selection */
  label = gtk_label_new("On a mmb click or the toggle signal :");
  gtk_container_add(GTK_CONTAINER(vbox),label);  
  
  button = gtk_check_button_new_with_label("Mute all mixer devices we control.");  
  gtk_container_add(GTK_CONTAINER(vbox),button);
  gtk_toggle_button_set_state((GtkToggleButton *) 
                                    button,sliderz->toggles & MUTE_CONFIG);  

  gtk_signal_connect(GTK_OBJECT(button),"toggled",
               (GtkSignalFunc) toggle_toggled,GINT_TO_POINTER(MUTE_CONFIG)); 
                             
                             
 button = gtk_check_button_new_with_label("Open/close mixer device.");  
  gtk_container_add(GTK_CONTAINER(vbox),button);
  gtk_toggle_button_set_state((GtkToggleButton *) 
                                      button, sliderz->toggles & MDCLOSE_CONFIG);  

  gtk_signal_connect(GTK_OBJECT(button),"toggled",
            (GtkSignalFunc) toggle_toggled,GINT_TO_POINTER(MDCLOSE_CONFIG));                              

/* >>>>>>>>> toggle signal selection */
  label = gtk_label_new("Toggle signal :");
  gtk_container_add(GTK_CONTAINER(vbox),label);  

  sliderz->config_signal = sliderz->toggle_signal;
	
	hbox = gtk_hbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(vbox),hbox);
  
	radio_button = gtk_radio_button_new_with_label(radio,"SIGUSR1");
  radio = gtk_radio_button_group(GTK_RADIO_BUTTON(radio_button));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_button)
                                 ,sliderz->config_signal == SIGUSR1);

  gtk_container_add(GTK_CONTAINER(hbox),radio_button); 
  gtk_signal_connect(GTK_OBJECT(radio_button),"pressed",
                      (GtkSignalFunc) signal_changed,GINT_TO_POINTER(SIGUSR1)); 

  radio_button = gtk_radio_button_new_with_label(radio,"SIGUSR2");
  radio = gtk_radio_button_group(GTK_RADIO_BUTTON(radio_button));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_button)
                                ,sliderz->config_signal == SIGUSR2);
  gtk_container_add(GTK_CONTAINER(hbox),radio_button); 
  gtk_signal_connect(GTK_OBJECT(radio_button),"pressed",
                      (GtkSignalFunc) signal_changed,GINT_TO_POINTER(SIGUSR2)); 

	hbox = gtk_hbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(vbox),hbox);

#ifdef SIGPWR 
  radio_button = gtk_radio_button_new_with_label(radio,"SIGPWR");
  radio = gtk_radio_button_group(GTK_RADIO_BUTTON(radio_button));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_button)
                                 ,sliderz->config_signal == SIGPWR);

  gtk_container_add(GTK_CONTAINER(hbox),radio_button); 
  gtk_signal_connect(GTK_OBJECT(radio_button),"pressed",
                      (GtkSignalFunc) signal_changed,GINT_TO_POINTER(SIGPWR)); 
#endif

  radio_button = gtk_radio_button_new_with_label(radio,"NONE");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_button)
                                ,sliderz->config_signal < 0);
  gtk_container_add(GTK_CONTAINER(hbox),radio_button); 
  gtk_signal_connect(GTK_OBJECT(radio_button),"pressed",
                      (GtkSignalFunc) signal_changed,GINT_TO_POINTER(-1));                       

/* >>>>> seperator before the other options */
seperator = gtk_hseparator_new();
gtk_container_add(GTK_CONTAINER(vbox),seperator);
/* toggle whether or not to save volumes on exit */
  button = gtk_check_button_new_with_label
                         ("Save volumes on exit and Restore them on startup.");  

  gtk_container_add(GTK_CONTAINER(vbox),button);
  gtk_toggle_button_set_state((GtkToggleButton *) 
                                    button,sliderz->toggles & SVOL_CONFIG);  

  gtk_signal_connect(GTK_OBJECT(button),"toggled",
               (GtkSignalFunc) toggle_toggled,GINT_TO_POINTER(SVOL_CONFIG)); 

 
/* >>>>>>>> wheel mouse adjust selection */

spin_adjust = (GtkAdjustment *) gtk_adjustment_new(0,0,100,1.0,0,0);
spin_button = gtk_spin_button_new(spin_adjust,1.0,0);

gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin_button)
                              ,sliderz->config_wheeladjust);
gtk_signal_connect(GTK_OBJECT(spin_button),"changed",
                    (GtkSignalFunc) spin_button_changed,NULL);
hbox = gtk_hbox_new(FALSE,0);

label = gtk_label_new("Mouse wheel adjust volume by (%) ");
gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);

gtk_box_pack_start(GTK_BOX(hbox),spin_button,FALSE,FALSE,0);
gtk_container_add(GTK_CONTAINER(vbox),hbox);

/* >>>>>>>> command to run */
if(sliderz->conf_ext_mixer) free(sliderz->conf_ext_mixer) ; 

sliderz->conf_ext_mixer = strdup(sliderz->ext_mixer);
hbox = gtk_hbox_new(FALSE,0);
label = gtk_label_new("External mixer command: ");
gtk_box_pack_start(GTK_BOX(hbox),label,FALSE,FALSE,0);

ext_mixer_entry = gtk_entry_new_with_max_length(255);
gtk_entry_set_text(GTK_ENTRY(ext_mixer_entry), sliderz->conf_ext_mixer);
gtk_entry_set_editable(GTK_ENTRY(ext_mixer_entry),TRUE);
gtk_box_pack_start(GTK_BOX(hbox),ext_mixer_entry,TRUE,TRUE,2);
gtk_signal_connect(GTK_OBJECT(ext_mixer_entry),"changed",
		(GtkSignalFunc) mixer_entry_changed,NULL);

gtk_container_add(GTK_CONTAINER(vbox),hbox);

/* ----------- help  text ---------------*/
  frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame),3);
  scrolled = gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
        GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
  
  gtk_container_add(GTK_CONTAINER(frame),scrolled);
  label = gtk_label_new("Info");
  gtk_notebook_append_page(GTK_NOTEBOOK(tabs),frame,label);

  text = gtk_text_new(NULL,NULL);
  gtk_text_set_editable(GTK_TEXT(text),FALSE);
  create_help_text(text);
  gtk_container_add(GTK_CONTAINER(scrolled),text);

/* ----------------- about text --------------------*/

plugin_about_text = g_strdup_printf(
   "Volumeplugin %d.%d\n" \
   "GKrellM volume Plugin\n\n" \
   "Copyright (C) 2000 Sjoerd Simons\n" \
   "sjoerd@luon.net\n" \
   "http://gkrellm.luon.net \n\n" \
   "Released under the GNU Public Licence",
   VOLUME_MAJOR_VERSION,VOLUME_MINOR_VERSION);

 text = gtk_label_new(plugin_about_text); 
 label = gtk_label_new("About");
 gtk_notebook_append_page(GTK_NOTEBOOK(tabs),text,label);
 g_free(plugin_about_text);
}

static void
save_plug_config(FILE *f) {
  int toggles = sliderz->toggles & ~(IS_MUTED | MUTE_CONFIG | MDCLOSE_CONFIG);
  Slider *slidr;
  fprintf(f,"%s DEV %d\n",CONFIG_KEYWORD,sliderz->show_devices) ;
  fprintf(f,"%s SIGNAL %d\n",CONFIG_KEYWORD,sliderz->toggle_signal);
  fprintf(f,"%s TOGGLES %d\n",CONFIG_KEYWORD,toggles);
  fprintf(f,"%s WHEEL %d\n",CONFIG_KEYWORD,sliderz->wheeladjust);
  fprintf(f,"%s EXTMIXER %s\n",CONFIG_KEYWORD,sliderz->ext_mixer);
  if (sliderz->toggles & SVOL) 
    for(slidr = sliderz->slider; slidr; slidr = slidr->next)
      fprintf(f, "%s %d %d\n", CONFIG_KEYWORD, slidr->dev, slidr->setting);
}

static void
load_plug_config(gchar *arg) {
  int n = 0;
  int x;
  gchar config[10],item[256];
  if(sscanf(arg,"%s %[^\n]",config,item) != 2 ) return ; 
  if(!strcmp("EXTMIXER",config)) {
	    sliderz->ext_mixer = strdup(item);
  }	
  /* all other config entry's use a numeric item */
  sscanf(item,"%d",&n);

  if(!strcmp("DEV",config)) {
    sliderz->show_devices = n;
    if (!n) return;
    open_mixer();
    ioctl(sliderz->fd,SOUND_MIXER_READ_DEVMASK,&x);
    /* making sure all the mixer devices we show, really exist */
    sliderz->show_devices = sliderz->show_devices & x; 
  }
  else if (config[0] >= '0' && config[0] <= '9') {	// if numeric
    /* mixer is already open, because DEV, is saved earlier then
       the volumes */
       set_volume(n,atoi(config));
  }
  else if (!strcmp("SIGNAL",config)) {
    sliderz->toggle_signal = n;
    if (n != -1) signal(n,(void *) do_toggle);
  }
  else if (!strcmp("TOGGLES",config)) sliderz->toggles = n;
  else if (!strcmp("WHEEL",config)) sliderz->wheeladjust = n;
}

static void
apply_plug_config(void) {
  Slider *slide, *prev;
  
  if (sliderz->show_devices != sliderz->config_devices) {
    sliderz->show_devices =  sliderz->config_devices ;
    prev = NULL;
    slide = sliderz->slider;
    sliderz->slider = NULL;
    for ( ; slide ; slide = slide->next ) {
      if (prev) free(prev);
      gkrellm_monitor_height_adjust(- slide->panel->h);
      gkrellm_destroy_krell_list(slide->panel);
      gkrellm_destroy_panel(slide->panel);
      prev = slide;
    }
    free(prev);
    create_sliderz(1);
  } 
  sliderz->wheeladjust =  sliderz->config_wheeladjust;
  
  sliderz->toggles &= ~(MDCLOSE | MUTE | SVOL); /* reset the settings */

  if (sliderz->toggles & MDCLOSE_CONFIG) sliderz->toggles |= MDCLOSE;

  if (sliderz->toggles & MUTE_CONFIG) sliderz->toggles |= MUTE;

  if (sliderz->toggles & SVOL_CONFIG) sliderz->toggles |= SVOL;

  if (!(sliderz->toggles & MUTE) && sliderz->toggles & IS_MUTED) mute(); 
      /* if you can't mute anymore, turn of the mutingstuff */

  if (!(sliderz->toggles & MDCLOSE)) open_mixer();
     /* if you can't open and close the mix-device, open the mixer */

  free(sliderz->ext_mixer);
	sliderz->ext_mixer = strdup(sliderz->conf_ext_mixer);

  if (sliderz->config_signal == sliderz->toggle_signal) return;
  if (sliderz->toggle_signal != -1) signal(sliderz->toggle_signal,SIG_DFL);
  if (sliderz->config_signal >= 0) {
    signal(sliderz->config_signal,(void *) do_toggle);
    sliderz->toggle_signal = sliderz->config_signal;
    }
  else sliderz->toggle_signal = -1;        
}

static Monitor	plugin_mon	=
	{
	"Volume Plugin",			/* Name, for config tab.    */
	0,					/* Id,  0 if a plugin       */
	create_plugin,		/* The create function      */
	update_slider,   		/* The update function      */
	create_plug_config,	/* The config tab create function   */
	apply_plug_config, 			 /* Apply the config function        */

	save_plug_config,	/* Save user config			*/
	load_plug_config,	/* Load user config			*/
	CONFIG_KEYWORD,		/* config keyword			*/

	NULL,			/* Undefined 2	*/
	NULL,			/* Undefined 1	*/
	NULL,			/* Undefined 0	*/

	LOCATION,

	NULL,				/* Handle if a plugin, filled in by GKrellM     */
	NULL				/* path if a plugin, filled in by GKrellM       */
	};


Monitor *
init_plugin(void)
{
  sliderz = g_new0(Sliders,1);
  sliderz->fd = -1; 
  sliderz->toggle_signal = -1;
  sliderz->show_devices = 0;
  sliderz->toggles = 0;
  sliderz->wheeladjust = 5;
  sliderz->ext_mixer = strdup("gmix");
  sliderz->conf_ext_mixer = NULL;

  #if ((VERSION_MAJOR>0)||(VERSION_MINOR>9))
    style_id = gkrellm_add_meter_style(&plugin_mon,"volume");
  #endif
  return &plugin_mon;
}
