/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <string.h>

#include "gimageview.h"
#include "gimv_image.h"
#include "thumbnail_support.h"
#include "fileutil.h"
#include "gfileutil.h"
#include "md5.h"
#include "prefs.h"

gboolean check_gimv (gpointer data);
gboolean check_nautilus (gpointer data);
gboolean check_konq_l (gpointer data);
gboolean check_konq_m (gpointer data);
gboolean check_konq_s (gpointer data);
gboolean check_gqview (gpointer data);
gboolean check_ee_l (gpointer data);
gboolean check_ee_m (gpointer data);
gboolean check_ee_s (gpointer data);
gboolean check_xv (gpointer data);
gboolean check_pic_l (gpointer data);
gboolean check_pic_m (gpointer data);


const gchar *cache_write_items[] = {
   N_("None"),
   N_("GImageView"),
   N_("Nautilus"),
   N_("Konqueror (Large)"),
   N_("Konqueror (Medium)"),
   N_("Konqueror (Small)"),
   N_("GQview"),
   N_("Electric Eyes (Preview)"),
   N_("Electric Eyes (Icon)"),
   N_("Electric Eyes (Mini)"),
   N_("XV thumbnail"),
   NULL
};


ThumbTypePrefs thumb_prefs [] = {
   {"None",                     0,   0, FixNULL,   NULL,                NULL,    NULL},
   {"GImageView",              96,  96, FixMax,    "~/.gimv/thumbnail", ".png",  check_gimv},
   {"Nautilus",               105, 105, FixMax,    ".thumbnails",       ".png",  check_nautilus},
   {"Konqueror (Large)",       90,  90, FixMax,    ".pics/large",       NULL,    check_konq_l},
   {"Konqueror (Medium)",      64,  64, FixMax,    ".pics/med",         NULL,    check_konq_m},
   {"Konqueror (Small)",       48,  48, FixMax,    ".pics/small",       NULL,    check_konq_s},
 
   /* CAUTION: This is the default setting */
   {"GQview",                   48,  48, FixMax, "~/.gqview/thumbnails", ".png", check_gqview},

   {"Electric Eyes (Preview)", 180, 180, FixMax,    "~/.ee/previews",    NULL,   check_ee_l},
   {"Electric Eyes (Icon)",    256,  64, FixHeight, "~/.ee/icons",       NULL,   check_ee_m},
   {"Electric Eyes (Mini)",    256,  24, FixHeight, "~/.ee/mini",        NULL,   check_ee_s},
   {"XV thumbnail",              0,   0, FixNULL,   ".xvpics",           NULL,   check_xv},
   {"Picview (Preview)",       256,  64, FixHeight, "~/.ee/previews",    NULL,   check_pic_l},
   {"Picview (Icon)",          256,  24, FixHeight, "~/.ee/icons",       NULL,   check_pic_m},
};
const gint thumb_prefs_num = sizeof (thumb_prefs) / sizeof (thumb_prefs[0]);


/* FIXME!! */
GQViewThumbSize gqview_thumb_size [] = {
   {24, 24},
   {32, 32},
   {48, 48},
   {64, 64},
   //   {85, 64},
   {85, 85},
   {100, 100},
   {125, 125},
   {150, 150},
   {175, 175},
   {200, 200},
   {256, 256},
   {0, 0},
};
/* END FIXME!! */


/* load check funcs */
gboolean
check_gimv (gpointer data)
{
   return conf.cache_gimv;
}

gboolean
check_nautilus (gpointer data)
{
   return conf.cache_nautilus;
}

gboolean
check_konq_l (gpointer data)
{
   return conf.cache_konqueror_l;
}

gboolean
check_konq_m (gpointer data)
{
   return conf.cache_konqueror_m;
}

gboolean
check_konq_s (gpointer data)
{
   return conf.cache_konqueror_s;
}

gboolean
check_gqview (gpointer data)
{
   return conf.cache_gqview;
}

gboolean
check_ee_l (gpointer data)
{
   return conf.cache_ee_l;
}

gboolean
check_ee_m (gpointer data)
{
   return conf.cache_ee_m;
}

gboolean
check_ee_s (gpointer data)
{
   return conf.cache_ee_s;
}

gboolean
check_xv (gpointer data)
{
   return conf.cache_xvpics;
}

gboolean
check_pic_l (gpointer data)
{
   return conf.cache_ee_l && conf.cache_picview;
}

gboolean
check_pic_m (gpointer data)
{
   return conf.cache_ee_m && conf.cache_picview;
}


/*
 *  thumbsupport_load_thumbnail_cache:
 *     @ load thumbnail cache to GimvImage struct. 
 *
 *  filename : File path of image.
 *  type     : Thuymbnail cache type.
 *  Return   : GimvImage struct.
 */
GimvImage *
thumbsupport_load_thumbnail_cache (const gchar *filename, ThumbCacheType type)
{
   GimvImage *image;
   gchar *thumb_file;

   if (!filename || !type)
      return NULL;

   thumb_file = thumbsupport_get_thumb_cache_path (filename, type);

   if (!thumb_file)
      return NULL;

   if (type == XVPIC)
      image = gimv_image_get_xv_thumbnail (thumb_file);
   else
      image = gimv_image_load_file (thumb_file);

   g_free (thumb_file);

   return image;
}


/*
 *  thumbsupport_save_thumbnail_cache:
 *     @ Save thumbnail cache to disk. 
 *
 *  filename : File path of image.
 *  image    : Pointer to GimvImage of original image.
 *             If this value is NULL, create from original image.
 *  type     : Thuymbnail cache type.
 *  Return   : TRUE if success.
 */
gboolean
thumbsupport_save_thumbnail_cache (const gchar *filename, GimvImage *image,
				   ThumbCacheType type)
{
   GimvImage *imcache;
   gboolean retval = FALSE;

   gchar *thumb_file;
   gint thumb_width, thumb_height;
   ThumbSizeType fix_type;

   gfloat x_scale, y_scale;
   gint width, height;

   thumb_file = thumbsupport_get_thumb_cache_path (filename, type);
   if (!thumb_file)
      return FALSE;

   thumbsupport_get_thumb_cache_size (type, &thumb_width, &thumb_height,
				      &fix_type);

   /* not cache because too small */
   if (thumb_width == 0 || thumb_height == 0
       || (gimv_image_width (image) <= thumb_width
	   && gimv_image_height (image) <= thumb_width))
   {
      retval = FALSE;
      goto FUNC_END;
   }

   /* create cache directory if not found */
   if (!mkdirs (thumb_file)) {
      retval = FALSE;
      goto FUNC_END;
   }

   /* set thumbnail size for cache */
   x_scale = (gfloat) thumb_width  / (gfloat) gimv_image_width (image);
   y_scale = (gfloat) thumb_height / (gfloat) gimv_image_height (image);

   if (fix_type == FixHeight || x_scale > y_scale) {
      width = gimv_image_width (image) * y_scale;
      height = thumb_height;
      if (fix_type == FixHeight && width > thumb_width) {
	 width = thumb_width;
	 height = (gfloat) gimv_image_height (image)
	    / (gfloat) gimv_image_width (image) * (gfloat) width;
      }
   } else {
      width = thumb_width;
      height = gimv_image_height (image) * x_scale;
      if (fix_type == FixWidth && height > thumb_height) {
	 height = thumb_height;
	 width = (gfloat) gimv_image_width (image)
	    / (gfloat) gimv_image_height (image) * (gfloat) height;
      }
   }

   /* cache to disk */
   if (image) 
      imcache = gimv_image_scale_and_dump (image, width, height);
   else
      imcache = gimv_image_load_file (thumb_file);

   if (imcache) {
      g_print ("creating cache: %s\n", thumb_file);
      /* Electric Eyes & Picview use PNM format */
      if (type >= EE_L && type <= PICVIEW_M)
	 gimv_image_save_file_as_ppm (imcache, thumb_file);
      else   /* other cache type use PNG */
	 gimv_image_save_file_as_png (imcache, thumb_file);

      gimv_image_kill (imcache);

      retval = TRUE;
   } else 
      retval = FALSE;

 FUNC_END:
   g_free (thumb_file);
   return retval;
}


/*
 *  thumbsupport_get_thumb_cache_path:
 *     @ Return thumbnail path for specified image file and thumbnail type.
 *
 *  filename : Image file name.
 *  type     : Thumbnail cache type.
 *  Return   : Thumbnail path.
 */
gchar *
thumbsupport_get_thumb_cache_path (const gchar *filename, ThumbCacheType type)
{
   gchar *abspath, *image_name, *image_dir, *dir;
   gchar cwd[MAX_PATH_LEN], buf[MAX_PATH_LEN], *ret = NULL;
   gint len;

   if (!type)
      return NULL;

   abspath = fileutil_conv_rel_path_to_abs (filename);

   /* get filename */
   image_name = g_basename (abspath);
   if (!image_name) return NULL;

   /* get dir name */
   dir = image_dir = g_dirname (abspath);
   if (!image_dir) return NULL;

   if (type == PICVIEW_L || type == PICVIEW_M) {
      /* effort to use picview thumbnails */
      if (getcwd (cwd, MAX_PATH_LEN)
	  && !strncmp (cwd, image_dir, strlen (cwd))) {
	 image_dir = image_dir + strlen(cwd) + 1;
      }
   }

   //buf = g_new0 (gchar, MAX_PATH_LEN);

   /* if create chahe for NEW_KONQ */
   if (conf.cache_kde_2_2 && !strncmp (thumb_prefs[type].path, ".pics",5)) {
      unsigned char md5buf[16], tndir[MAX_PATH_LEN + 5], kdetn[MAX_PATH_LEN];
      gchar *tmpstr;
      size_t cnt;       

      tmpstr = link2abs (image_dir);

      snprintf(tndir, MAX_PATH_LEN+5, "file:%s/", tmpstr);
      md5_buffer(tndir, strlen(tndir + 1), md5buf);
      for(cnt = 0; cnt < 16; ++cnt) {
	 snprintf(tndir + (cnt * 2), MAX_PATH_LEN+5,"%02x", md5buf[cnt]);
      }
      strncpy(kdetn,tndir,(size_t) 4);
      strcpy(kdetn + 4, "/");
      strncpy(kdetn + 5,tndir+4, (size_t) 4);
      strcpy(kdetn + 9, "/");
      strcpy(kdetn + 10, tndir+8);
      
      g_snprintf(buf,MAX_PATH_LEN,"%s/.kde/share/thumbnails/%s/%s/%s",
		 getenv("HOME"), kdetn, thumb_prefs[type].path + 6, image_name);

      g_free (tmpstr);

   /* if create cache under home directory */
   } else if (!strncmp (thumb_prefs[type].path, "~/", 2)) {
      if (image_dir)
	 g_snprintf (buf, MAX_PATH_LEN, "%s/%s%s/%s",
		     getenv("HOME"), thumb_prefs[type].path + 2,
		     image_dir, image_name);
      else
	 g_snprintf (buf, MAX_PATH_LEN, "%s/%s/%s",
		     getenv("HOME"), thumb_prefs[type].path + 2, image_name);
   /* if create cache under image directory */
   } else {
      g_snprintf (buf, MAX_PATH_LEN, "%s/%s/%s",
		  image_dir, thumb_prefs[type].path, image_name);
   }

   len = strlen (buf) + 1;
   if (thumb_prefs[type].suffix) {
      len += strlen (thumb_prefs[type].suffix);
      if (len <= MAX_PATH_LEN)
	 strcat (buf, thumb_prefs[type].suffix);
   }

   if (len > 1) {
      ret = g_new0 (gchar, len);
      g_snprintf (ret, len, "%s", buf);
   }

   g_free (abspath);
   g_free (dir);

   return ret;
}


/*
 *  thumbsupport_get_thumb_cache_size:
 *     @ Return thumbnail size for various thumnail type.
 *
 *  type       : Thumbnail Cache type.
 *  width_ret  : Return thumbnail width.
 *  height_ret : Return thumbnail height.
 *  type_ret   : Which axis to fix.
 *  Return     : Thumbnail size [pixel]
 *               @ Zero means not allow (or not support) for writing cache
 */
gint
thumbsupport_get_thumb_cache_size (ThumbCacheType type,
				   gint *width_ret, gint *height_ret,
				   ThumbSizeType *type_ret)
{
   gint retval = 0;

   *width_ret = *height_ret = 0;
   *width_ret  = retval = thumb_prefs[type].width;
   *height_ret = thumb_prefs[type].height;
   *type_ret   = thumb_prefs[type].type;

   if (type == GIMAGEVIEW && conf.cache_gimv
       && conf.cache_gimv_write_size >= CACHE_GIMV_MIN_SIZE)
   {
      *height_ret = conf.cache_gimv_write_size;
      *width_ret  = conf.cache_gimv_write_size;
   }

   if (type == GQVIEW && conf.cache_gqview) {
      retval = *width_ret = gqview_thumb_size [conf.cache_gqview_write_size].width;
      *height_ret = gqview_thumb_size [conf.cache_gqview_write_size].height;
   }

   return retval;
}
