/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 *  These codes are mostly taken from Enfle.
 *  Enfle Copyright (C) 1998 Hiroshi Takekawa
 *
 *  See also http://www.jisyo.com/viewer/faq/mag_tech.htm
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "mag.h"

#define LINE200  0x01
#define COLOR8   0x02
#define DIGITAL  0x04
#define COLOR256 0xf0

#ifndef BUF_SIZE
#define BUF_SIZE 4096
#endif

#define BitSet(byte, bit)  (((byte) & (bit)) == (bit))
#define ReadOK(file,buffer,len)  (fread(buffer, len, 1, file) != 0)
#define ToL(buffer, off)  (buffer[(off)] | buffer[(off)+1]<<8 | buffer[(off)+2]<<16 | buffer[(off)+3]<<24)
#define ToS(buffer, off)  (buffer[(off)] | buffer[(off)+1]<<8);

static guchar *mag_decode_image (FILE     *ar,
				 mag_info *minfo,
				 guchar    colormap[256][3]);


gboolean
mag_get_header(const gchar *filename,
	       mag_info *info)
{
   return TRUE;
}


guchar *
mag_load(const gchar *filename,
	 gint *widthp,
	 gint *heightp)
{
   FILE *fd;
   mag_info minfo;
   guchar buffer[32], comment[BUF_SIZE], colormap[256][3], *dest;
   gint i;

   g_return_val_if_fail (filename && widthp && heightp, NULL);

   fd = fopen (filename, "rb");
   if (!fd) return NULL;

   if (!ReadOK(fd, buffer, 8)
       || !(!strncmp (buffer, "MAKI02  ", 8))) /* check data: 8byte */
   {
      fclose(fd);
      return NULL;
   }

   if (!ReadOK (fd, buffer, 4 + 18 + 2)) {     /* machine code:  4byte
					          user name:    18byte
					          unused:        2byte */
      fclose(fd);
      return NULL;
   }

   i = 0;
   do {                                        /* comment: unlimited */
      if (!ReadOK(fd, buffer, 1));
      comment[i++] = buffer[0];
   } while (buffer[0] != '\0' && i < 4096);    /* start of header */

   minfo.offset = ftell (fd) - 1;

   if (!ReadOK(fd, buffer, 31))                /* header: 32byte (include end of comment) */
   {
      fclose(fd);
      return NULL;
   }

   minfo.mcode  = buffer[0];
   minfo.dcode  = buffer[1];
   minfo.screen = buffer[2];
   minfo.lx     = ToS (buffer, 3);
   minfo.uy     = ToS (buffer, 5);
   minfo.rx     = ToS (buffer, 7);
   minfo.dy     = ToS (buffer, 9);
   minfo.off_a  = ToL (buffer, 11);
   minfo.off_b  = ToL (buffer, 15);
   minfo.size_b = ToL (buffer, 19);
   minfo.off_p  = ToL (buffer, 23);
   minfo.size_p = ToL (buffer, 27);

   minfo.ncolors = (minfo.screen & COLOR256) ? 256 : 16;

   minfo.width  = minfo.rx - minfo.lx + 1;
   minfo.height = minfo.dy - minfo.uy + 1;
   minfo.lpad   = minfo.lx & 7;
   minfo.rpad   = 7 - (minfo.rx & 7);
   minfo.lx    -= minfo.lpad;
   minfo.rx    += minfo.rpad;


   /* read palette */
   for (i = 0; i < minfo.ncolors; i++) {
      /* ORDER: GRB */
      fread (buffer, 3, 1, fd);
      colormap[i][1] = buffer[0]; /* G */
      colormap[i][0] = buffer[1]; /* R */
      colormap[i][2] = buffer[2]; /* B */
   }

   minfo.flagperline = minfo.width >> ((minfo.screen & COLOR256) ? 2 : 3);

   dest = mag_decode_image (fd, &minfo, colormap);

   fclose(fd);

   *widthp  = minfo.width;
   *heightp = minfo.height;

   return dest;
}


static guchar *
mag_decode_image (FILE *ar, mag_info *minfo, guchar colormap[256][3])
{
   gint i, j, k, pr, f, t, sa, width, height;
   guchar *flaga, *flagb, *fa, *fb, *pix, *d, *dest, *src;
   guchar bitmask[] = {
      128, 64, 32, 16, 8, 4, 2, 1
   };
   gint offset[][2] = {
      { 0,  0},    {-1,  0},    {-2,  0},    {-4,  0},
      { 0, -1},    {-1, -1},    { 0, -2},    {-1, -2},
      {-2, -2},    { 0, -4},    {-1, -4},    {-2, -4},
      { 0, -8},    {-1, -8},    {-2, -8},    { 0, -16}
   };

   g_return_val_if_fail (minfo, NULL);

   width  = minfo->width;
   height = minfo->height;

   /* memory allocate for flag */
   sa = minfo->off_b - minfo->off_a;
   fa = flaga = g_malloc(sa);
   if (!fa)
      return NULL;
   fb = flagb = g_malloc(sa << 3);
   if (!fb) {
      free(flaga);
      return NULL;
   }

   /* decode flag */
   fseek(ar, minfo->offset + minfo->off_a, 0);
   fread(flaga, sa, 1, ar);
   fseek(ar, minfo->offset + minfo->off_b, 0);
   for (i = 0; i < sa; i++) {
      f = *fa++;
      for (j = 0; j < 8; j++) {
	 guchar c;
	 if (f & bitmask[j]) {
	    fread(&c, 1, 1, ar);
	    *fb++ = c;
	 } else {
	    *fb++ = 0;
	 }
      }
   }
   free(flaga);

   /* do xor */
   fb = flagb + minfo->flagperline;
   for (i = 0; i < minfo->flagperline * (height - 1); i++, fb++)
      *fb ^= *(fb - minfo->flagperline);

   /* memory allocate for data  */
   dest = d = g_malloc0 (sizeof(guchar) * width * height * 3);
   if (!d) {
      free(flagb);
      return NULL;
   }
			 
   /* read pixel */
   fseek(ar, minfo->offset + minfo->off_p, SEEK_SET);
   pix = g_malloc0 (minfo->size_p);
   if (!pix) {
      free(flagb);
      return NULL;
   }
   if (!ReadOK (ar, pix, minfo->size_p)) {
      free(flagb);
      fprintf(stderr, "Premature MAG file\n");
      return NULL;
   }

   /* put pixels */
   pr = 0;
   fb = flagb;
   for (i = 0; i < minfo->flagperline * height; i++) {
      f = *fb++;
      j = f >> 4;
      if (j) {   /* copy from specified pixel */
	 if (minfo->screen & COLOR256) {
	    src = d + (offset[j][0] * 2 + offset[j][1] * width) * 3;
	    for (k = 0; k < 6; k++)
	       *d++ = *src++;
	 } else {
	    src = d + (offset[j][0] * 4 + offset[j][1] * width) * 3;
	    for (k = 0; k < 12; k++)
	       *d++ = *src++;
	 }
      } else {  /* put pixel directly */
	 if (minfo->screen & COLOR256) {
	    for (k = 0; k < 2; k++) {
	       t = pix[pr++];
	       *d++ = colormap[t][0];
	       *d++ = colormap[t][1];
	       *d++ = colormap[t][2];
	    }
	 } else {
	    for (k = 0; k < 2; k++) {
	       t = pix[pr++];
	       *d++ = colormap[t >> 4][0];
	       *d++ = colormap[t >> 4][1];
	       *d++ = colormap[t >> 4][2];
	       *d++ = colormap[t & 15][0];
	       *d++ = colormap[t & 15][1];
	       *d++ = colormap[t & 15][2];
	    }
	 }
      }
      j = f & 15;
      if (j) {
	 if (minfo->screen & COLOR256) {
	    src = d + (offset[j][0] * 2 + offset[j][1] * width) * 3;
	    for (k = 0; k < 6; k++)
	       *d++ = *src++;
	 } else {
	    src = d + (offset[j][0] * 4 + offset[j][1] * width) * 3;
	    for (k = 0; k < 12; k++)
	       *d++ = *src++;
	 }
      } else {
	 if (minfo->screen & COLOR256) {
	    for (k = 0; k < 2; k++) {
	       t = pix[pr++];
	       *d++ = colormap[t][0];
	       *d++ = colormap[t][1];
	       *d++ = colormap[t][2];
	    }
	 } else {
	    for (k = 0; k < 2; k++) {
	       t = pix[pr++];
	       *d++ = colormap[t >> 4][0];
	       *d++ = colormap[t >> 4][1];
	       *d++ = colormap[t >> 4][2];
	       *d++ = colormap[t & 15][0];
	       *d++ = colormap[t & 15][1];
	       *d++ = colormap[t & 15][2];
	    }
	 }
      }
   }
   free(flagb);
   free(pix);

   return dest;
}
