/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <utime.h>

#include "fileutil.h"
#include "gfileutil.h"
#include "gimv_image.h" /* FIXME!! file filter for get_dir () must become more generalize */
#include "gtkutils.h"
#include "intl.h"

#ifndef BUF_SIZE
#define BUF_SIZE 4096
#endif

#ifndef MAX_PATH_LEN
#define MAX_PATH_LEN 1024
#endif


gboolean getting_dir = FALSE;
gboolean stop_getting_dir = FALSE;


/*
 *  get_dir:
 *     @ Get image files in specified directory.
 *
 *  dirname  : Directory to scan.
 *  files    : Pointer to OpenFiles struct for store directory list.
 */
void
get_dir (const gchar *dirname, GetDirFlags flags,
	 GList **filelist_ret, GList **dirlist_ret)
{
   DIR *dp;
   struct dirent *entry;
   gchar buf[MAX_PATH_LEN], *path;
   GList *filelist = NULL, *dirlist = NULL;

   getting_dir = TRUE;

   if (flags & GETDIR_DISP_STDERR)
      fprintf (stderr, _("scandir = %s\n"), dirname);
   else if (flags & GETDIR_DISP_STDOUT)
      fprintf (stdout, _("scandir = %s\n"), dirname);

   if ((dp = opendir (dirname))) {
      while ((entry = readdir (dp))) {
	 if (flags & GETDIR_ENABLE_CANCEL) {
	    while (gtk_events_pending()) gtk_main_iteration();
	    if (stop_getting_dir) break;
	 }

	 /* ignore dot file */
	 if (!(flags & GETDIR_READ_DOT) && entry->d_name[0] == '.')
	    continue;

	 /* get full path */
	 if (dirname [strlen (dirname) - 1] == '/')
	    g_snprintf (buf, MAX_PATH_LEN, "%s%s", dirname, entry->d_name);
	 else
	    g_snprintf (buf, MAX_PATH_LEN, "%s/%s", dirname, entry->d_name);

	 /* if path is file */
	 if (!isdir (buf) || (!(flags & GETDIR_FOLLOW_SYMLINK) && islink (buf))) {
	    if (!filelist_ret) continue;

	    if (!(flags & GETDIR_DETECT_EXT)
		|| gimv_image_detect_type_by_ext (buf))
	    {
	       path = g_strdup (buf);

	       if (flags & GETDIR_DISP_STDERR)
		  fprintf (stderr, _("filename = %s\n"), path);
	       else if (flags & GETDIR_DISP_STDOUT)
		  fprintf (stdout, _("filename = %s\n"), path);

	       filelist = g_list_append (filelist, path);
	    }

	 /* if path is dir */
	 } else if (isdir(buf)) {
	    if (dirlist_ret && strcmp(entry->d_name, ".")
		            && strcmp(entry->d_name, ".."))
	    {
	       path = g_strdup (buf);

	       if (flags & GETDIR_DISP_STDERR)
		  fprintf (stderr, _("dirname = %s\n"), path);
	       else if (flags & GETDIR_DISP_STDOUT)
		  fprintf (stdout, _("dirname = %s\n"), path);

	       dirlist = g_list_append (dirlist, path);
	    }
	 }
      }
      closedir (dp);
      if (filelist)
	 filelist = g_list_sort (filelist, gtkutil_comp_spel);
      if (dirlist)
	 dirlist = g_list_sort (dirlist, gtkutil_comp_spel);
   }
   if (filelist_ret)
      *filelist_ret = filelist;
   if (dirlist_ret)
      *dirlist_ret = dirlist;

   getting_dir = FALSE;
   stop_getting_dir = FALSE;
}


void
get_dir_stop ()
{
   if (getting_dir)
      stop_getting_dir = TRUE;
}


/*
 *  get_dir_all:
 *     @
 *
 *  dirname :
 *  Return  :
 */
GList *
get_dir_all (const gchar *dirname)
{
   GList *filelist, *dirlist, *sub_dirlist, *node;
   gchar *sub_dirname;

   get_dir (dirname, GETDIR_READ_DOT, &filelist, &dirlist);

   if (dirlist) {
      node = dirlist;
      while (node) {
	 sub_dirname = node->data;
	 sub_dirlist = get_dir_all (sub_dirname);
	 if (sub_dirlist)
	    filelist = g_list_concat (sub_dirlist, filelist);
	 node = g_list_next (node);
      }
      filelist = g_list_concat (filelist, dirlist);      
   }
   return filelist;
}


/*
 *  get_dir_all_file:
 *     @
 *
 *  dirname :
 *  Return  :
 */
GList *
get_dir_all_file (const gchar *dirname)
{
   GList *filelist, *dirlist, *sub_dirlist, *node;
   gchar *sub_dirname;

   get_dir (dirname, GETDIR_READ_DOT, &filelist, &dirlist);

   if (dirlist) {
      node = dirlist;
      while (node) {
	 sub_dirname = node->data;
	 sub_dirlist = get_dir_all_file (sub_dirname);
	 if (sub_dirlist)
	    filelist = g_list_concat (sub_dirlist, filelist);
	 node = g_list_next (node);
      }
   }
   return filelist;
}


/*
 *  move_file:
 *     @
 *
 *  from_path  :
 *  dir        :
 *  action     :
 *  show_error :
 *  Return     : TRUE if success to move file.
 */
gboolean
move_file (const gchar *from_path, const gchar *dir,
	   ConfirmType *action, gboolean show_error)
{
   gchar *from_dir, *to_path, error_message[BUF_SIZE];
   struct stat from_st, to_st, todir_st;
   gboolean move_file = FALSE, move_faild = FALSE, copy_success = FALSE;
   gint exist;
   struct utimbuf ut;

   g_return_val_if_fail (action, FALSE);

   /********************
    * check source file
    ********************/
   if (lstat (from_path, &from_st)) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Can't find source file :\n%s"),
		     from_path);
	 gtkutil_message_dialog ("Error!!", error_message);
      }
      return FALSE;
   }

   /*****************
    * check dest dir
    *****************/
   if (!iswritable (dir)) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Can't move file : %s\n"
		       "Permission denied: %s\n"),
		     from_path, dir);
	 gtkutil_message_dialog ("Error!!", error_message);
      }
      return FALSE;
   }

   /*******************
    * check source dir
    *******************/
   from_dir = g_dirname (from_path);
   if (!iswritable (from_dir)) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Can't move file : %s\n"
		       "Permission denied: %s\n"),
		     from_path, from_dir);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      g_free (from_dir);
      return FALSE;
   }
   g_free (from_dir);

   /* set dest path */
   to_path = g_strconcat(dir, g_basename(from_path), NULL);

   /************************
    * checke for over write
    ************************/
   exist = !lstat(to_path, &to_st);
   if (exist && (!strcmp (from_path, to_path)
		 || from_st.st_ino == to_st.st_ino))
   {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Same file :\n%s"), to_path);
	 gtkutil_message_dialog ("Error!!", error_message);
      }
      return FALSE;
   }  else if (exist && *action == CONFIRM_ASK) {
      if (isdir (from_path)) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("File exist : %s"), to_path);
	 gtkutil_message_dialog (_("ERROR!!"), error_message);
      } else {
	 g_snprintf (error_message, BUF_SIZE,
		     _("File exist : %s\n\n"
		       "Overwrite?"),
		     to_path);
	 *action = gtkutil_confirm_dialog (_("File exist!!"), error_message, TRUE);
      }
   }

   switch (*action) {
   case CONFIRM_YES:
      move_file = TRUE;
      break;
   case CONFIRM_YES_TO_ALL:
      move_file = TRUE;
      break;
   case CONFIRM_NO:
      move_file = FALSE;
      break;
   case CONFIRM_CANCEL:
      move_file = FALSE;
      break;
   default:
      if (!exist)
	 move_file = TRUE;
      break;
   }

   if (!move_file) {
      g_free (to_path);
      return FALSE;
   }

   /**************
    * move file!!
    **************/
   stat (dir, &todir_st);
   if (from_st.st_dev != todir_st.st_dev) {
      copy_success = copy_file_to_file (from_path, to_path, action, show_error);
      if (copy_success) {
	 /* reset new file's time info */
	 ut.actime = from_st.st_atime;
	 ut.modtime = from_st.st_mtime;
	 utime(to_path, &ut);

	 /* remove old file */
	 if (remove (from_path) < 0) {   /* faild to remove file */
	    if (show_error) {
	       g_snprintf (error_message, BUF_SIZE,
			   _("Faild to remove file :\n"
			     "%s"),
			   from_path);
	       gtkutil_message_dialog (_("Error!!"), error_message);
	    }
	    g_free (to_path);
	    return FALSE;
	 }
      } else {
	 move_faild = TRUE;
      }
   } else {
      move_faild = rename(from_path, to_path);
   }

   /************************
    * if faild to move file
    ************************/
   if (move_faild) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Faild to move file :\n"
		       "From : %s\n"
		       "To : %s"),
		     from_path, to_path);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      g_free (to_path);
      return FALSE;
   }

   g_free (to_path);
   return TRUE;
}


gchar *
link2abs (const gchar *path)
{
   gchar *retval = NULL, **dirs, buf[MAX_PATH_LEN], *tmpstr;
   gint i, num;

   g_return_val_if_fail (path, NULL);

   if (path[0] != '/') return NULL;

   dirs = g_strsplit (path, "/", -1);
   if (!dirs) return NULL;

   for (i = 0; dirs[i]; i++) {
      if (!dirs[i]) continue;

      if (!retval) {
	 retval = g_strconcat ("/", dirs[i], NULL);
      } else {
	 tmpstr = g_strconcat (retval, "/", dirs[i], NULL);
	 g_free (retval);
	 retval = tmpstr;
      }

      num = readlink (retval, buf, MAX_PATH_LEN);
      if (num > MAX_PATH_LEN) {
	 g_free (retval);
	 return NULL;
      } else if (num > 0) {
	 buf[num] = '\0';
	 g_free (retval);
	 retval = g_strdup (buf);
      }
   }

   g_strfreev (dirs);

   return retval;
}


gboolean
copy_dir_check_source (const gchar *from_dir, gboolean show_error)
{
   gchar error_message[BUF_SIZE];

   if (islink (from_dir)) {   /* check path is link or not */
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("%s is link!!.\n"),
		     from_dir);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      return FALSE;
   }

   if (!file_exists (from_dir)) {   /* check path exists or not */
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Can't find source file :\n%s"),
		     from_dir);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      return FALSE;
   }

   if (!isdir (from_dir)) {   /* check path is directory or not */
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("%s is not directory!!.\n"),
		     from_dir);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      return FALSE;
   }

   return TRUE;
}


gboolean
copy_dir_check_dest (const gchar *from_path, const gchar *dirname,
		     const gchar *to_dir, gboolean show_error)
{
   gchar error_message[BUF_SIZE];

   if (!iswritable (dirname)) {   /* check permission */
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Can't copy directory : %s\n"
		       "Permission denied: %s\n"),
		     from_path, dirname);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      return FALSE;
   }

   if (file_exists (to_dir)) {   /* check dest path */
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("File exists!! : %s\n"),
		     to_dir);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      return FALSE;
   }

   return TRUE;
}


/*
 *  copy_dir:
 *     @
 *
 *  from_path  :
 *  dir        :
 *  action     :
 *  show_error :
 *  Return     : TRUE if success to copy directory.
 */
gboolean
copy_dir (const gchar *from_path, const gchar *dir,
	  ConfirmType *action, gboolean show_error)
{
   GtkWidget *progress_win;
   gchar message[BUF_SIZE];
   GList *filelist, *node;
   gchar *from_dir, *to_dir, *to_path, *dirname;
   ConfirmType confirm;
   gboolean result, cancel = FALSE;
   gfloat progress;
   gint pos, length;

   from_dir = g_strdup (from_path);
   if (from_dir[strlen (from_dir) - 1] == '/')
      from_dir[strlen (from_dir) - 1] = '\0';

   dirname = g_strdup (dir);
   if (dirname[strlen (dirname) - 1] == '/')
      dirname[strlen (dirname) - 1] = '\0';

   to_dir = g_strconcat (dirname, "/", g_basename (from_dir), NULL);

   /*******************
    * check source dir
    *******************/
   result = copy_dir_check_source (from_dir, show_error);
   if (!result) goto ERROR;

   /*****************
    * check dest dir
    *****************/
   result = copy_dir_check_dest (from_path, dirname, to_dir, show_error);
   g_free (dirname);
   if (!result) goto ERROR;


   /****************
    * do copy files
    ****************/
   filelist = node = get_dir_all_file (from_path);
   confirm = CONFIRM_YES_TO_ALL;

   progress_win = gtkutil_create_progress_window (_("Copy directory"), "...",
						  &cancel, 300, -1);
   gtk_grab_add (progress_win);
   length = g_list_length (filelist);

   while (node) {
      gint len;
      gchar *filename = node->data;
      gchar *tmpstr;

      while (gtk_events_pending()) gtk_main_iteration();

      pos = g_list_position (filelist, node);
      progress = (gfloat) pos / (gfloat) length;

      g_snprintf (message, BUF_SIZE, _("Copying %s ..."), filename);
      gtkutil_progress_window_update (progress_win, _("Copying directory"),
				      message, NULL, progress);

      len = strlen (from_path);

      if (strlen (filename) > len) {
	 tmpstr = filename + len;
	 if (tmpstr[0] == '/') tmpstr++;
	 to_path = g_strconcat (to_dir, "/", tmpstr, NULL);

	 /* realy do copy :-) */
	 mkdirs (to_path);
	 copy_file_to_file (filename, to_path, &confirm, show_error);

	 g_free (to_path);
      }

      node = g_list_next (node);
   }

   gtk_grab_remove (progress_win);
   gtk_widget_destroy (progress_win);

   g_list_foreach (filelist, (GFunc) g_free, NULL);
   g_list_free (filelist);

   g_free (from_dir);
   g_free (to_dir);

   return TRUE;

 ERROR:
   g_free (from_dir);
   g_free (to_dir);
   return FALSE;
}


/*
 *  copy_file_to_file:
 *     @
 *
 *  from_path  :
 *  to_path    :
 *  action     :
 *  show_error :
 *  Return     : TRUE if success to copy file.
 */
gboolean
copy_file_to_file (const gchar *from_path, const gchar *to_path,
		   ConfirmType *action, gboolean show_error)
{
   gint b;
   char buf[BUFSIZ];
   gchar error_message[BUF_SIZE];
   FILE *from, *to;
   struct stat from_st, to_st;
   gboolean copy_file = FALSE;
   gint exist;

   g_return_val_if_fail (action, FALSE);

   /******************** 
    * check source file
    ********************/
   if (isdir (from_path)) {
      return copy_dir (from_path, to_path, action, show_error);
   }

   if (lstat (from_path, &from_st)) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Can't find source file :\n%s"),
		     from_path);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      return FALSE;
   }

   /******************
    * check dest file
    ******************/
   exist = !lstat(to_path, &to_st);
   if (exist && (!strcmp (from_path, to_path)
		 || from_st.st_ino == to_st.st_ino))
   {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Same file :\n%s"), to_path);
	 gtkutil_message_dialog ("Error!!", error_message);
      }
      return FALSE;
   } else if (exist && *action == CONFIRM_ASK) {
      g_snprintf (error_message, BUF_SIZE,
		  _("File exist : %s\n\n"
		    "Overwrite?"),
		  to_path);
      *action = gtkutil_confirm_dialog (_("File exist!!"), error_message, TRUE);
   }

   switch (*action) {
   case CONFIRM_YES:
      copy_file = TRUE;
      break;
   case CONFIRM_YES_TO_ALL:
      copy_file = TRUE;
      break;
   case CONFIRM_NO:
      copy_file = FALSE;
      break;
   case CONFIRM_CANCEL:
      copy_file = FALSE;
      break;
   default:
      if (!exist)
	 copy_file = TRUE;
      break;
   }

   /**********
    * do copy
    **********/
   if (copy_file) {
      from = fopen (from_path, "rb");
      if (!from) {
	 if (show_error) {
	    g_snprintf (error_message, BUF_SIZE,
			_("Can't open file for read :\n%s"), to_path);
	    gtkutil_message_dialog (_("Error!!"), error_message);
	 }
	 return FALSE;
      }

      to = fopen (to_path, "wb");
      if (!to) {
	 fclose (from);
	 if (show_error) {
	    g_snprintf (error_message, BUF_SIZE,
			_("Can't open file for write :\n%s"), to_path);
	    gtkutil_message_dialog (_("Error!!"), error_message);
	 }
	 return FALSE;
      }

      while ((b = fread (buf, sizeof (char), BUFSIZ, from)) > 0) {
	 fwrite (buf, sizeof (char), b, to);
	 if (ferror (to)) {
	    fclose (from);
	    fclose (to);

	    if (show_error) {
	       g_snprintf (error_message, BUF_SIZE,
			   _("An error occured while copying file :\n%s"), to_path);
	       gtkutil_message_dialog (_("Error!!"), error_message);
	    }
	    return FALSE;
	 }
      }

      fclose (from);
      fclose (to);
   }

   return TRUE;
}


/*
 *  copy_file:
 *     @
 *
 *  from_path  :
 *  dir        :
 *  action     :
 *  show_error :
 *  Return     : TRUE if success to copy file.
 */
gboolean
copy_file (const gchar *from_path, const gchar *dir,
	   ConfirmType *action, gboolean show_error)
{
   gchar *to_path;
   gboolean retval;
   gchar error_message[BUF_SIZE];

   g_return_val_if_fail (action, FALSE);

   /* check source file is directory or not */
   if (isdir (from_path)) {
      return copy_dir (from_path, dir, action, show_error);
   }

   /*****************
    * check dest dir
    *****************/
   if (!iswritable (dir)) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Can't copy file : %s\n"
		       "Permission denied: %s\n"),
		     from_path, dir);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      return FALSE;
   }

   to_path = g_strconcat (dir, g_basename (from_path), NULL);
   retval = copy_file_to_file (from_path, to_path, action, show_error);
   g_free (to_path);

   return retval;
}


/*
 *  link_file:
 *     @
 *
 *  from_path  :
 *  dir        :
 *  action     :
 *  show_error :
 *  Return     : TRUE if success to link file.
 */
gboolean
link_file (const gchar *from_path, const gchar *dir,
	   gboolean show_error)
{
   gchar *to_path;
   struct stat from_st, to_st;
   gboolean link_faild;
   gchar error_message[BUF_SIZE];

   /********************
    * check source file
    ********************/
   if (lstat (from_path, &from_st)) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Can't find source file :\n%s"),
		     from_path);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      return FALSE;
   }

   /*****************
    * check dest dir
    *****************/
   if (!iswritable (dir)) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Can't create link : %s\n"
		       "Permission denied: %s\n"),
		     from_path, dir);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      return FALSE;
   }

   to_path = g_strconcat(dir, g_basename(from_path), NULL);

   /******************
    * check dest path
    ******************/
   if (!lstat (to_path, &to_st)) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("File exist : %s"),
		     to_path);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      g_free (to_path);
      return FALSE;
   }

   link_faild = symlink(from_path, to_path);

   if (link_faild) {
      if (show_error) {
	 g_snprintf (error_message, BUF_SIZE,
		     _("Faild to create link :\n"
		       "From : %s\n"
		       "To : %s"),
		     from_path, to_path);
	 gtkutil_message_dialog (_("Error!!"), error_message);
      }
      g_free(to_path);
      return FALSE;
   }

   g_free(to_path);
   return TRUE;;
}
