/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * These codes are mostly taken from G-thumB.
 * G-thumB code Copyright (C) 1999-2000 MIZUNO Takehiko <mizuno@bigfoot.com>
 */

#include <stdio.h>
#include <string.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gimageview.h"
#include "dirview.h"
#include "dnd.h"
#include "fileload.h"
#include "fileutil.h"
#include "gfileutil.h"
#include "gtkutils.h"
#include "icon.h"
#include "menu.h"
#include "prefs.h"
#include "thumbnail_window.h"


/* for auto-scroll and auto-expand at drag */
#define SCROLL_EDGE 20
typedef gboolean (*desirable_fn) (DirView *dv, gint x, gint y);
typedef gboolean (*scroll_fn)    (gpointer data);


typedef struct DirNode_Tag
{
   gboolean scanned;
   gchar *path;
} DirNode;


/* private functions */
static void     dirview_create_ctree      (DirView *dv);
static void     adjust_ctree              (GtkWidget      *widget,
					   GtkCTreeNode   *node);
static gboolean check_for_subdir          (DirView        *dv,
					   const gchar    *path);
GtkCTreeNode   *dirview_insert_dummy_node (DirView        *dv,
					   GtkCTreeNode   *parent);
GtkCTreeNode   *dirview_insert_node       (DirView        *dv,
					   GtkCTreeNode   *parent,
					   const gchar    *path);
void            wind_up_children          (DirView        *dv,
					   GtkCTreeNode   *parent);
static void     expand_dir                (DirView        *dv,
					   GtkCTreeNode   *parent);
static void     refresh_dir_tree          (DirView        *dv,
					   GtkCTreeNode   *parent);


/* for drag motion */
static void     dirview_cancel_drag_scroll     (DirView *dv);
static void     dirview_cancel_drag_scroll     (DirView *dv);
static gboolean dirview_scrolling_is_desirable (DirView *dv,
						gint     x,
						gint     y);
static gboolean timeout_dirview_scroll         (gpointer data);
static gint     timeout_auto_expand_directory  (gpointer data);


/* callback functions */
static void     cb_dirview_destroy    (GtkWidget      *widget,
				       DirView        *dv);
static void     cb_node_destroy       (gpointer        data);
static void     cb_expand             (GtkWidget      *widget,
				       GtkCTreeNode   *parent,
				       DirView        *dv);
static void     cb_collapse           (GtkWidget      *widget,
				       GtkCTreeNode   *parent,
				       DirView        *dv);
static void     cb_button_press       (GtkWidget      *widget,
				       GdkEventButton *event,
				       DirView        *dv);

/* callback functions for popup menu */
static void     cb_open_thumbnail     (DirView        *dv,
				       gboolean        scan_subdir,
				       GtkWidget      *menuitem);
static void     cb_go_to_here         (DirView        *dv,
				       guint           action,
				       GtkWidget      *menuitem);
static void     cb_refresh_dir_tree   (DirView        *dv,
				       guint           action,
				       GtkWidget      *menuitem);
static void     cb_mkdir              (DirView        *dv,
				       guint           action,
				       GtkWidget      *menuitem);
static void     cb_rename_dir         (DirView        *dv,
				       guint           action,
				       GtkWidget      *menuitem);
static void     cb_delete_dir         (DirView        *dv,
				       guint           action,
				       GtkWidget      *menuitem);

/* callback functions for toolbar buttons */
static void     cb_home_button        (GtkWidget      *widget,
				       DirView        *dv);
static void     cb_up_button          (GtkWidget      *widget,
				       DirView        *dv);
static void     cb_refresh_button     (GtkWidget      *widget,
				       DirView        *dv);
static void     cb_dotfile_button     (GtkWidget      *widget,
				       DirView        *dv);

/* callback functions for DnD */
static void     cb_drag_motion        (GtkWidget      *dirtree,
				       GdkDragContext *context,
				       gint            x,
				       gint            y,
				       gint            time,
				       gpointer        data);
static void     cb_drag_leave         (GtkWidget      *dirtree,
				       GdkDragContext *context,
				       guint           time,
				       gpointer        data);
static void     cb_drag_data_get      (GtkWidget *widget,
				       GdkDragContext *context,
				       GtkSelectionData *seldata,
				       guint info,
				       guint time,
				       gpointer data);
static void     cb_drag_data_received (GtkWidget      *dirtree,
				       GdkDragContext *context,
				       gint            x,
				       gint            y,
				       GtkSelectionData *seldata,
				       guint           info,
				       guint32         time,
				       gpointer        data);
static void     cb_drag_end           (GtkWidget        *dirtree,
				       GdkDragContext   *context,
				       gpointer          data);


static GtkItemFactoryEntry dirview_popup_items [] =
{
   {N_("/Load Thumbnail"),             NULL, cb_open_thumbnail,   FALSE, NULL},
   {N_("/Load Thumbnail recursively"), NULL, cb_open_thumbnail,   TRUE,  NULL},
   {N_("/---"),                        NULL, NULL,                0,     "<Separator>"},
   {N_("/Go to here"),                 NULL, cb_go_to_here,       0,     NULL},
   {N_("/Refresh Tree"),               NULL, cb_refresh_dir_tree, 0,     NULL},
   {N_("/---"),                        NULL, NULL,                0,     "<Separator>"},
   {N_("/Make Directory"),             NULL, cb_mkdir,            0,     NULL},
   {N_("/Rename Directory"),           NULL, cb_rename_dir,       0,     NULL},
   {N_("/Delete Directory"),           NULL, cb_delete_dir,       0,     NULL},
   {NULL, NULL, NULL, 0, NULL},
};


extern GtkItemFactoryEntry dnd_file_popup_items [];


/******************************************************************************
 *
 *   Private functions
 *
 ******************************************************************************/
/*
 *  dirview_create_ctree:
 *     @ create a new GtkCTree widget.
 *
 *  dv : Pointer to the DirView struct.
 */
static void
dirview_create_ctree (DirView *dv)
{
   GtkWidget *dirtree;
   GtkCTreeNode *root_node;
   GtkCTreeNode *node;
   DirNode *dirnode;
   Icon *folder, *ofolder;
   gchar *node_text = "dummy";
   gchar *tmpstr;

   g_return_if_fail (dv);

   if (dv->root_dir [strlen (dv->root_dir) - 1] != '/') {
      tmpstr = g_strconcat (dv->root_dir, "/", NULL);
      g_free (dv->root_dir);
      dv->root_dir = tmpstr;
   }

   dirtree = dv->dirtree = gtk_ctree_new (1,0);
   gtk_clist_set_column_auto_resize (GTK_CLIST (dirtree), 0, TRUE);
   gtk_clist_set_selection_mode (GTK_CLIST (dirtree), GTK_SELECTION_BROWSE);
   gtk_ctree_set_line_style (GTK_CTREE (dirtree), conf.dirview_line_style);
   gtk_ctree_set_expander_style (GTK_CTREE (dirtree), conf.dirview_expander_style);
   gtk_clist_set_row_height (GTK_CLIST (dirtree), 18);

   gtk_container_add (GTK_CONTAINER (dv->scroll_win), dirtree);

   gtk_signal_connect (GTK_OBJECT (dirtree), "tree_expand",
		       GTK_SIGNAL_FUNC (cb_expand), dv);
   gtk_signal_connect (GTK_OBJECT (dirtree), "tree_collapse",
		       GTK_SIGNAL_FUNC (cb_collapse), dv);

   gtk_signal_connect_after (GTK_OBJECT (dirtree),"button_press_event",
		       GTK_SIGNAL_FUNC (cb_button_press), dv);

   /* for drag file list */
   dnd_src_set  (dirtree, dnd_types, dnd_types_num);
   dnd_dest_set (dirtree, dnd_types, dnd_types_num);
   gtk_signal_connect (GTK_OBJECT (dirtree), "drag_motion",
		       GTK_SIGNAL_FUNC (cb_drag_motion), dv);
   gtk_signal_connect (GTK_OBJECT (dirtree), "drag_leave",
		       GTK_SIGNAL_FUNC (cb_drag_leave), dv);
   gtk_signal_connect (GTK_OBJECT (dirtree), "drag_data_get",
		       GTK_SIGNAL_FUNC (cb_drag_data_get), dv);
   gtk_signal_connect (GTK_OBJECT (dirtree), "drag_data_received",
		       GTK_SIGNAL_FUNC (cb_drag_data_received), dv);
   gtk_signal_connect (GTK_OBJECT (dirtree), "drag_end",
		       GTK_SIGNAL_FUNC (cb_drag_end), dv);

   folder = icon_get ("folder");
   ofolder = icon_get ("folder-open");
   root_node = gtk_ctree_insert_node (GTK_CTREE (dirtree), NULL, NULL, &dv->root_dir, 4,
				      folder->pixmap, folder->mask,
				      ofolder->pixmap, ofolder->mask,
				      FALSE, FALSE);

   dirnode = g_new0(DirNode, 1);
   dirnode->path = g_strdup(dv->root_dir);
   gtk_ctree_node_set_row_data_full (GTK_CTREE(dirtree), root_node,
				     dirnode, cb_node_destroy);
   node = gtk_ctree_insert_node (GTK_CTREE(dirtree), root_node, NULL, &node_text, 4,
				 NULL, NULL, NULL, NULL, TRUE, TRUE);
   gtk_ctree_expand (GTK_CTREE (dirtree), root_node);

   gtk_widget_show (dirtree);
}


/*
 *  adjust_ctree:
 *     @ set position of viewport
 *
 *  dirtree : ctree widget.
 *  parent  : ctree node that contain direcotry to expand.
 */
static void
adjust_ctree (GtkWidget *dirtree, GtkCTreeNode *node)
{
   gint row = GTK_CLIST (dirtree)->rows - g_list_length ((GList*)node);

   GTK_CLIST (dirtree)->focus_row = row;
   gtk_ctree_select (GTK_CTREE (dirtree), node);

   gtk_clist_freeze (GTK_CLIST (dirtree));
   if (gtk_ctree_node_is_visible (GTK_CTREE (dirtree), node) != GTK_VISIBILITY_FULL)
      gtk_ctree_node_moveto (GTK_CTREE (dirtree), node, 0, 0, 0);
   gtk_clist_thaw (GTK_CLIST (dirtree));
}


/*
 *  check_for_subdir:
 *     @ check whether sub directory is existing or not.
 *
 *  path :   The directory path name for check.
 *  Return : TRUE if sub direcotry is existing.
 */
static gboolean
check_for_subdir (DirView *dv, const gchar *path)
{
   DIR *dir;
   struct dirent *dirent;
   struct stat statbuf;
   gboolean exist;
   gchar *npath;

   g_return_val_if_fail (dv, FALSE);

   dir = opendir (path);
   if (!dir) return FALSE;

   while ((dirent = readdir(dir))) {
      if (!(!dv->show_dotfile && dirent->d_name[0] == '.')
	  && strcmp (dirent->d_name, ".") && strcmp (dirent->d_name, ".."))
      {
	 npath = g_strconcat (path, dirent->d_name, "/", NULL);
	 exist = !stat(npath, &statbuf);
	 g_free(npath);
	 if (!exist) continue;
	 if (S_ISDIR(statbuf.st_mode)) {
	    closedir(dir);
	    return TRUE;
	 }
      }
   }

   closedir(dir);

   return FALSE;
}


/*
 *  dirview_insert_dummy_node:
 *     @ insert dummy node to display expander.
 *
 *  dv     : Pointer to the Dir View struct.
 *  parnt  : Parent node to insert dummy node.
 *  Return : Pointer to the new GtkCTreeNode.
 */
GtkCTreeNode *
dirview_insert_dummy_node (DirView *dv, GtkCTreeNode *parent)
{
   GtkCTreeNode *node = NULL;
   gchar *dummy = "dummy", *node_text;
   DirNode *dirdata;
   gboolean is_leaf, expanded;

   dirdata = gtk_ctree_node_get_row_data (GTK_CTREE (dv->dirtree), parent);
   gtk_ctree_get_node_info (GTK_CTREE (dv->dirtree), parent, &node_text, NULL,
			    NULL, NULL, NULL, NULL, &is_leaf, &expanded);

   if (!expanded && !dirdata->scanned
       && !GTK_CTREE_ROW (parent)->children
       && strcmp (node_text, ".") && strcmp (node_text, "..")
       && check_for_subdir (dv, dirdata->path))
   {
      node = gtk_ctree_insert_node (GTK_CTREE (dv->dirtree), parent, NULL, &dummy,
				    4, NULL, NULL, NULL, NULL, FALSE, FALSE);
   }

   return node;
}


/*
 *  dirview_insert_node:
 *     @ insert new node to the dirview.
 *
 *  dv     : Pointer to the Dir View struct.
 *  parnt  : Parent node to insert new node.
 *  path   : Dir name to insert.
 *  Return : Pointer to the new GtkCTreeNode.
 */
GtkCTreeNode *
dirview_insert_node (DirView *dv, GtkCTreeNode *parent, const gchar *path)
{
   GtkCTreeNode *node = NULL;
   DirNode *dirnode;
   Icon *folder, *ofolder;
   gchar *text, *end;
   gchar *filename;

   g_return_val_if_fail (dv && parent && path, NULL);

   filename = g_basename (path);

   if (isdir (path)
       && (!(!dv->show_dotfile && (filename[0] == '.'))
	   || (conf.dirview_show_current_dir && !strcmp (filename, "."))
	   || (conf.dirview_show_parent_dir  && !strcmp (filename, ".."))))
   {
      dirnode = g_new0 (DirNode, 1);
      if (!strcmp(filename, ".")) {
	 dirnode->path = g_strdup(path);
	 end = strrchr (dirnode->path, '/');
	 if (end) *(end + 1) = '\0';
      } else if (!strcmp(filename, "..")) {
	 dirnode->path = g_strdup(path);
	 end = strrchr (dirnode->path, '/');
	 if (end && end != dv->root_dir) *end = '\0';
	 end = strrchr (dirnode->path, '/');
	 if (end) *(end + 1) = '\0';
      } else {
	 dirnode->path = g_strconcat (path, "/", NULL);
      }
      text = filename;

      folder = icon_get ("folder");
      ofolder = icon_get ("folder-open");
      node = gtk_ctree_insert_node (GTK_CTREE (dv->dirtree), parent,
				    NULL, &text, 4,
				    folder->pixmap, folder->mask,
				    ofolder->pixmap, ofolder->mask,
				    FALSE, FALSE);

      gtk_ctree_node_set_row_data_full (GTK_CTREE (dv->dirtree), node,
					dirnode, cb_node_destroy);

      /* set expander */
      dirview_insert_dummy_node (dv, node);
   }

   return node;
}


/*
 *  wind_up_children:
 *     @ delete "." and ".." directory if needed.
 *
 *  dv     : Pointer to the Dir View struct.
 *  parnt  : Parent node to wind up.
 */
void
wind_up_children (DirView *dv, GtkCTreeNode *parent)
{
   GtkCTreeNode *node, *node1 = NULL, *node2 = NULL;
   gchar *row1, *row2;
   gint num;
   gboolean is_leaf, expanded;

   g_return_if_fail (dv);

   if (!GTK_CTREE_ROW (parent)->children) return;

   node = GTK_CTREE_ROW (parent)->children;
   for (num = 0; node; num++) node = GTK_CTREE_ROW (node)->sibling;

   if (num > 2) return;

   if (num > 0) {
      node1 = GTK_CTREE_ROW (parent)->children;
      gtk_ctree_get_node_info (GTK_CTREE (dv->dirtree), node1, &row1, NULL,
			       NULL, NULL, NULL, NULL, &is_leaf, &expanded);
   }
   if (num > 1) {
      node2 = GTK_CTREE_ROW (node1)->sibling;
      gtk_ctree_get_node_info (GTK_CTREE (dv->dirtree), node2, &row2, NULL,
			       NULL, NULL, NULL, NULL, &is_leaf, &expanded);
   }

   if (num == 1 && (!strcmp (row1, ".") || !strcmp (row1, ".."))) {
      gtk_ctree_remove_node (GTK_CTREE (dv->dirtree), node1);
   } else if (num == 2 && !strcmp (row1, ".") && !strcmp (row2, "..")) {
      gtk_ctree_remove_node (GTK_CTREE (dv->dirtree), node1);
      gtk_ctree_remove_node (GTK_CTREE (dv->dirtree), node2);
   }
}


/*
 *  expand_dir:
 *     @ Expand specified direcotry tree.
 *
 *  dv      : Pointer to the DirView struct.
 *  parent  : ctree node that contain direcotry to expand.
 */
static void
expand_dir (DirView *dv, GtkCTreeNode *parent)
{
   GtkWidget *dirtree;
   DirNode *parent_dirnode;
   DIR *dir;
   struct dirent *dirent;
   gchar *path;

   g_return_if_fail (dv);

   dirtree = dv->dirtree;
   if (!dirtree) return;

   parent_dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(dirtree), parent);

   dir = opendir (parent_dirnode->path);
   if (dir) {
      while ((dirent = readdir(dir))) {
	 path = g_strconcat (parent_dirnode->path, dirent->d_name, NULL);
	 if (isdir (path))
	    dirview_insert_node (dv, parent, path);
	 g_free (path);
      }
      closedir (dir);
      gtk_ctree_sort_node (GTK_CTREE(dirtree), parent);
   }
   parent_dirnode->scanned = TRUE;
}


/*
 *  refresh_dir_tree:
 *     @ Refresh sub direcotry list.
 *
 *  dv      : Pointer to the DirView struct.
 *  parent  : ctree node that contain direcotry to refresh.
 */
static void
refresh_dir_tree (DirView *dv,  GtkCTreeNode *parent)
{
   GtkWidget *dirtree;
   GtkCTreeNode *node, *node_tmp;
   DirNode *dirnode, *parent_dirdata, *dirdata;
   gboolean is_leaf, expanded, found = FALSE;
   GList *subdir_list, *listnode = NULL;
   gint length, i, flags;
   gchar *dirname, *node_text;

   g_return_if_fail (dv);

   dirtree = dv->dirtree;
   g_return_if_fail (dirtree);

   /* select parent node to refresh */
   if (!parent) {
      if (GTK_CLIST (dirtree)->selection) {
	 parent = GTK_CLIST (dirtree)->selection->data;
      } else {
	 return;
      }
   }

   /* get data of parent node */
   parent_dirdata = gtk_ctree_node_get_row_data (GTK_CTREE (dirtree), parent);
   gtk_ctree_get_node_info (GTK_CTREE (dirtree), parent, &node_text, NULL,
			    NULL, NULL, NULL, NULL, &is_leaf, &expanded);

   /* check whether parent directory contains sub directory or not */
   if (!expanded) {
      dirview_insert_dummy_node (dv, parent);
      return;
   }

   gtk_clist_freeze (GTK_CLIST (dirtree));

   dirnode = gtk_ctree_node_get_row_data (GTK_CTREE (dirtree), parent);

   if (!dirnode) return;

   /* get file list from disk */
   flags = GETDIR_FOLLOW_SYMLINK;
   if (dv->show_dotfile)
      flags = flags | GETDIR_READ_DOT;
   get_dir (dirnode->path, flags, NULL, &subdir_list);

   /* compare file list on memory to file list on disk */
   node = GTK_CTREE_ROW (parent)->children;
   i = 0;
   while (node) {
      gtk_ctree_get_node_info (GTK_CTREE (dirtree), node, &node_text, NULL,
			       NULL, NULL, NULL, NULL, &is_leaf, &expanded);
      dirdata = gtk_ctree_node_get_row_data (GTK_CTREE (dirtree), node);

      if (!dirdata) continue;

      /* remove last "/" character if exist */
      dirname = g_strdup (dirdata->path);
      length = strlen (dirname);
      if (dirname [length - 1] == '/' && length != 1)
	 dirname [length - 1] = '\0';

      /* remove from disk file list if exist on memory */
      listnode = subdir_list;
      while (listnode) {
	 if (!strcmp (dirname, (gchar *) listnode->data)) {
	    g_free (listnode->data);
	    subdir_list = g_list_remove (subdir_list, listnode->data);
	    found = TRUE;
	    break; 
	 }
	 listnode = g_list_next (listnode);
      }

      node_tmp = node;
      node = GTK_CTREE_ROW (node)->sibling;

      /* remove node if not exist on disk */
      if (!found && !(!strcmp (node_text, ".") || !strcmp (node_text, ".."))) {
	 gtk_ctree_remove_node (GTK_CTREE (dirtree), node_tmp);
      } else {
	 dirview_insert_dummy_node (dv, node_tmp);
	 refresh_dir_tree (dv, node_tmp);
      }

      found = FALSE;
      g_free (dirname);
   }

   /* add new directories */
   listnode = subdir_list;
   while (listnode) {
      if (isdir ((gchar *) listnode->data)) {
	 dirview_insert_node (dv, parent, (gchar *) listnode->data);
      }
      listnode = g_list_next (listnode);
   }
   gtk_ctree_sort_node (GTK_CTREE(dirtree), parent);

   /* adjust_ctree (dirtree, parent); */
   gtk_clist_thaw (GTK_CLIST (dirtree));

   g_list_foreach (subdir_list, (GFunc) g_free, NULL);
   g_list_free (subdir_list);
}



/******************************************************************************
 *
 *   for drag motion related functions
 *
 ******************************************************************************/
/*
 *  dirview_setup_drag_scroll:
 *     @ set timer for auto scroll.
 *
 *  dv        : Pointer to the DirView struct.
 *  x         : X position of mouse cursor.
 *  y         : Y positoon of mouse cursor.
 *  desirable :
 *  scroll    : time out function for auto scroll.
 */
static void
dirview_setup_drag_scroll (DirView *dv, gint x, gint y,
			   desirable_fn desirable, scroll_fn scroll)
{
   dirview_cancel_drag_scroll (dv);

   dv->drag_motion_x = x;
   dv->drag_motion_y = y;

   if ((* desirable) (dv, x, y))
      dv->scroll_timer_id = gtk_timeout_add (conf.dirview_auto_scroll_time,
					     scroll, dv);
}


/*
 *  dirview_cancel_drag_scroll:
 *     @ remove timer for auto scroll..
 *
 *  dv : Pointer to the DirView struct.
 */
static void
dirview_cancel_drag_scroll (DirView *dv)
{
   g_return_if_fail (dv);

   if (dv->scroll_timer_id != -1){
      gtk_timeout_remove (dv->scroll_timer_id);
      dv->scroll_timer_id = -1;
   }
}


/*
 *  dirview_scrolling_is_desirable:
 *     @ If the cursor is in a position close to either edge (top or bottom)
 *     @ and there is possible to scroll the window, this routine returns
 *     @ true.
 *
 *  dv     : Pointer to the DirView struct for scrolling.
 *  x      : X position of mouse cursor.
 *  y      : Y position of mouse cursor.
 *  Return : possiblilty of scrolling (TRUE or FALSE).
 */
static gboolean
dirview_scrolling_is_desirable (DirView *dv, gint x, gint y)
{
   GtkCTree *dirtree;
   GtkAdjustment *vadj;

   dirtree = GTK_CTREE (dv->dirtree);

   vadj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (dv->scroll_win));

   if (y < SCROLL_EDGE) {
      if (vadj->value > vadj->lower)
	 return TRUE;
   } else {
      if (y > (vadj->page_size - SCROLL_EDGE)){
	 if (vadj->value < vadj->upper - vadj->page_size)
	    return TRUE;
      }
   }

   return FALSE;
}


/*
 *  timeout_dirview_scroll:
 *     @ Timer callback to scroll the tree.
 *
 *  data   : Pointer to the DirView struct.
 *  Return : TRUE
 */
static gboolean
timeout_dirview_scroll (gpointer data)
{
   DirView *dv = data;
   GtkAdjustment *vadj;
   gfloat vpos;

   vadj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (dv->scroll_win));

   if (dv->drag_motion_y < SCROLL_EDGE) {
      vpos = vadj->value - vadj->step_increment;
      if (vpos < vadj->lower)
	 vpos = vadj->lower;

      gtk_adjustment_set_value (vadj, vpos);
   } else {
      vpos = vadj->value + vadj->step_increment;
      if (vpos > vadj->upper - vadj->page_size)
	 vpos = vadj->upper - vadj->page_size;

      gtk_adjustment_set_value (vadj, vpos);
   }

   return TRUE;
}


/*
 *  timeout_auto_expand_directory:
 *     @ This routine is invoked in a delayed fashion if the user
 *     @ keeps the drag cursor still over the widget.
 *
 *  data   : Pointer to the DirView struct.
 *  Retuan : FALSE
 */
static gint
timeout_auto_expand_directory (gpointer data)
{
   DirView *dv = data;
   GtkCTreeNode *node;

   g_return_val_if_fail (dv, FALSE);

   node = gtk_ctree_node_nth (GTK_CTREE (dv->dirtree), dv->drag_tree_row);
   if (!node) {
      dv->auto_expand_timeout_id = 0;
      return FALSE;
   }

   if (!GTK_CTREE_ROW (node)->expanded) {
      gtk_ctree_expand (GTK_CTREE (dv->dirtree), node);
   }

   return FALSE;
}



/******************************************************************************
 *
 *   Callback functions
 *
 ******************************************************************************/
/*
 *  cb_open_thumbnail:
 *     @ Callback for scrolled window destroy event.
 *
 *  dv       : Pointer to the DirView struct.
 *  action   :
 *  menuitem :
 */
static void
cb_dirview_destroy (GtkWidget *widget, DirView *dv)
{
   g_return_if_fail (dv);

   g_free (dv->root_dir);
   g_free (dv);
}


/*
 *  cb_node_destroy
 *     @ for gtk_ctree_node_set_row_data_full ()
 *     @ (GtkDestroyNotify)
 *
 *  data :
 */
static void
cb_node_destroy (gpointer data)
{
   DirNode *node = data;
   g_free (node->path);
   g_free (node);
}


/*
 *  cb_expand:
 *     @ Callback function for GtkCtree tree_expand event.
 *
 *  dirtree :
 *  parent  :
 *  dv      : Pointer to the DirView struct.
 */
static void
cb_expand (GtkWidget *dirtree, GtkCTreeNode *parent, DirView *dv)
{
   GtkCTreeNode *node;
   DirNode *dirnode;

   g_return_if_fail (dv);

   if (!GTK_CTREE_ROW (parent)->children) return;

   dirnode = gtk_ctree_node_get_row_data (GTK_CTREE (dirtree), parent);

   if (!dirnode->scanned) {
      gtk_clist_freeze (GTK_CLIST (dirtree));

      node = gtk_ctree_find_by_row_data (GTK_CTREE (dirtree), parent, NULL);
      gtk_ctree_remove_node (GTK_CTREE (dirtree), node);

      expand_dir (dv, parent);

      gtk_clist_thaw (GTK_CLIST (dirtree));
   }
}


/*
 *  cb_collapse:
 *     @ Callback function for GtkCtree tree_collapse event.
 *
 *  dirtree :
 *  parent  :
 *  dv      : Pointer to the DirView struct.
 */
static void
cb_collapse (GtkWidget *dirtree, GtkCTreeNode *parent, DirView *dv)
{
   g_return_if_fail (dv && parent);

   wind_up_children (dv, parent);
}


/*
 *  cb_button_press:
 *     @ Callback function for GtkCtree button_press event.
 *
 *  dirtree :
 *  parent  :
 *  dv      : Pointer to the DirView struct.
 */
static void
cb_button_press (GtkWidget *widget, GdkEventButton *event, DirView *dv)
{
   ThumbWindow *tw;
   GtkWidget *dirtree, *menuitem;
   GtkCTreeNode *node;
   GtkItemFactory *ifactory;
   DirNode *dirnode;
   gchar *node_text;
   gint row;
   gboolean is_leaf, expanded;

   g_return_if_fail (dv);

   dirtree = dv->dirtree;
   tw = dv->tw;

   gtk_clist_get_selection_info(GTK_CLIST(widget),
				event->x, event->y, &row, NULL);
   node = gtk_ctree_node_nth(GTK_CTREE(widget), row);
   if (!node) return;

   gtk_ctree_get_node_info (GTK_CTREE (widget), node, &node_text, NULL,
			    NULL, NULL, NULL, NULL, &is_leaf, &expanded);

   gtk_clist_freeze(GTK_CLIST(widget));
   gtk_ctree_select(GTK_CTREE(widget), node);
   GTK_CLIST(dirtree)->focus_row = row;
   gtk_clist_thaw(GTK_CLIST(widget));
   dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(widget), node);

   if ((event->type == GDK_BUTTON_PRESS && event->button == 2)
       || (event->type == GDK_2BUTTON_PRESS && event->button == 1))
   {
      if (!strcmp (node_text, ".") || !strcmp (node_text, "..")) {
	 dirview_change_root (dv, dirnode->path);
      } else {
	 open_dir_images (dirnode->path, dv->tw,
			  LOAD_CACHE, conf.scan_dir_recursive);
	 //gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "button_press_event");
      }
   } else if (event->type == GDK_BUTTON_PRESS && event->button == 3) {
      GtkWidget *dirview_popup;
      gint n_menu_items;
      gchar *path, *parent;

      n_menu_items = sizeof(dirview_popup_items)
	                / sizeof(dirview_popup_items[0]) -  1;
      dirview_popup = menu_create_items(NULL, dirview_popup_items,
					n_menu_items, "<DirViewPop>", dv);

      /* set sensitive */
      ifactory = gtk_item_factory_from_widget (dirview_popup);

      if (!iswritable (dirnode->path)
	  || !strcmp (node_text, ".") || !strcmp (node_text, ".."))
      {
	 menuitem = gtk_item_factory_get_item (ifactory, "/Make Directory");
	 gtk_widget_set_sensitive (menuitem, FALSE);
	 menuitem = gtk_item_factory_get_item (ifactory, "/Refresh Tree");
	 gtk_widget_set_sensitive (menuitem, FALSE);
      }

      path = g_strdup (dirnode->path);
      if (path [strlen (path) - 1] == '/')
	 path [strlen (path) - 1] = '\0';
      parent = g_dirname (path);

      if (!parent || !strcmp (parent, ".") || !iswritable (parent)
	  || !strcmp (node_text, ".") || !strcmp (node_text, ".."))
      {
	 menuitem = gtk_item_factory_get_item (ifactory, "/Rename Directory");
	 gtk_widget_set_sensitive (menuitem, FALSE);
	 menuitem = gtk_item_factory_get_item (ifactory, "/Delete Directory");
	 gtk_widget_set_sensitive (menuitem, FALSE);
      }

      g_free (path);
      g_free (parent);

      /* popup menu */
      gtk_menu_popup(GTK_MENU (dirview_popup), NULL, NULL,
		     NULL, NULL, event->button, event->time);
   }
}



/******************************************************************************
 *
 *   Callback functions for popup menu
 *
 ******************************************************************************/
/*
 *  cb_open_thumbnail:
 *     @ Callback function for popup menu. ("/Load Thumbnail")
 *
 *  dv       : Pointer to the DirView struct.
 *  action   :
 *  menuitem :
 */
static void
cb_open_thumbnail (DirView *dv, gboolean scan_subdir, GtkWidget *menuitem)
{
   GtkWidget *dirtree;
   DirNode *dirnode;
   GtkCTreeNode *parent = NULL;
   ThumbWindow *tw;

   dirtree = dv->dirtree;
   if (!dirtree) return;

   tw = dv->tw;

   if (GTK_CLIST (dirtree)->selection) {
      parent = GTK_CLIST (dirtree)->selection->data;
   } else {
      return;
   }
   dirnode = gtk_ctree_node_get_row_data (GTK_CTREE (dirtree), parent);
   open_dir_images (dirnode->path, tw, LOAD_CACHE, scan_subdir);
}


/*
 *  cb_go_to_here:
 *     @ Callback function for popup menu. ("/Go to here")
 *
 *  dv       : Pointer to the DirView struct.
 *  action   :
 *  menuitem :
 */
static void
cb_go_to_here (DirView *dv, guint action, GtkWidget *menuitem)
{
   GtkWidget *dirtree;
   DirNode *dirnode;
   GtkCTreeNode *parent = NULL;
   ThumbWindow *tw;

   dirtree = dv->dirtree;
   if (!dirtree) return;

   tw = dv->tw;

   if (GTK_CLIST (dirtree)->selection) {
      parent = GTK_CLIST (dirtree)->selection->data;
   } else {
      return;
   }

   dirnode = gtk_ctree_node_get_row_data (GTK_CTREE (dirtree), parent);
   dirview_change_root (dv, dirnode->path);
}


/*
 *  cb_refresh_dir_tree:
 *     @ Callback function for popup menu. ("/Refresh")
 *
 *  dv       : Pointer to the DirView struct.
 *  action   :
 *  menuitem :
 */
static void
cb_refresh_dir_tree (DirView *dv, guint action, GtkWidget *menuitem)
{
   refresh_dir_tree (dv, NULL);
}


/*
 *  cb_mkdir:
 *     @ Callback function for popup menu. ("/Make Directory")
 *
 *  dv       : Pointer to the DirView struct.
 *  action   :
 *  menuitem :
 */
static void
cb_mkdir (DirView *dv, guint action, GtkWidget *menuitem)
{
   GtkCTreeNode *node;
   DirNode *dirnode;
   gchar *dirname, *path;
   gboolean success, exist;
   struct stat st;
   gchar error_message[BUF_SIZE];

   g_return_if_fail (dv);
   node = GTK_CLIST (dv->dirtree)->selection->data;
   if (!node) return;

   dirnode = gtk_ctree_node_get_row_data (GTK_CTREE(dv->dirtree), node);
   if (!dirnode) return;

   if (!iswritable (dirnode->path)) {
      g_snprintf (error_message, BUF_SIZE,
		  _("Permission denied : %s"), dirnode->path);
      gtkutil_message_dialog (_("Error!!"), error_message);
      return;
   }

   dirname = gtkutil_popup_textentry (_("Make directory"),
				      _("New directory name: "),
				      NULL);
   if (!dirname) return;

   path = g_strconcat (dirnode->path, dirname, NULL);

   exist = !lstat (path, &st);
   if (exist) {
      if (isdir (path))
	 g_snprintf (error_message, BUF_SIZE,
		     _("Directory exist : %s"), path);
      else
	 g_snprintf (error_message, BUF_SIZE,
		     _("File exist : %s"), path);
      gtkutil_message_dialog (_("Error!!"), error_message);
      g_free (path);
      return;
   }

   success = makedir (path);
   if (!success) {
      g_snprintf (error_message, BUF_SIZE,
		  _("Faild to create directory : %s"), path);
      gtkutil_message_dialog (_("Error!!"), error_message);
   }

   refresh_dir_tree (dv, node);

   g_free (path);
}


/*
 *  cb_rename_dir:
 *     @ Callback function for popup menu. ("/Rename directtory")
 *
 *  dv       : Pointer to the DirView struct.
 *  action   :
 *  menuitem :
 */
static void
cb_rename_dir (DirView *dv, guint action, GtkWidget *menuitem)
{
   GtkCTreeNode *node, *parent;
   DirNode *dirnode;
   gboolean exist;
   struct stat st;
   gchar message[BUF_SIZE];
   gchar *parent_dir, *dirname, *src_path, *dest_path;

   g_return_if_fail (dv);
   node = GTK_CLIST (dv->dirtree)->selection->data;
   if (!node) return;

   dirnode = gtk_ctree_node_get_row_data (GTK_CTREE(dv->dirtree), node);
   if (!dirnode) return;

   /* check for direcotry exist */
   exist = !lstat (dirnode->path, &st);
   if (!exist) {
      g_snprintf (message, BUF_SIZE,
		  _("Directory not exist :%s"), dirnode->path);
      gtkutil_message_dialog (_("Error!!"), message);
      return;
   }

   src_path = g_strdup (dirnode->path);
   if (src_path [strlen (src_path) - 1] == '/')
      src_path [strlen (src_path) - 1] = '\0';
   parent_dir = g_dirname (src_path);

   /* popup rename directory dialog */
   dirname = gtkutil_popup_textentry (_("Rename directory"),
				      _("New directory name: "),
				       g_basename (src_path));
   if (!dirname) {
      g_free (src_path);
      g_free (parent_dir);
      return;
   }

   dest_path = g_strconcat (parent_dir, "/", dirname, NULL);

   if (rename (src_path, dest_path) < 0) {
      g_snprintf (message, BUF_SIZE,
		  _("Faild to rename directory : %s"), src_path);
      gtkutil_message_dialog (_("Error!!"), message);
   }

   /* refresh dir tree */
   parent = (GTK_CTREE_ROW (node))->parent;
   if (parent)
      refresh_dir_tree (dv, parent);

   g_free (parent_dir);
   g_free (dirname);
   g_free (dest_path);
}


/*
 *  cb_delete_dir:
 *     @ Callback function for popup menu. ("/Delete directtory")
 *
 *  dv       : Pointer to the DirView struct.
 *  action   :
 *  menuitem :
 */
static void
cb_delete_dir (DirView *dv, guint action, GtkWidget *menuitem)
{
   GtkCTreeNode *node, *parent;
   DirNode *dirnode;
   gboolean exist, cancel, not_empty = FALSE;
   struct stat st;
   gchar message[BUF_SIZE], *path;
   GList *filelist, *listnode;
   gint length, pos;
   gfloat progress;
   GtkWidget *progress_win;

   g_return_if_fail (dv);
   node = GTK_CLIST (dv->dirtree)->selection->data;
   if (!node) return;

   dirnode = gtk_ctree_node_get_row_data (GTK_CTREE(dv->dirtree), node);
   if (!dirnode) return;

   path = g_strdup (dirnode->path);
   if (path [strlen (path) - 1] == '/')
      path [strlen (path) - 1] = '\0';

   /* check for direcotry exist */
   exist = !lstat (path, &st);
   if (!exist) {
      g_snprintf (message, BUF_SIZE,
		  _("Directory not exist : %s"), path);
      gtkutil_message_dialog (_("Error!!"), message);
      return;
   }

   /* chack path is link or not */
   if (islink (path)) {
       g_snprintf (message, BUF_SIZE,
		  _("%s is symbolic link.\n"
		    "Remove link ?"), path);
      action = gtkutil_confirm_dialog (_("Confirm Deleting Directory"),
				       message, FALSE);
      if (action == CONFIRM_YES) {
	 remove (path);
      }
      parent = (GTK_CTREE_ROW (node))->parent;
      if (parent)
	 refresh_dir_tree (dv, parent);
      g_free (path);
      return;
   }

   g_free (path);

   /* confirm */
   g_snprintf (message, BUF_SIZE,
	       _("Delete %s\n"
		 "OK?"), dirnode->path);
   action = gtkutil_confirm_dialog (_("Confirm Deleting Directory"),
				    message, FALSE);
   if (action != CONFIRM_YES) {
      return;
   }

   /* remove sub directories recursively */
   filelist = get_dir_all (dirnode->path);
   if (filelist) {
      g_snprintf (message, BUF_SIZE,
		  _("%s is not empty\n"
		    "Delete all files under %s ?"),
		  dirnode->path, dirnode->path);
      action = gtkutil_confirm_dialog (_("Confirm Deleting Directory"),
				       message, FALSE);
      if (action != CONFIRM_YES) {
	 return;
      }

      /* create progress bar */
      progress_win = gtkutil_create_progress_window ("Delete File", "Deleting Files",
						     &cancel, 300, -1);
      gtk_grab_add (progress_win);

      length = g_list_length (filelist);
      listnode = filelist;
      while (listnode) {
	 /* update progress */
	 pos = g_list_position (filelist, listnode);
	 if ((pos % 50) == 0) {
	    while (gtk_events_pending()) gtk_main_iteration();
	    pos = g_list_position (filelist, listnode);
	    progress = (gfloat) pos / (gfloat) length;
	    g_snprintf (message, BUF_SIZE, _("Deleting %s ..."),
			(gchar *) listnode->data);
	    gtkutil_progress_window_update (progress_win, NULL,
					    message, NULL, progress);
	 }

	 /* remove a file */
	 if (remove ((gchar *) listnode->data) < 0)
	    not_empty = TRUE;
	 listnode = g_list_next (listnode);

	 /* cancel */
	 if (cancel) break;
      }
      gtk_grab_remove (progress_win);
      gtk_widget_destroy (progress_win);

      g_list_foreach (filelist, (GFunc) g_free, NULL);
      g_list_free (filelist);
   }

   /* remove the directory */
   if (not_empty) {
      g_snprintf (message, BUF_SIZE,
		  _("Faild to remove directory :\n"
		    "%s is not empty."), dirnode->path);
      gtkutil_message_dialog (_("Error!!"), message);
   } else if (remove (dirnode->path) < 0) {
      g_snprintf (message, BUF_SIZE,
		  _("Faild to remove directory : %s"), dirnode->path);
      gtkutil_message_dialog (_("Error!!"), message);
   }

   /* refresh dir tree */
   parent = (GTK_CTREE_ROW (node))->parent;
   if (parent)
      refresh_dir_tree (dv, parent);
}



/******************************************************************************
 *
 *   Callback functions for toolbar buttons.
 *
 ******************************************************************************/
static void
cb_home_button (GtkWidget *widget, DirView *dv)
{
   g_return_if_fail (dv);

   dirview_go_home (dv);
}


static void
cb_up_button (GtkWidget *widget, DirView *dv)
{
   gchar *end;

   gtk_widget_destroy (dv->dirtree);
   end = strrchr (dv->root_dir, '/');
   if (end && end != dv->root_dir) *end = '\0';
   end = strrchr (dv->root_dir, '/');
   if (end) *(end + 1) = '\0';
   dirview_create_ctree (dv);
}


static void
cb_refresh_button (GtkWidget *widget, DirView *dv)
{
   GtkCTreeNode *parent;

   g_return_if_fail (dv);

   parent = gtk_ctree_node_nth (GTK_CTREE (dv->dirtree), 0);
   if (parent)
      refresh_dir_tree (dv, parent);
}


static void
cb_dotfile_button (GtkWidget *widget, DirView *dv)
{
   GtkCTreeNode *parent;

   g_return_if_fail (widget && dv);

   parent = gtk_ctree_node_nth (GTK_CTREE (dv->dirtree), 0);

   dv->show_dotfile = !dv->show_dotfile;
   refresh_dir_tree (dv, parent);
}



/******************************************************************************
 *
 *   Callback functions for DnD
 *
 ******************************************************************************/
static void
cb_drag_motion (GtkWidget *dirtree, GdkDragContext *context,
		gint x, gint y, gint time, gpointer data)
{
   DirView *dv = data;
   GtkCTreeNode *node;
   DirNode *node_data;
   gint row, on_row;

   g_return_if_fail (dv);

   /* Set up auto-scrolling */
   if (conf.dirview_auto_scroll)
      dirview_setup_drag_scroll (dv, x, y,
				 dirview_scrolling_is_desirable,
				 timeout_dirview_scroll);

   if (!GTK_CLIST (dirtree)->selection) return;

   on_row = gtk_clist_get_selection_info (GTK_CLIST (dirtree), x, y, &row, NULL);

   /* Remove the auto-expansion timeout if we are on the blank area of the
    * tree or on a row different from the previous one.
    */
   if ((!on_row || row != dv->drag_tree_row) && dv->auto_expand_timeout_id != 0) {
      gtk_timeout_remove (dv->auto_expand_timeout_id);
      dv->auto_expand_timeout_id = 0;
   }

   if (on_row) {
      node = gtk_ctree_node_nth (GTK_CTREE (dv->dirtree), row);
      node_data = gtk_ctree_node_get_row_data (GTK_CTREE (dirtree),
					       gtk_ctree_node_nth (GTK_CTREE (dirtree),
								   row));

      /* hilight row under cursor */
      if (node_data && dv->hilit_dir != row) {
	 if (dv->hilit_dir != -1) {
	    gtk_clist_set_foreground (GTK_CLIST (dirtree), dv->hilit_dir,
				      &(dirtree->style->fg[GTK_STATE_NORMAL]));
	    gtk_clist_set_background (GTK_CLIST (dirtree), dv->hilit_dir,
				      &(dirtree->style->base[GTK_STATE_NORMAL]));
	 }

	 dv->hilit_dir = row;

	 gtk_clist_set_foreground (GTK_CLIST (dirtree), dv->hilit_dir,
				   &(dirtree->style->fg[GTK_STATE_SELECTED]));
	 gtk_clist_set_background (GTK_CLIST (dirtree), dv->hilit_dir,
				   &(dirtree->style->bg[GTK_STATE_SELECTED]));
      }

      /* Install the timeout handler for auto-expansion */
      if (conf.dirview_auto_expand && row != dv->drag_tree_row) {
	 dv->auto_expand_timeout_id = gtk_timeout_add (conf.dirview_auto_expand_time,
						       timeout_auto_expand_directory,
						       dv);
	 dv->drag_tree_row = row;
      }
   }
}


static void
cb_drag_leave (GtkWidget *dirtree, GdkDragContext *context, guint time,
	       gpointer data)
{
   DirView *dv = data;

   dirview_cancel_drag_scroll (dv);

   gtk_clist_set_foreground (GTK_CLIST (dirtree), dv->hilit_dir,
			    &(dirtree->style->fg[GTK_STATE_NORMAL]));
   gtk_clist_set_background (GTK_CLIST (dirtree), dv->hilit_dir,
			    &(dirtree->style->base[GTK_STATE_NORMAL]));
   dv->hilit_dir = -1;


   /* remove auto expand handler */
   if (dv->auto_expand_timeout_id != 0) {
      gtk_timeout_remove (dv->auto_expand_timeout_id);
      dv->auto_expand_timeout_id = 0;
   }

   dv->drag_tree_row = -1;
}


static void
cb_drag_data_get (GtkWidget *dirtree,
		  GdkDragContext *context,
		  GtkSelectionData *seldata,
		  guint info,
		  guint time,
		  gpointer data)
{
   DirView *dv = data;
   GtkCTreeNode *node;
   DirNode *dirnode;
   gchar *uri, *tmpstr;

   g_return_if_fail (dv && dirtree);

   dirview_cancel_drag_scroll (dv);

   node = GTK_CLIST (dirtree)->selection->data;
   if (!node) return;

   dirnode = gtk_ctree_node_get_row_data(GTK_CTREE(dirtree), node);
   g_return_if_fail (dirnode);

   tmpstr = g_strdup (dirnode->path);
   if (tmpstr [strlen (tmpstr) - 1] == '/')
      tmpstr [strlen (tmpstr) - 1] = '\0';
   uri = g_strconcat ("file:", tmpstr, "\r\n", NULL);

   gtk_selection_data_set(seldata, seldata->target,
			  8, uri, strlen(uri));
   g_free (tmpstr);
   g_free (uri);
}


static void
cb_drag_data_received (GtkWidget *dirtree,
		       GdkDragContext *context,
		       gint x, gint y,
		       GtkSelectionData *seldata,
		       guint info,
		       guint32 time,
		       gpointer data)
{
   gint row;
   DirNode *node;
   DirView *dv = data;
   GtkCTreeNode *parent;

   g_return_if_fail (dv);

   dirview_cancel_drag_scroll (dv);

   if (!GTK_CLIST (dirtree)->selection) return;

   gtk_clist_get_selection_info (GTK_CLIST (dirtree), x, y, &row, NULL);
   parent = gtk_ctree_node_nth (GTK_CTREE (dirtree), row);
   node = gtk_ctree_node_get_row_data (GTK_CTREE (dirtree), parent);

   if (node && iswritable (node->path)) {
      dnd_file_operation (node->path, context, seldata, time, dv->tw);
      refresh_dir_tree (dv, parent);
   }
}


static void
cb_drag_end (GtkWidget *dirtree, GdkDragContext *context, gpointer data)
{
   DirView *dv = data;
   GtkCTreeNode *parent;

   g_return_if_fail (dirtree && dv);

   dirview_cancel_drag_scroll (dv);

   parent = gtk_ctree_node_nth (GTK_CTREE (dv->dirtree), 0);

   refresh_dir_tree (dv, parent);
}



/******************************************************************************
 *
 *   Public  functions
 *
 ******************************************************************************/
/*
 *  dirview_change_root:
 *     @ Change top directory.
 *
 *   dirtree   : GtkCtree widget.
 *   root_drir : path to change.
 */
void
dirview_change_root (DirView *dv, gchar *root_dir)
{
   gchar *tmpstr;
   GtkCTreeNode *node;

   g_return_if_fail (dv && root_dir);

   if (!root_dir) return;
   if (!strcmp (dv->root_dir, root_dir)) return;

   tmpstr = g_strdup(root_dir);
   gtk_widget_destroy (dv->dirtree);

   if (dv->root_dir)
      g_free (dv->root_dir);
   dv->root_dir = tmpstr;
   dirview_create_ctree (dv);

   node = gtk_ctree_node_nth (GTK_CTREE (dv->dirtree), 0);
   if (node)
      adjust_ctree (dv->dirtree, node);
}


/*
 *  dirview_change_dir:
 *     @ Change selection to specified directory.
 *
 *   dirtree : GtkCtree widget:
 *   str     : path to change.
 */
void
dirview_change_dir (DirView *dv, gchar *str)
{
   GtkWidget *dirtree;
   GtkCTreeNode *node;
   gchar *path;
   gchar *row;
   gchar *ptr;
   gboolean is_leaf, expanded;
   gint length;

   g_return_if_fail (dv);

   dirtree = dv->dirtree;

   /* add '/' character to last of string */
   if (str [strlen (str) - 1] == '/')
      path = g_strdup (str);
   else
      path = g_strconcat (str, "/", NULL);
   ptr = path;

   /* get root node */
   node = gtk_ctree_node_nth (GTK_CTREE (dirtree), 0);
   gtk_ctree_get_node_info (GTK_CTREE (dirtree), node, &row, NULL,
			    NULL, NULL, NULL, NULL, &is_leaf, &expanded);
   if (!is_leaf && !expanded)
      gtk_ctree_expand (GTK_CTREE (dirtree), node);

   /* if root node is specified directory, adjust to root */
   if (!(strcmp (ptr, row))) {
      g_free (path);
      adjust_ctree (dirtree, node);
      return;
   }

   /* get first child */
   node = GTK_CTREE_ROW(node)->children;
   if (!node) {
      g_free (path);
      dirview_change_root (dv, str);
      return;
   }

   /*
    *  if specified directory string is shorter than root node's one,
    *  change top direcotry.
    */
   length = strlen (dv->root_dir);
   if (strlen (ptr) >= length) {
      ptr = ptr + length;
   } else {
      g_free (path);
      dirview_change_root (dv, str);
      return;
   }

   /* search children */
   gtk_clist_freeze (GTK_CLIST (dirtree));
   while (TRUE) {
      gtk_ctree_get_node_info (GTK_CTREE (dirtree), node, &row, NULL,
			       NULL, NULL, NULL, NULL, &is_leaf, &expanded);
      if (!strncmp (row, ptr, strlen (row)) && *(ptr+strlen (row)) == '/' ) {
	 ptr = ptr + strlen (row) + 1;
	 while (*ptr == '/')
	    ptr++;
	 if (!is_leaf && GTK_CTREE_ROW (node)->children && !expanded) {
	    gtk_ctree_expand(GTK_CTREE (dirtree), node);
	 }
	 if (!(ptr && *ptr))
	    break;
	 if (!(node = GTK_CTREE_ROW (node)->children))
	    break;
      }
      else if (!(node = GTK_CTREE_ROW (node)->sibling))
	 break;
   }
   gtk_clist_thaw (GTK_CLIST (dirtree));

   g_free (path);

   if (node) {
      adjust_ctree (dirtree, node);
   } else {
      dirview_change_root (dv, str);
   }
}


/*
 *  dirview_go_home:
 *     @ Change top directory to home directory.
 *
 *  dv : Pointer to the DirView struct.
 */
void
dirview_go_home (DirView *dv)
{
   gchar *home = g_getenv ("HOME");

   dirview_change_root (dv, home);
   dirview_change_dir (dv, home);
}


static gint
idle_go_home (gpointer data)
{
   DirView *dv = data;

   if (dv)
      dirview_go_home (dv);

   return FALSE;
}


/*
 *  dirview_show_toolbar:
 *     @
 *
 *  dv : Pointer to the DirView struct.
 */
void
dirview_show_toolbar (DirView *dv)
{
   g_return_if_fail (dv);

   dv->show_toolbar = TRUE;
   gtk_widget_show (dv->toolbar);
}


/*
 *  dirview_hide_toolbar:
 *     @
 *
 *  dv : Pointer to the DirView struct.
 */
void
dirview_hide_toolbar (DirView *dv)
{
   g_return_if_fail (dv);

   dv->show_toolbar = FALSE;
   gtk_widget_hide (dv->toolbar);
}


/*
 *  dirview_create_toolbar:
 *     @ 
 *
 *  dv : Pointer to the DirView struct.
 */
static GtkWidget *
dirview_create_toolbar (DirView *dv)
{
   GtkWidget *toolbar;
   GtkWidget *button;
   GtkWidget *iconw;   
   GtkWidget *container;

   g_return_val_if_fail (dv, NULL);

   if (dv->tw)
      container = dv->tw->window;
   else
      container = dv->window;

   toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);

   gtk_toolbar_set_button_relief (GTK_TOOLBAR(toolbar), GTK_RELIEF_NONE);
   gtk_toolbar_set_space_style   (GTK_TOOLBAR(toolbar), GTK_TOOLBAR_SPACE_LINE);
   gtk_toolbar_set_space_size    (GTK_TOOLBAR(toolbar), 16);

   /* file open button */
   iconw = icon_get_widget ("small_home");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Home"),
				    _("Home"),
				    _("Home"),
				    iconw, cb_home_button, dv);

   /* preference button */
   iconw = icon_get_widget ("small_up");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Up"),
				    _("Up"), _("Up"),
				    iconw, cb_up_button, dv);

   /* refresh button */
   iconw = icon_get_widget ("small_refresh");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Refresh"),
				    _("Refresh"), _("Refresh"),
				    iconw, cb_refresh_button, dv);

   /* preference button */
   iconw = icon_get_widget ("dotfile");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Dotfile"),
				    _("Show/Hide dotfile"), _("Show/Hide dotfile"),
				    iconw, cb_dotfile_button, dv);

   gtk_widget_show_all (toolbar);
   gtk_toolbar_set_style         (GTK_TOOLBAR(toolbar), GTK_TOOLBAR_ICONS);

   return toolbar;
}


/******************************************************************************
 *
 *   dirview_create:
 *      @ create directory view widget.
 *
 *   tw     :
 *   Return : directory view packed container widget.
 *
 ******************************************************************************/
DirView *
dirview_create (const gchar *root_dir, GtkWidget *parent_win, ThumbWindow *tw)
{
   DirView *dv;
   gchar *home = g_getenv ("HOME");

   dv = g_new0 (DirView, 1);
   if (root_dir)
      dv->root_dir = g_strdup (root_dir);
   else
      dv->root_dir = g_strdup (home);

   dv->show_toolbar = conf.dirview_show_toolbar;
   dv->show_dotfile = conf.dirview_show_dotfile;
   dv->tw = tw;
   dv->hilit_dir = -1;
   dv->scroll_timer_id = -1;

   /* main vbox */
   dv->container = gtk_vbox_new (FALSE, 0);
   gtk_widget_show (dv->container);

   /* toolbar */
   dv->toolbar = dirview_create_toolbar (dv);
   gtk_box_pack_start (GTK_BOX (dv->container), dv->toolbar, FALSE, FALSE, 0);
   if (!dv->show_toolbar)
      gtk_widget_hide (dv->toolbar);

   /* scrolled window */
   dv->scroll_win = gtk_scrolled_window_new (NULL, NULL);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (dv->scroll_win),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_box_pack_start(GTK_BOX(dv->container), dv->scroll_win, TRUE, TRUE, 0);
   gtk_signal_connect (GTK_OBJECT (dv->scroll_win), "destroy",
		       GTK_SIGNAL_FUNC (cb_dirview_destroy), dv);
   gtk_widget_show (dv->scroll_win);

   /* ctree */
   dirview_create_ctree (dv);

   gtk_idle_add (idle_go_home, dv);
   /* dirview_go_home (dirtree); */

   return dv;
}
