// T3MM1DOCV.CPP

// Copyright (C) 1998-2000 Tommi Hassinen, Geoff Hutchison.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "t3mm1docv.h"

#include "glade_interface.h"

#include "t3main.h"
#include "t3dialog.h"
#include "t3errutil.h"
#include "t3views.h"

#include "mm1alg.h"
#include "mm1eng1.h"
#include "mm1eng9.h"

#include "mm1rbn.h"

#include "plane.h"
#include "surface.h"

#include "color.h"
#include "views.h"

#include "filetrans.h"

#include <sys/stat.h> // for stat
#include <dirent.h> // for scandir

#include <fstream>
#include <strstream>
using namespace std;

/*################################################################################################*/

const char mm1_file_open_dialog::title[] = "Open Project File - mm1gp";
gnome_mm1_docv * mm1_file_open_dialog::docv = NULL;

mm1_file_open_dialog::mm1_file_open_dialog(gnome_mm1_docv * p1) : file_dialog(title, NULL, NULL)
{
	docv = p1;
}

mm1_file_open_dialog::~mm1_file_open_dialog(void)
{
}

bool mm1_file_open_dialog::OkEvent(const char * filename)
{
	if (docv == NULL)
	{
		docv = new gnome_mm1_docv(& cout, * gnome_class_factory::GetInstance());
		docv->InitGnomeMDIChild();
	}
	
	bool open = docv->IsEmpty();	// open, not insert
	
	ifstream ifile;
	ifile.open(filename, ios::in);
	docv->ReadStream(ifile, !open);
	ifile.close();
	
	if (open) 	// we're "switching" to this file
	{
		docv->ParseProjectFileNameAndPath(filename);
		
		gnome_mdi_child_set_name(GNOME_MDI_CHILD(docv->child), docv->GetProjectFileName(true));
		docv->UpdateAllWindowTitles();
	}
	else		// we append this file to the current project
	{
		if (docv->selected_object != NULL)		// if an object was selected,
		{						// remove that selection and leave
			docv->selected_object = NULL;		// the newly added atoms as selection;
			docv->event_SelectedObjectChanged();	// now the mouse tools also affect the
		}						// newly added atoms!
	}
	
	docv->UpdateAllGraphicsViews();
	return true;
}

void mm1_file_open_dialog::CancelEvent(void)
{
}

/*################################################################################################*/

const char mm1_file_save_dialog::title[] = "Save Project File - mm1gp";
gnome_mm1_docv * mm1_file_save_dialog::docv = NULL;

mm1_file_save_dialog::mm1_file_save_dialog(gnome_mm1_docv * p1) :
	file_dialog(title, MM1_FILTER, p1->GetFullProjectFileName())
{
	docv = p1;
}

mm1_file_save_dialog::~mm1_file_save_dialog(void)
{
}

bool mm1_file_save_dialog::OkEvent(const char * filename)
{
	if (docv == NULL)
	{
	  docv->err->ErrorMessage("mm1_file_save_dialog::OkEvent() failed");
	  exit(EXIT_FAILURE);
	}
	
  	// should check to see if file already exists...

	docv->ParseProjectFileNameAndPath(filename);
	
	gnome_mdi_child_set_name(GNOME_MDI_CHILD(docv->child), docv->GetProjectFileName(true));
	docv->UpdateAllWindowTitles();
	
	ofstream ofile;
	ofile.open(docv->GetFullProjectFileName(), ios::out);
	docv->WriteStream(ofile);
	ofile.close();

	return true;
}

void mm1_file_save_dialog::CancelEvent(void)
{
}

/*################################################################################################*/

const char mm1_file_save_graphics_dialog::title[] = "Save Graphics File - ps";
gnome_mm1_docv * mm1_file_save_graphics_dialog::docv = NULL;

mm1_file_save_graphics_dialog::mm1_file_save_graphics_dialog(gnome_mm1_docv * p1) :
	file_dialog(title, "ps", p1->GetFullProjectFileName())
{
	docv = p1;
}

mm1_file_save_graphics_dialog::~mm1_file_save_graphics_dialog(void)
{
}

bool mm1_file_save_graphics_dialog::OkEvent(const char * filename)
{
	if (docv == NULL)
	{
	  docv->err->ErrorMessage("mm1_file_save_graphics_dialog::OkEvent() failed");
	  exit(EXIT_FAILURE);
	}
	
	// write graphics event
	return true;
}

void mm1_file_save_graphics_dialog::CancelEvent(void)
{
}

/*################################################################################################*/

const char mm1_trajfile_dialog::title[] = "Open Trajectory File - traj";
mm1_docv * mm1_trajfile_dialog::docv = NULL;

mm1_trajfile_dialog::mm1_trajfile_dialog(mm1_docv * p1) : file_dialog(title, NULL, NULL)
{
	docv = p1;
}

mm1_trajfile_dialog::~mm1_trajfile_dialog(void)
{
}

bool mm1_trajfile_dialog::OkEvent(const char * filename)
{
	docv->OpenTrajectory(filename);
	
	gtk_widget_destroy(dialog);	// close the dialog -> must return false...
	
	gnome_docv::glade_modal_dialog_owner = (gnome_mm1_docv *) docv;		// requires modal dialogs!!!
	
	GtkWidget * dialog = create_t1_dialog_trajectory();
	gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
	
	docv->CloseTrajectory();
	return false;
}

void mm1_trajfile_dialog::CancelEvent(void)
{
}

/*################################################################################################*/

GnomeUIInfo gnome_mm1_docv::file_submenu[] =
{
	GNOMEUIINFO_MENU_OPEN_ITEM(popup_FileOpen, NULL),

	GNOMEUIINFO_ITEM_STOCK("Import...", "import other file types...",
		popup_FileImport, GNOME_STOCK_MENU_OPEN),

	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_MENU_SAVE_AS_ITEM(popup_FileSaveAs, NULL),

	GNOMEUIINFO_ITEM_STOCK("Export...", "export other file types...",
		popup_FileExport, GNOME_STOCK_MENU_SAVE_AS),
	
//	GNOMEUIINFO_ITEM_STOCK("Export PostScript", "export graphic...",
//		popup_FileExportPS, GNOME_STOCK_MENU_SAVE_AS),
	
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_MENU_CLOSE_ITEM(gnome_graphics_view::popup_FileClose, NULL),
	
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::selmodes_radio[] =
{
	GNOMEUIINFO_ITEM("Select Atoms", "select atoms",
		NULL, NULL),
	GNOMEUIINFO_ITEM("Select Residues", "select residues",
		NULL, NULL),
	GNOMEUIINFO_ITEM("Select Chains", "select chains",
		NULL, NULL),
	GNOMEUIINFO_ITEM("Select Molecules", "select molecules",
		NULL, NULL),

	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::selmodes_submenu[] =
{
        GNOMEUIINFO_RADIOLIST(selmodes_radio),
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::select_submenu[] =
{
	GNOMEUIINFO_ITEM("Select All", "select all atoms...",
		gnome_graphics_view::popup_SelectAll, NULL),
		
	GNOMEUIINFO_ITEM("Select None", "select nothing...",
		gnome_graphics_view::popup_SelectNone, NULL),

	GNOMEUIINFO_ITEM("Invert Selection", "invert the selection...",
		gnome_graphics_view::popup_InvertSelection, NULL),
	
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_SUBTREE("_Selection Mode", selmodes_submenu),
	
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::settings_submenu[] =
{
	GNOMEUIINFO_ITEM("Set Current Element...", "el...",
		gnome_mdi_app::shortcut_Element, NULL),
		
	GNOMEUIINFO_ITEM("Set Current Bondtype...", "bt...",
		gnome_mdi_app::shortcut_BondType, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::viewscreate_submenu[] =
{
	GNOMEUIINFO_ITEM("New Camera", "create a new view with a new camera...",
		gnome_graphics_view::popup_ViewsNewCam, NULL),
		
	GNOMEUIINFO_ITEM("New View", "create a new view using the current camera...",
		gnome_graphics_view::popup_ViewsNewView, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::views_submenu[] =
{
	GNOMEUIINFO_ITEM("Attach/Detach This View", "attach/detach this view...",
		gnome_graphics_view::popup_ViewsAttachDetach, NULL),
		
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_SUBTREE("_Create", viewscreate_submenu),
	
	GNOMEUIINFO_ITEM("Delete This View", "remove this view...",
		gnome_graphics_view::popup_ViewsDeleteView, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::lights_submenu[] =
{
	GNOMEUIINFO_ITEM("Create New Light", "create a new light object...",
		gnome_graphics_view::popup_LightsNewLight, NULL),
		
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_TOGGLEITEM("Display Local Lights", "loc",
		gnome_graphics_view::popup_LightsSwitchLoc, NULL),
		
	GNOMEUIINFO_TOGGLEITEM("Display Global Lights", "glob",
		gnome_graphics_view::popup_LightsSwitchGlob, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::rendermode_style_radio[] =
{
	GNOMEUIINFO_ITEM("Ball & Stick", "bs",
		popup_RModeBallAndStick, NULL),
		
	GNOMEUIINFO_ITEM("Wireframe", "wf",
		popup_RModeWireFrame, NULL),
		
	GNOMEUIINFO_ITEM("van der Waals", "vdw",
		popup_RModeVanDerWaals, NULL),
		
	GNOMEUIINFO_ITEM("Cylinders", "cyl",
		popup_RModeCylinders, NULL),
		
	GNOMEUIINFO_ITEM("Nothing", "null",
		popup_RModeNothing, NULL),

	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::rendermode_label_radio[] =
{
	GNOMEUIINFO_ITEM("Label Nothing", "lab-null",
		popup_LModeNothing, NULL),
		
	GNOMEUIINFO_ITEM("Label Index", "lab-ind",
		popup_LModeIndex, NULL),
		
	GNOMEUIINFO_ITEM("Label Charge", "lab-ch",
		popup_LModeCharge, NULL),
		
	GNOMEUIINFO_ITEM("Label Element", "lab-el",
		popup_LModeElement, NULL),
		
	GNOMEUIINFO_ITEM("Label AtomType", "lab-at",
		popup_LModeAtomType, NULL),
		
	GNOMEUIINFO_ITEM("Label BondType", "lab-bt",
		popup_LModeBondType, NULL),
	
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::rendermode_color_radio[] =
{
	GNOMEUIINFO_ITEM("Color Elements", "color-el",
		popup_CModeElement, NULL),

	GNOMEUIINFO_ITEM("Color State", "color-st",
		popup_CModeState, NULL),

	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::rendermode_submenu[] =
{
        GNOMEUIINFO_RADIOLIST(rendermode_style_radio),
	GNOMEUIINFO_SEPARATOR,
        GNOMEUIINFO_RADIOLIST(rendermode_label_radio),
	GNOMEUIINFO_SEPARATOR,
        GNOMEUIINFO_RADIOLIST(rendermode_color_radio),
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::projection_radio[] =
{
	GNOMEUIINFO_ITEM("Perspective", "persp",
		gnome_graphics_view::popup_ProjPerspective, NULL),
		
	GNOMEUIINFO_ITEM("Orthographic", "ortho",
		gnome_graphics_view::popup_ProjOrthographic, NULL),
		
	GNOMEUIINFO_ITEM("Red-Green Stereo", "redgre",
		gnome_graphics_view::popup_ProjRedGreen, NULL),
		
	GNOMEUIINFO_ITEM("Relaxed-Eye Stereo", "releye",
		gnome_graphics_view::popup_ProjRelaxedEye, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::projection_submenu[] =
{
        GNOMEUIINFO_RADIOLIST(projection_radio),
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::render_submenu[] =
{
	GNOMEUIINFO_SUBTREE("_Views", views_submenu),
	GNOMEUIINFO_SUBTREE("_Lights", lights_submenu),
	GNOMEUIINFO_SUBTREE("_Rendering Mode", rendermode_submenu),
	GNOMEUIINFO_SUBTREE("_Projection", projection_submenu),
	
	GNOMEUIINFO_TOGGLEITEM("Quick Update Mode", "quick update...",
		gnome_graphics_view::popup_RenderQuickUpdate, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::objects_submenu[] =
{
	GNOMEUIINFO_ITEM("Ribbon", "...",
		popup_ObjRibbon, NULL),
		
	GNOMEUIINFO_ITEM("ESP-plane", "...",
		popup_ObjEPlane, NULL),
		
	GNOMEUIINFO_ITEM("volume-rendered ESP", "...",
		popup_ObjEVolume, NULL),
		
	GNOMEUIINFO_ITEM("ESP-surface", "...",
		popup_ObjESurface, NULL),
		
	GNOMEUIINFO_ITEM("ESP-colored vdW-surface", "...",
		popup_ObjVDWSurface, NULL),
		
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_ITEM("Delete Current Object", "delete",
		gnome_graphics_view::popup_ObjectsDeleteCurrent, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::compute_submenu[] =
{
	GNOMEUIINFO_ITEM("Setup...", "...",
		popup_CompSetup, NULL),
		
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_ITEM("Energy", "...",
		popup_CompEnergy, NULL),
		
	GNOMEUIINFO_ITEM("Geometry Optimization...", "...",
		popup_CompGeomOpt, NULL),
		
	GNOMEUIINFO_ITEM("Molecular Dynamics...", "...",
		popup_CompMolDyn, NULL),
		
	GNOMEUIINFO_ITEM("Random Conformational Search...", "...",
		popup_CompRandomSearch, NULL),
	
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_ITEM("Plot Energy vs. 1 Torsion Angle...", "...",
		popup_CompEPlot1D, NULL),
	
	GNOMEUIINFO_ITEM("Plot Energy vs. 2 Torsion Angles...", "...",
		popup_CompEPlot2D, NULL),
	
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_ITEM("Formula", "...",
		popup_CompFormula, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::hydrogens_submenu[] =
{
	GNOMEUIINFO_ITEM("Add", "...",
		popup_HAdd, NULL),
		
	GNOMEUIINFO_ITEM("Remove", "...",
		popup_HRemove, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::library_submenu[] =
{
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::build_submenu[] =
{
	GNOMEUIINFO_SUBTREE("_Hydrogens", hydrogens_submenu),
	
	GNOMEUIINFO_SUBTREE("Library", library_submenu),

	GNOMEUIINFO_ITEM("Sequence Builder...", "...",
		popup_Builder, NULL),
		
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_ITEM("Center", "...",
		popup_Center, NULL),
		
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_ITEM("Zap All", "...",
		popup_Clear, NULL),
		
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::popupmenu_info[] =
{
	GNOMEUIINFO_SUBTREE("_File", file_submenu),
	GNOMEUIINFO_SUBTREE("_Select", select_submenu),
	
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_SUBTREE("_Settings...", settings_submenu),
	
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_SUBTREE("_Render", render_submenu),
	GNOMEUIINFO_SUBTREE("_Objects", objects_submenu),
	
	GNOMEUIINFO_SEPARATOR,
	
	GNOMEUIINFO_SUBTREE("_Compute", compute_submenu),
	
	GNOMEUIINFO_ITEM("MD Trajectory Viewer...", "...",
		popup_TrajView, NULL),
	
	GNOMEUIINFO_SEPARATOR,

	GNOMEUIINFO_SUBTREE("_Build", build_submenu),
	
	GNOMEUIINFO_ITEM("Convert to QM project", "Convert this project into a qm1gp project.",
		popup_ConvertQM, NULL),
	
	GNOMEUIINFO_END
};

GnomeUIInfo gnome_mm1_docv::popupmenu_dummy[] =
{
        GNOMEUIINFO_SUBTREE("Ghemical", popupmenu_info),
        GNOMEUIINFO_END
};

/*################################################################################################*/

mm1_geomopt_param * gnome_mm1_docv::goparam = NULL;
mm1_moldyn_param * gnome_mm1_docv::mdparam = NULL;

gnome_mm1_docv::gnome_mm1_docv(ostream * p1, gnome_class_factory & p2) :
	gnome_docv(p1, p2), mm1_docv(p1, p2), docview(p1, p2), model_simple(p1, p2)
{
  // new (with tearoff):
  GtkMenuItem *mi;
//  gnome_popup_menu_new(popupmenu_dummy);
//  mi = GTK_MENU_ITEM(popupmenu_dummy[0].widget);
//  popupmenu = GTK_WIDGET(mi->submenu);

  popupmenu = gnome_popup_menu_new(popupmenu_info);	// no tearoff...
  
  // create builder "Library" submenu
  mi = GTK_MENU_ITEM(build_submenu[1].widget);
  
  struct stat stat_buf;
  char buffer [1024];
  ostrstream str(buffer, sizeof(buffer));
  str << PRIMARY_PATH << "library" << DIR_SEPARATOR << ends;
  if ( stat(buffer, &stat_buf) == 0 && S_ISDIR(stat_buf.st_mode) )
    MakeLibraryWidgets(mi,
		     model_prefs->Find("MM1Interface/LibraryDir", buffer),
		     model_prefs->Value("MM1Interface/LibraryLevels", 1));
  else
    MakeLibraryWidgets(mi,
		       model_prefs->Find("MM1Interface/LibraryDir", "library/"),
		       model_prefs->Value("MM1Interface/LibraryLevels", 1));
}

gnome_mm1_docv::~gnome_mm1_docv(void)
{
}

/*##############################################*/
/*##############################################*/

// the mm1-dependent popup-menu callbacks are here...
// the mm1-dependent popup-menu callbacks are here...
// the mm1-dependent popup-menu callbacks are here...

void gnome_mm1_docv::popup_FileOpen(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv) new mm1_file_open_dialog(docv);	// memory leak???
}

void gnome_mm1_docv::popup_FileImport(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	glade_modal_dialog_owner = docv;	// requires modal dialogs!!!
	
	GtkWidget * dialog = create_f1_dialog_mm1_import();
	gtk_widget_show(dialog);
}

void gnome_mm1_docv::popup_FileExport(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	glade_modal_dialog_owner = docv;	// requires modal dialogs!!!
	
	GtkWidget * dialog = create_f2_dialog_mm1_export();
	gtk_widget_show(dialog);
}

void gnome_mm1_docv::popup_FileExportGraphic(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv) new mm1_file_save_graphics_dialog(docv);	// memory leak???
}

void gnome_mm1_docv::popup_FileSaveAs(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv) new mm1_file_save_dialog(docv);	// memory leak???
}

void gnome_mm1_docv::popup_RModeWireFrame(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->render = RENDER_WIREFRAME; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_RModeBallAndStick(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->render = RENDER_BALL_AND_STICK; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_RModeVanDerWaals(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->render = RENDER_VAN_DER_WAALS; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_RModeCylinders(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->render = RENDER_CYLINDERS; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_RModeNothing(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->render = RENDER_NOTHING; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_CModeState(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->colormode = & mm1_docv::cm_state; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_CModeElement(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->colormode = & mm1_docv::cm_element; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_LModeNothing(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->label = LABEL_NOTHING; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_LModeIndex(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->label = LABEL_INDEX; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_LModeCharge(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->label = LABEL_CHARGE; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_LModeElement(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->label = LABEL_ELEMENT; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_LModeAtomType(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->label = LABEL_ATOMTYPE; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_LModeBondType(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gv->label = LABEL_BONDTYPE; gv->docv->UpdateAllGraphicsViews();
}

void gnome_mm1_docv::popup_ObjRibbon(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		mm1_mdl::amino_builder.Identify(docv);
		DefineSecondaryStructure(docv);
		
		vector<mm1_chn_info> & ci_vector = (* docv->ref_civ);
		for (i32u n1 = 0;n1 < ci_vector.size();n1++)
		{
			if (ci_vector[n1].length < 3) continue;
			else docv->AddObject(new mm1_ribbon(docv, gv->colormode, n1, 4));	// min. order is 2!!!
		}
		
		docv->UpdateAllGraphicsViews();
	}
}

// planes/surfaces/etc... are done here, in a simple and straightforward way.
// just equivalent stuff is repeated at TARGET2. how to improve this???

// an advanced dialog box would be needed here, to replace the text stuff.
// the dialog should be flexible enough, so it could be used in all models, not just here...

// does it make any sense to write DoPlane, DoSurface, etc???

void gnome_mm1_docv::popup_ObjEPlane(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		fGL tmp1;
		cp_param cpp; cpp.np = docv->model_prefs->Value("MM1Graphics/PlaneResolution", 30); 
		cpp.docv = docv; cpp.ref = (mm1_docv *) docv;
		
		cout << "give dimension of the plane [nm] (1.0 might be ok) ??? "; cin >> tmp1;
		
		cpp.dim = tmp1; cpp.transparent = true;
		cpp.vf = (ValueFunction *) mm1_GetESPValue; cpp.cf = (ColorFunction *) GetRBRange1;
		cpp.value = 138.9354518 * 1.0; cpp.alpha = 0.75;
		
		docv->AddObject(new color_plane_object(ol_static(), cpp, "ESP-"));
		docv->UpdateAllGraphicsViews();
	}
}

void gnome_mm1_docv::popup_ObjEVolume(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		fGL tmp1;
		cp_param cpp; cpp.np = docv->model_prefs->Value("MM1Graphics/SurfaceResolution", 30);
		cpp.docv = docv; cpp.ref = (mm1_docv *) docv;
		
		cout << "give dimension of the volume [nm] (1.0 might be ok) ??? "; cin >> tmp1;
		
		cpp.dim = tmp1; cpp.transparent = true;
		cpp.vf = (ValueFunction *) mm1_GetESPValue; cpp.cf = (ColorFunction *) GetRBRange2;
		cpp.value = 138.9354518 * 1.0; cpp.alpha = 0.25;
		
		docv->AddObject(new volume_rendering_object(ol_static(), cpp, 20, tmp1 / 2.0, (* gv->cam), "ESP-"));
		docv->UpdateAllGraphicsViews();
	}
}

void gnome_mm1_docv::popup_ObjESurface(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		fGL tmp1; fGL tmp2; char input;
		
		i32s size = docv->model_prefs->Value("MM1Graphics/SurfaceResolution", 50);
		i32s sz[3] = { size, size, size };
		fGL dm[3] = { 0.55, 0.50, 0.45 };
		
		cs_param csp2a; cs_param csp2b;
		csp2a.np = sz; csp2a.dim = dm; csp2a.ostr = & cout;
		csp2a.docv = docv; csp2a.ref = (mm1_docv *) docv; csp2a.next = & csp2b;
		csp2b.np = sz; csp2b.dim = dm; csp2b.ostr = & cout;
		csp2b.docv = docv; csp2b.ref = (mm1_docv *) docv; csp2b.next = NULL;
		
		tmp1 = 138.9354518 / 4.0;
		cout << "give dimension of the surface [nm] (2.5 might be ok) ??? ";
		cin >> tmp2; dm[0] = dm[1] = dm[2] = tmp2;
		
		cout << "choose the type of surface : solid or wireframe ? (s/w) ";
		cin >> input; csp2a.wireframe = csp2b.wireframe = (input == 'w' || input == 'W');
		
		csp2a.transparent = !true;
		csp2a.vf1 = (ValueFunction *) mm1_GetESPValue; csp2a.vf2 = (ValueFunction *) GetUnity;
		csp2a.cvalue = 1.0; csp2a.alpha = 0.25; csp2a.toler = 1.0e-6 * tmp1; csp2a.maxc = 500;
		csp2a.cf = (ColorFunction *) GetRedColor; csp2a.svalue = +tmp1;
		
		csp2b.transparent = !true;
		csp2b.vf1 = (ValueFunction *) mm1_GetESPValue; csp2b.vf2 = (ValueFunction *) GetUnity;
		csp2b.cvalue = 1.0; csp2b.alpha = 0.25; csp2b.toler = 1.0e-6 * tmp1; csp2b.maxc = 500;
		csp2b.cf = (ColorFunction *) GetBlueColor; csp2b.svalue = -tmp1;
		
		docv->AddObject(new color_surface_object(ol_static(), csp2a, "ESP-"));
		docv->UpdateAllGraphicsViews();
	}
}

void gnome_mm1_docv::popup_ObjVDWSurface(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		fGL tmp1; fGL tmp2; char input;
		
		i32s size = docv->model_prefs->Value("MM1Graphics/SurfaceResolution", 50);
		i32s sz[3] = { size, size, size };
		fGL dm[3] = { 0.55, 0.50, 0.45 };
		
		cs_param csp1;
		csp1.np = sz; csp1.dim = dm; csp1.ostr = & cout;
		csp1.docv = docv; csp1.ref = (mm1_docv *) docv; csp1.next = NULL;
		
		tmp1 = 138.9354518 / 2.0;
		cout << "give dimension of the surface [nm] (2.5 might be ok) ??? ";
		cin >> tmp2; dm[0] = dm[1] = dm[2] = tmp2;
		
		cout << "choose the type of surface : solid or wireframe ? (s/w) ";
		cin >> input; csp1.wireframe = (input == 'w' || input == 'W');
		
		csp1.transparent = true; csp1.cf = (ColorFunction *) GetRBRange1;
		csp1.vf1 = (ValueFunction *) mm1_GetVDWSValue; csp1.vf2 = (ValueFunction *) mm1_GetESPValue;
		csp1.svalue = 1.0; csp1.cvalue = tmp1; csp1.alpha = 0.50; csp1.toler = 1.0e-6 * tmp1; csp1.maxc = 500;
		
		docv->AddObject(new color_surface_object(ol_static(), csp1, "ESP-colored VDW-"));
		docv->UpdateAllGraphicsViews();
	}
}

void gnome_mm1_docv::popup_CompSetup(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	glade_modal_dialog_owner = docv;	// requires modal dialogs!!!
	
	GtkWidget * dialog = create_e2_dialog_mm1_setup();
	gtk_widget_show(dialog);
}

void gnome_mm1_docv::popup_CompEnergy(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv) docv->DoEnergy();
}

void gnome_mm1_docv::popup_CompGeomOpt(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		mm1_geomopt_param param;
		param.treshold_nsteps = 500;
		param.treshold_grad = 1.0e-3;
		param.treshold_delta_e = 1.0e-7;

		docv->DoGeomOpt(param);
	}
}

void gnome_mm1_docv::popup_CompMolDyn(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		mm1_moldyn_param param;
		param.nsteps = 10000;
		param.temperature = 300.0;
		param.timestep = 0.5;
		
		docv->DoMolDyn(param);
	}
}

void gnome_mm1_docv::popup_CompRandomSearch(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		docv->DoRandomSearch();
	}
}

void gnome_mm1_docv::popup_CompEPlot1D(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv) docv->DoEnergyPlot1D();
}

void gnome_mm1_docv::popup_CompEPlot2D(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv) docv->DoEnergyPlot2D();
}

void gnome_mm1_docv::popup_CompFormula(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv) docv->DoFormula();
}

void gnome_mm1_docv::popup_TrajView(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (!docv->GetTrajectoryFile())
	{
		if (docv) new mm1_trajfile_dialog(docv);	// memory leak???
	}
	else docv->err->ErrorMessage("Trajectory already open?!?!?!");
}

void gnome_mm1_docv::popup_HAdd(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		docv->AddHydrogens();
		docv->UpdateAllGraphicsViews();
	}
}

void gnome_mm1_docv::popup_HRemove(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		docv->RemoveHydrogens();
		docv->UpdateAllGraphicsViews();
	}
}

void gnome_mm1_docv::popup_Clear(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv && 
	    docv->err->Question("Are you sure you want to clear everything?"))
	{
	  docv->Clear();
	  docv->UpdateAllGraphicsViews();
	}
}

void gnome_mm1_docv::popup_Library(GtkWidget *item, gpointer data)
{
  docview * docv1 = (docview *) gtk_object_get_user_data(GTK_OBJECT(item));
  gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(docv1);

  static char buffer[256];
  g_snprintf(buffer, 256, "%s", data);
cout << "preparing to open a library file : " << buffer << endl;	// just for debugging...

  bool open = docv->IsEmpty();	// open, not insert
  
  ifstream ifile;
  ifile.open(buffer, ios::in);
  docv->ReadStream(ifile, true);
  
  // This should probably go into the docview class 
  // that is, the docview::ReadStream() should override 
  // the mm1_mdl::ReadStream() and the rest
  	
  if (!open)	// we append this file to the current project
    {
      if (docv->selected_object != NULL)  // if an object was selected,
	{				  // remove that selection and leave
	  docv->selected_object = NULL;	 // the newly added atoms as selection;
	  docv->event_SelectedObjectChanged();	// now the mouse tools also affect the
	}				 // newly added atoms!
    }
	
  ifile.close();
}

void gnome_mm1_docv::popup_Builder(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		char seq[1024]; char input;
		
		f64 aab_ah[] = { 302.0 * M_PI / 180.0, 313.0 * M_PI / 180.0, M_PI };
		f64 aab_bs[] = { 180.0 * M_PI / 180.0, 180.0 * M_PI / 180.0, M_PI };
		
		f64 nab[] =
		{
			44.6 * M_PI / 180.0, 261.0 * M_PI / 180.0, 320.8 * M_PI / 180.0, 208.6 * M_PI / 180.0,
			273.8 * M_PI / 180.0, 105.6 * M_PI / 180.0, 356.0 * M_PI / 180.0, 24.7 * M_PI / 180.0,
			88.7 * M_PI / 180.0, 264.6 * M_PI / 180.0
		};
		
		cout << "build amino/nucleic acid sequences ? (a/n) "; cin >> input;
		
		if (input == 'a')
		{
			cout << "give the sequence using symbols ARNDCQEGHILKMFPSTWYV ???"; cin >> seq;
			cout << "please select the conformation : H = helix, S = sheet ???"; cin >> input;
			mm1_mdl::amino_builder.Build(docv, seq, input == 'h' || input == 'H' ? aab_ah : aab_bs);
			docv->UpdateAllGraphicsViews();
		}
		
		if (input == 'n')
		{
			cout << "give the sequence using symbols AGTC and aguc ???"; cin >> seq;
			mm1_mdl::nucleic_builder.Build(docv, seq, nab);
			docv->UpdateAllGraphicsViews();
		}
	}
}

void gnome_mm1_docv::popup_Center(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		docv->CenterCRDSet(0);
		
		// Which is the current Coord Set?
		// usually it's 0, but it could be any/all. the crdset-system is not yet in GUI...
		
		// 2001-06-18 TH: yes, this is not quite ready. but i guess in the end we will
		// move ALL crd-sets...
		
// sometimes this centering won't work, if the camera won't point to the origo.
// so, here we try to turn it there. ANY EFFECTS TO LIGHTS, ETC??????

	//	for (i32u n1 = 0;n1 < docv->camera_vector.size();n1++)
	//	{
			// rotation or translation???
			// translation would affect less to lights (since it's accurate)????
	//	}
		
		docv->UpdateAllGraphicsViews();
	}
}

void gnome_mm1_docv::popup_ConvertQM(GtkWidget *, gpointer data)
{
	gnome_graphics_view * gv = gnome_graphics_view::GetGV((GtkWidget *) data);
	gnome_mm1_docv * docv = dynamic_cast<gnome_mm1_docv *>(gv->docv);
	if (docv)
	{
		const char message[] = "Should I close this old project?";
		bool close_old = docv->err->Question(message);
		
		gnome_qm1_docv * qm_docv;
		qm_docv = new gnome_qm1_docv(& cout, * gnome_class_factory::GetInstance());
	// keep the same project name only if old was closed??? might be confusing otherwise???
	if (close_old) qm_docv->SetProjectFileName(docv->GetProjectFileName(false));
		qm_docv->InitGnomeMDIChild();
		
// when calling InitGnomeMDIChild(), also a graphics_view and therefore a new camera object were added.
// is order to retain as much grahics settings as possible, copy the current view's camera settings to
// the new project's view. WARNING!!! this is a bit ugly way to do it: in the new project, there is
// only a single view object and a single camera object; so we just pick them! TODO : lights???
qm_docv->camera_vector.front()->CopySettings(gv->cam);
// an initial solution to the lights problem : remove the old light and make a new one using default
// settings. looks good if the user has not modified the lights, but otherwise restores defaults.
qm_docv->RemoveLight(qm_docv->light_vector.front());
qm_docv->AddLocalLight(new directional_light(ol_static(qm_docv->camera_vector.front()->GetLocData())), qm_docv->camera_vector.front());

		// convert mm1gp -> qm1gp using all_atoms_interface...
		
		qm_docv->aai_MakeCopy(docv);
		
		// ...the conversion is now ready.
		
		if (close_old) gnome_mdi_remove_child(gnome_mdi_app::mdi, GNOME_MDI_CHILD(gv->docv->child), TRUE);
	}
}

/*##############################################*/
/*##############################################*/

void gnome_mm1_docv::GeomOptGetParam(mm1_geomopt_param & param)
{
	glade_modal_dialog_owner = this;	// requires modal dialogs!!!
	
	// here we just set "param" to a static variable,
	// where it's then available in glade code...
	
	goparam = & param;
	
	GtkWidget * dialog = create_a1_dialog_geomopt();
	gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
	
	goparam = NULL;
}

#define START 10000	// see mm1docv.cpp!!!!!!!!

void gnome_mm1_docv::MolDynGetParam(mm1_moldyn_param & param)
{
	glade_modal_dialog_owner = this;	// requires modal dialogs!!!
	
	// here we just set "param" to a static variable,
	// where it's then available in glade code...
	
	mdparam = & param;
	
	GtkWidget * dialog = create_a2_dialog_moldyn();
	gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
	
	mdparam = NULL;
}


void gnome_mm1_docv::MakeLibraryWidgets(GtkMenuItem *menu, 
					const char *directory_path, 
					int recursion)
{
  struct stat stat_buf;

  if ( stat(directory_path, &stat_buf) == 0 && S_ISDIR(stat_buf.st_mode) )
    {
      struct dirent *namelist;
      DIR *dirList;
      GtkWidget *library_item;

       if (( dirList = opendir(directory_path) ))
	 {
//	   char *filename = new char[1024];		// OLD
	   while (( namelist = readdir(dirList) ))
	     {
	       char *filename = new char[1024];		// a new string for each item???
	       
	       strncpy(filename, directory_path, 512);
	       strncat(filename, namelist->d_name, 511);
	       
	       if ( namelist->d_name[0] != '.' &&
		    stat(filename, &stat_buf) == 0 &&
		    strstr(namelist->d_name, ".mm1gp") != NULL)
		 {
		   library_item = gtk_menu_item_new_with_label(namelist->d_name);
		   
		   // we pass the string "filename" here as user data.
		   // "filename" must be unique for each item, and it must stay in memory.
		   
		   gtk_signal_connect(GTK_OBJECT(library_item), "activate",
				      GTK_SIGNAL_FUNC(popup_Library),
				      filename);
		   
		   gtk_object_set_user_data(GTK_OBJECT(library_item), 
					    (docview *) this);

		   gtk_widget_show(library_item);

		   gtk_menu_append(GTK_MENU(menu->submenu),
				   library_item);
		 }
	       else if ( namelist->d_name[0] != '.' &&
			 stat(filename, &stat_buf) == 0 &&
			 S_ISDIR(stat_buf.st_mode) &&
			 recursion != 0)
		 {
		   library_item = gtk_menu_item_new_with_label(namelist->d_name);
		   gtk_menu_item_set_submenu(GTK_MENU_ITEM(library_item),
					     gtk_menu_new());

		   gtk_widget_show(library_item);
		   
		   gtk_menu_append(GTK_MENU(menu->submenu),
				   library_item);
		   
		   strcat(filename, "/");
		   MakeLibraryWidgets(GTK_MENU_ITEM(library_item), filename, recursion-1);
		 }
	// this would be the correct place to delete the string, if it were temporary.
	// however, we must retain the strings in memory because we will use them later, so this will create a (small) memory leak.
	// we should either store the strings to a static block of memory, or someway arrange the memory release when program stops.
//	   delete [] filename;		// can't do this, or we might use freed (and possibly overwritten) memory later...
	     }
//	   delete [] filename;		// OLD
	   closedir(dirList);
	 }

     }
}

/*################################################################################################*/

// eof
