// INTERFACE.H : here we define some common interfaces between different models.

// Copyright (C) 2001 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef INTERFACE_H
#define INTERFACE_H

class all_atoms_interface;
class trajectory_interface;

#include "typedef.h"
#include "utility.h"

#include <fstream>
using namespace std;

////////////////////////////////////////////////////////////////////////////////////////////////////

/**	"##all_atoms_interface" tries to build a bridge between MM and QM models. 
	It offers simple methods to access atoms and bonds regardless of the model type. 
	Also the contents of an object can be copied into another object -> conversions!
	
	The mm1_mdl and qm1_mdl classes implement all_atoms_interface.
*/

// many const declarations are commented out here; g++ doesn't accept inline functions as const???
// many const declarations are commented out here; g++ doesn't accept inline functions as const???
// many const declarations are commented out here; g++ doesn't accept inline functions as const???

class all_atoms_interface
{
	protected:
	
// this is just an interface definition -> there is no relevant common data?!?!?
// this is just an interface definition -> there is no relevant common data?!?!?
// this is just an interface definition -> there is no relevant common data?!?!?

	public:
	
	all_atoms_interface(void);
	virtual ~all_atoms_interface(void);
	
	// a simple index-based access to add/remove atom/bond tasks...
	
	virtual i32u aai_GetAtomCount(void) /*const*/ = 0;
	virtual i32u aai_GetBondCount(void) /*const*/ = 0;
	
/**	aai_FindAtomByInd() will convert the atom index (which is >= 0 but < aai_GetAtomCount()) to 
	the atom pointer. If the atom is not found, the return value is NULL.
*/
	virtual void * aai_FindAtomByInd(i32s) /*const*/ = 0;
	
/**	aai_FindAtomByPtr() will convert the atom pointer returned by aai_AddAtom() to the atom index. 
	If the atom is not found, the return value is -1.
*/
	virtual i32s aai_FindAtomByPtr(void *) /*const*/ = 0;
	
/**	aai_FindBond() will search for a bond using atom indices (obtained from aai_FindAtomByInd() 
	or aai_AddAtom()). If the bond is not found, the return value is -1.
*/
	virtual i32s aai_FindBond(i32s, i32s) /*const*/ = 0;
	
/**	aai_AddAtom() will add a new atom to the model. 
	The parameters are the element and the coordinates of the atom to be added. 
	The return value is a pointer of void type that works as an unique identifier of the atom
	that is guaranteed to be valid as long as the atom is removed from the model. NOTE: if an
	atom is removed, and after that an another is added, it is *possible* that the new atom
	has the same pointer as the old one had... so in this sense the pointers are NOT unique!
	
	Why all this? Because the atom indexing will change, if some atoms are removed from the model. 
	The atom indexes are used, on the other hand, for example to add bonds. To get the atom index,
	either use aai_FindAtom() or, if you just added the atom in question, aai_GetAtomCount().
	
	In practice, the atoms and bonds are stored to STL lists (see mm1_mdl and qm1_mdl).
*/
	virtual void * aai_AddAtom(element, fGL *) = 0;
	
/**	aai_RemoveAtom() will remove an atom using the given atom index. 
	The return value indicates if the operation was succesful.
*/
	virtual bool aai_RemoveAtom(i32s) = 0;
	
/**	aai_AddBond() will add a bond using the given atom indices and bondtype. 
	The return value indicates if the operation was succesful.
*/
	virtual bool aai_AddBond(i32s, i32s, bondtype) = 0;
	
/**	aai_RemoveBond() will remove a bond using the given bond index. 
	The return value indicates if the operation was succesful.
*/
	virtual bool aai_RemoveBond(i32s) = 0;
	
	virtual bool aai_AtomGetElement(i32s, element &) /*const*/ = 0;
	virtual bool aai_AtomSetElement(i32s, element) = 0;
	
	virtual bool aai_AtomGetXYZ(i32s, fGL *) /*const*/ = 0;
	virtual bool aai_AtomSetXYZ(i32s, const fGL *) = 0;
	
	virtual bool aai_BondGetBondType(i32s, bondtype &) /*const*/ = 0;
	virtual bool aai_BondSetBondType(i32s, bondtype) = 0;
	
	virtual bool aai_BondGetAtomIndices(i32s, i32s *) /*const*/ = 0;
	
// how to handle the coordinate sets??? should there be a "current coordinate set" or what?
// how to handle the coordinate sets??? should there be a "current coordinate set" or what?
// how to handle the coordinate sets??? should there be a "current coordinate set" or what?

/**	aai_MakeCopy() will copy the contents (atoms and bonds) of an another all_atoms_interface 
	object to this object. This allows model conversions MM <-> QM.
*/
	void aai_MakeCopy(const all_atoms_interface *);
	
// what else could be added here??? the file_trans OELib/OpenBabel compatibility methods ?!?!?!
// or don't bring file_trans methods here, but convert them to work through all_atoms_interface???

};

////////////////////////////////////////////////////////////////////////////////////////////////////

class trajectory_interface
{
	protected:
	
	ifstream * trajfile;
	
	i32s total_traj_frames;
	i32s current_traj_frame;
	
	public:
	
	trajectory_interface(void);
	virtual ~trajectory_interface(void);
	
	i32s GetTotalFrames(void);
	ifstream * GetTrajectoryFile(void);
	
	i32s GetCurrentFrame(void);
	void SetCurrentFrame(i32s);
	
	virtual void OpenTrajectory(const char *) = 0;
	virtual void CloseTrajectory(void) = 0;
	
	virtual void ReadFrame(void) = 0;
};

////////////////////////////////////////////////////////////////////////////////////////////////////

#endif	// INTERFACE_H

// eof
