{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2001  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to table 6.10 (Lighting (cont.)) of the OpenGL 1.2.1 specs.
-}

module GL_QueryLighting2 (
   VarLightColor(..),
   VarPosition(..),
   VarAttenuation(..),
   VarSpotDirection(..),
   VarSpotExponent(..),
   VarSpotCutoff(..),
   VarColorIndexes(..)
) where

import GL_Constants     ( gl_POSITION, gl_SPOT_DIRECTION, gl_SPOT_EXPONENT,
                          gl_SPOT_CUTOFF, gl_COLOR_INDEXES )
import GL_BasicTypes    ( GLint, GLfloat, Gettable(..), Capability(..) )
import GL_Colors        ( Light, Face, marshalFace, LightType, marshalLightType,
                          Attenuation, marshalAttenuation )
import GL_QueryUtils    ( getLightColor4f, getLightPosition4f, getLightf,
                          getLightNormal3f, getMaterialColorIndex3i )
import GL_VertexSpec    ( Vertex4, Normal3, Color4, ColorIndex )

---------------------------------------------------------------------------

data VarLightColor = VarLightColor Light LightType deriving (Eq,Ord)

instance Gettable VarLightColor (Color4 GLfloat) where
   get (VarLightColor l t) = getLightColor4f (marshalCapability l) (marshalLightType t)

data VarPosition = VarPosition Light deriving (Eq,Ord)

instance Gettable VarPosition (Vertex4 GLfloat) where
   get (VarPosition l) = getLightPosition4f (marshalCapability l) gl_POSITION

data VarAttenuation = VarAttenuation Light Attenuation deriving (Eq,Ord)

instance Gettable VarAttenuation GLfloat where
   get (VarAttenuation l a) = getLightf (marshalCapability l) (marshalAttenuation a)

data VarSpotDirection = VarSpotDirection Light deriving (Eq,Ord)

instance Gettable VarSpotDirection (Normal3 GLfloat) where
   get (VarSpotDirection l) = getLightNormal3f (marshalCapability l) gl_SPOT_DIRECTION

data VarSpotExponent = VarSpotExponent Light deriving (Eq,Ord)

instance Gettable VarSpotExponent GLfloat where
   get (VarSpotExponent l) = getLightf (marshalCapability l) gl_SPOT_EXPONENT

data VarSpotCutoff = VarSpotCutoff Light deriving (Eq,Ord)

instance Gettable VarSpotCutoff (Maybe GLfloat) where
   get (VarSpotCutoff l) = do c <- getLightf (marshalCapability l) gl_SPOT_CUTOFF
                              return $ if c == 180 then Nothing else Just c

data VarColorIndexes = VarColorIndexes Face

instance Gettable VarColorIndexes (ColorIndex GLint, ColorIndex GLint, ColorIndex GLint) where
   get (VarColorIndexes f) = getMaterialColorIndex3i (marshalFace f) gl_COLOR_INDEXES
