/*
** 2000-02-12 -	A Join command, to counter the long-existing (um, but still incomplete)
**		Split command. Features neat drag-and-drop support for ordering the files
**		to be joined.
*/

#include "gentoo.h"

#include <ctype.h>
#include <fcntl.h>

#include "cmd_delete.h"
#include "convutil.h"
#include "dialog.h"
#include "dirpane.h"
#include "errors.h"
#include "fileutil.h"
#include "overwrite.h"
#include "progress.h"
#include "sizeutil.h"

#include "cmd_join.h"

#define	CMD_ID	"join"

/* ----------------------------------------------------------------------------------------- */

/* 2000-02-12 -	Do the actual joining of the selected files. Returns TRUE on success. */
static gboolean do_join(MainInfo *min, DirPane *src, DirPane *dst, GtkListStore *store, GtkEntry *entry, gsize total_size)
{
	gchar		buf[PATH_MAX], *file;
	const gchar	*str;
	gsize		done_size = 0U;
	OvwRes		owres;
	gboolean	doit = TRUE;
	Conv		conv;

	/* Protect against empty destination name. */
	if((str = gtk_entry_get_text(entry)) == NULL || *str == '\0')
	{
		err_set(min, EINVAL, CMD_ID, NULL);
		return FALSE;
	}

	file = conv_begin(&conv, str);

	ovw_overwrite_begin(min, _("\"%s\" Already Exists - Continue With Join?"), 0U);
	g_snprintf(buf, sizeof buf, "%s/%s", dst->dir.path, file);
	owres = ovw_overwrite_file(min, buf, NULL);
	switch(owres)
	{
	case OVW_SKIP:
	case OVW_CANCEL:
		doit = FALSE;
		break;
	case OVW_PROCEED:
		break;
	case OVW_PROCEED_FILE:
		doit = del_delete_file(min, buf);
		break;
	case OVW_PROCEED_DIR:
		doit = del_delete_dir(min, buf, FALSE);
		break;
	}
	if(doit)
	{
		gint	fout;

		if((fout = open(buf, O_CREAT | O_WRONLY | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) >= 0)
		{
			GtkTreeIter	iter;
			gint		fin;
			DirRow		*dr;
			gsize		put;

			pgs_progress_begin(min, _("Joining..."), PFLG_BYTE_VISIBLE);
			gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);
			while(gtk_list_store_iter_is_valid(store, &iter))
			{
				gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, 2, &dr, -1);
				if((fin = open(dp_full_name(src, DP_ROW_INDEX(src, dr)), O_RDONLY)) >= 0)
				{
					pgs_progress_item_begin(min, DP_ROW_NAME(dr), DP_ROW_STAT(dr).st_size);
					put = fut_copy(fin, fout, 256 << 10);
					pgs_progress_item_end(min);
					if(close(fin) != 0)
					{
						err_set(min, errno, CMD_ID, DP_ROW_NAME(dr));
						break;
					}
					if(put != DP_ROW_STAT(dr).st_size)
					{
						err_set(min, 000, CMD_ID, DP_ROW_NAME(dr));
						break;
					}
					done_size += put;
					dp_unselect(src, DP_ROW_INDEX(src, dr));
				}
				else
				{
					err_set(min, errno, CMD_ID, DP_ROW_NAME(dr));
					break;
				}
				if(!gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter))
					break;
			}
			pgs_progress_end(min);
			close(fout);
			if(done_size == total_size)
				dp_rescan_post_cmd(dst);
			else
				unlink(buf);
		}
		else
			err_set(min, errno, CMD_ID, buf);
	}
	ovw_overwrite_end(min);
	conv_end(&conv);

	return doit ? done_size == total_size : TRUE;
}

gint cmd_join(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	GSList	*sel;
	gint	ret = 1;

	err_clear(min);

	if((sel = dp_get_selection(src)) != NULL)
	{
		GSList	*iter;
		gsize	tot = 0, num = 0;

		/* Compute size of all regular files in the selection. */
		for(iter = sel; iter != NULL; iter = g_slist_next(iter))
		{
			if(S_ISREG(DP_ROW_STAT((DirRow *) iter->data).st_mode))
			{
				tot += DP_ROW_STAT((DirRow *) iter->data).st_size;
				num++;
			}
		}
		
		/* Only open the dialog and proceed with command if there were, in
		** fact, more than one regular file selected.
		*/
		if(num > 1)
		{
			Dialog			*dlg;
			GtkWidget		*scwin, *vbox, *label, *entry;
			GtkListStore		*store;
			GtkWidget		*view;
			GtkCellRenderer		*cr, *sr;
			GtkTreeViewColumn	*vc;
			GtkTreeSelection	*treesel;
			gchar			buf[FILENAME_MAX + 64];
			const gchar		*first = NULL;
			gint			len;

			vbox  = gtk_vbox_new(FALSE, 0);
			label = gtk_label_new(_("Click and Drag Files to Reorder, Then Click Join."));
			gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

			if(tot > (1 << 10))
			{
				gchar	sbuf[16];

				sze_put_size(sbuf, sizeof sbuf, tot, SZE_AUTO);
				g_snprintf(buf, sizeof buf, _("The total size is %s (%lu bytes)."), sbuf, (unsigned long) tot);
			}
			else
				g_snprintf(buf, sizeof buf, _("The total size is %lu bytes."), (unsigned long) tot);
			label = gtk_label_new(buf);
			gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

			scwin = gtk_scrolled_window_new(NULL, NULL);
			gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

			/* Create and initialize list store with selected files and their sizes. */
			store = gtk_list_store_new(3, G_TYPE_STRING, G_TYPE_UINT64, G_TYPE_POINTER);
			for(iter = sel; iter != NULL; iter = g_slist_next(iter))
			{
				if(S_ISREG(DP_ROW_STAT((DirRow *) iter->data).st_mode))
				{
					GtkTreeIter	ti;

					gtk_list_store_append(store, &ti);
					gtk_list_store_set(store, &ti, 0, DP_ROW_NAME((DirRow *) iter->data),
								       1, DP_ROW_STAT((DirRow *) iter->data).st_size,
								       2, iter->data,
								       -1);
					if(first == NULL)
						first = DP_ROW_NAME((DirRow *) iter->data);
				}
			}

			view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
			cr = gtk_cell_renderer_text_new();
			vc = gtk_tree_view_column_new_with_attributes("(title)", cr, "text", 0, NULL);
			gtk_tree_view_append_column(GTK_TREE_VIEW(view), vc);
			sr = gtk_cell_renderer_text_new();
			g_object_set(G_OBJECT(sr), "xalign", 1.0f, NULL);
			vc = gtk_tree_view_column_new_with_attributes("(title)", sr, "text", 1, NULL);
			gtk_tree_view_append_column(GTK_TREE_VIEW(view), vc);
			gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), FALSE);

			treesel = gtk_tree_view_get_selection(GTK_TREE_VIEW(view));
			gtk_tree_selection_set_mode(treesel, GTK_SELECTION_NONE);
			gtk_tree_view_set_reorderable(GTK_TREE_VIEW(view), TRUE);
			gtk_widget_set_size_request(view, 0, 256);	/* Get a sensible minmum height. */
			gtk_container_add(GTK_CONTAINER(scwin), view);
			gtk_box_pack_start(GTK_BOX(vbox), scwin, TRUE, TRUE, 0);

			label = gtk_label_new(_("Enter Destination File Name"));
			gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 5);
			entry = gtk_entry_new_with_max_length(FILENAME_MAX - 1);
			/* Heuristic: remove trailing hexadecimal number and period, and position cursor before last remaining period. */
			if((len = g_snprintf(buf, sizeof buf, "%s", first)) > 0)
			{
				const gchar	*dot;
				int	i;

				for(i = len - 1; i > 0 && (isxdigit(buf[i]) || buf[i] == '.'); i--)
					buf[i] = '\0';
				gtk_entry_set_text(GTK_ENTRY(entry), buf);
				if((dot = strrchr(buf, '.')) != NULL)
					gtk_entry_set_position(GTK_ENTRY(entry), dot - buf);
			}
			gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);

			dlg = dlg_dialog_sync_new(vbox, _("Join"), _("_Join|_Cancel"));
			gtk_widget_show_all(vbox);
			gtk_widget_grab_focus(entry);
			if(dlg_dialog_sync_wait(dlg) == DLG_POSITIVE)
				ret = do_join(min, src, dst, store, GTK_ENTRY(entry), tot);
			dlg_dialog_sync_destroy(dlg);
		}
		dp_free_selection(sel);
	}
	err_show(min);
	return ret;
}
